﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <G3dModelPreview.h>


//---------------------------------------------------------------------------
//  EftVwrModelクラスのインスタンスを生成します。
//---------------------------------------------------------------------------
nw::eftvw2::Preview* G3dModelPreview::CreateModelPreview( nw::eft2::Heap* heap, nw::eftvw2::ViewerSystem* viewerSystem, void* binary, u32 binarySize, nw::eftcom::Guid guid )
{
    nw::eftut2::NwHeap* nwHeap = reinterpret_cast<nw::eftut2::NwHeap*>( heap );
    if ( !nwHeap ) return NULL;

    // モデルプレビュークラス生成
    G3dModelPreview* modelPreview = NULL;
    void* buffer = heap->Alloc( sizeof( G3dModelPreview ) );
    if ( !buffer ) { return NULL; }
    modelPreview = new (buffer) G3dModelPreview();

    // プレビュー初期化
    modelPreview->Initialize( heap, Preview::EFT_VWR_PRV_MODEL, guid );
    if ( !modelPreview->InitializeModelPreview( nwHeap, binary, binarySize ) )
    {
        heap->Free( modelPreview );
        return NULL;
    }

    // 破棄コールバックをセット
    modelPreview->SetDestroyPreviewCallback( G3dModelPreview::DestroyModelPreview );

    // ボーン情報バッファを生成します
    modelPreview->DumpBones();
    modelPreview->Calc( false, 1.0f, nw::math::Matrix34::Identity(), nw::math::Matrix34::Identity() );

    // モデル情報開始メッセージを送信
    viewerSystem->SendBeginPreviewModelV2E();

    // モデル内の情報を送信
    viewerSystem->SendModelInfoV2E( (void*)modelPreview->GetModelInfo(), modelPreview->GetModelInfoSize() );

    // モデル情報終了メッセージを送信
    viewerSystem->SendEndPreviewModelV2E();

    return modelPreview;
}

//---------------------------------------------------------------------------
//  EftVwrModelクラスのインスタンスを破棄します。
//---------------------------------------------------------------------------
void G3dModelPreview::DestroyModelPreview( nw::eft2::Heap* heap, nw::eftvw2::Preview* preview )
{
    G3dModelPreview* modelPreview = reinterpret_cast<G3dModelPreview*>( preview );

    // モデルの終了処理
    modelPreview->FinalizeModelPreview( heap );

    // プレビューの終了処理を行います。
    modelPreview->Finalize();

    // メモリを解放します。
    heap->Free( modelPreview );
}

//---------------------------------------------------------------------------
//  モデルプレビュークラスの初期化します。
//---------------------------------------------------------------------------
bool G3dModelPreview::InitializeModelPreview( nw::eftut2::NwHeap* heap, void* binary, u32 binarySize )
{
    // モデルバイナリを確保する
    mModelBinary = heap->Alloc( binarySize, 8192 );
    if ( !mModelBinary ) return false;
    memcpy( mModelBinary, binary, binarySize );

    mNwAllocator = heap->GetNwAllocator();

    // モデル初期化
    mModel.Initialize( mModelBinary, heap->GetNwAllocator() );

    // モデル情報を収集

    return true;
}

//---------------------------------------------------------------------------
//  モデルプレビュークラスの終了処理をします。
//---------------------------------------------------------------------------
void G3dModelPreview::FinalizeModelPreview( nw::eft2::Heap* heap )
{
    EFT_ASSERT( mModelBinary );
    heap->Free( mModelBinary );

    mModel.Finalize( mNwAllocator );
}

//---------------------------------------------------------------------------
//  コンストラクタです。
//---------------------------------------------------------------------------
G3dModelPreview::G3dModelPreview()
{
    mModelBinary            = NULL;
    mNwAllocator            = NULL;
    mpModelDataMessage      = NULL;
    mpModelDataMessageSize  = 0;

    mAnimFrame              = 0.0f;
}

//---------------------------------------------------------------------------
//  プレビュー計算処理です。
//---------------------------------------------------------------------------
void G3dModelPreview::Calc( bool bPause, f32 frameRate, const nw::math::MTX34& centerMatrix, const nw::math::MTX34& viewMatrix )
{
    // 規定クラスの計算処理
    Preview::Calc( bPause, frameRate, centerMatrix, viewMatrix );

    if (mIsSetResModelPreview) {
        if (!bPause && mTime >= mResModelPreview.animStartFrame)
        {
            mAnimFrame += mResModelPreview.animSpeed / frameRate;
        }

        // SRT制御マトリクスを計算
//        mDrawMatrix.SetIdentity();
        mModel.SetAnimationFrame( mAnimFrame );
        mModel.SetAnimationIndex( 0 );
        nw::g3d::math::Mtx34* pmtx =
            (nw::g3d::math::Mtx34 *)nw::math::Cast<nw::g3d::math::Mtx34*, nw::math::MTX34*>( &mDrawMatrix );
        mModel.SetDrawMatrix( *pmtx );
        mModel.Calc();
    }
}

//------------------------------------------------------------------------------
//  モデルブロック計算処理
//------------------------------------------------------------------------------
void G3dModelPreview::CalcBlock( const nw::g3d::math::Mtx34* viewMtx, const nw::g3d::math::Mtx44* projMtx, const nw::g3d::Vec4* lightPos, nw::eftdemo::SimpleShader::ViewId id )
{
    mModel.CalcBlock( viewMtx, projMtx, lightPos, id );
}

//------------------------------------------------------------------------------
//  プレビュー描画処理
//------------------------------------------------------------------------------
void G3dModelPreview::Draw( nw::eftdemo::SimpleShader::ViewId id )
{
    mModel.Draw( id );
}

//---------------------------------------------------------------------------
//  プレビューをリセットします。
//---------------------------------------------------------------------------
void G3dModelPreview::ResetPreview( bool bFade )
{
    EFT_UNUSED_VARIABLE( bFade );
    mTime = mAnimFrame = 0.0f;
}

//---------------------------------------------------------------------------
//  マトリクスを取得します。
//---------------------------------------------------------------------------
void G3dModelPreview::GetMatrix( nw::math::MTX34* matrix, s32 idx )
{
    const nw::g3d::math::Mtx34* world = mModel.GetBoneWorldMatrix( idx );

    const nw::math::MTX34* pmtx =
        (const nw::math::MTX34 *)nw::math::Cast<const nw::math::MTX34*, const nw::g3d::math::Mtx34*>( world );

    MTX34Copy( matrix, pmtx );
}

//---------------------------------------------------------------------------
//  SimpleModel ボーンバッファのサイズを計算
//---------------------------------------------------------------------------
size_t CalcSize_BoneNameBuff( nw::eftdemo::SimpleModel* model )
{
    size_t  szDataBuf   = 0;
    int     boneCount   = model->GetBoneNum();

    for( int j = 0; j < boneCount ; ++j )
    {
        const char* boneName = model->GetBoneName( j );
        size_t szName = strlen( boneName );
        szDataBuf += szName;
        szDataBuf += 1; // ","
    }

    szDataBuf += 1; // "\0"
    szDataBuf = nw::ut::RoundUp( szDataBuf, 4 );
    return szDataBuf;
}

//---------------------------------------------------------------------------
//  ModelDataMessage パケットを生成
//---------------------------------------------------------------------------
void MakeModelDataMessage(
    nw::eftcom::ModelDataMessage*   modelDataMessage,
    size_t                          boneNameBufSize,
    nw::eftdemo::SimpleModel*       model,
    nw::eftcom::Guid&               modelId )
{
    modelDataMessage->boneCount    = model->GetBoneNum();

    modelDataMessage->isGame       = false;
    modelDataMessage->modelGuid    = modelId;
    modelDataMessage->boneDataSize = boneNameBufSize;
  //  modelDataMessage->modelID   = modelId;

    nw::ut::strcpy( modelDataMessage->modelName, nw::eftcom::max_modelName, model->GetModelName() );


    char* pBoneName = &modelDataMessage->boneData;
    for( u32 i = 0; i < modelDataMessage->boneCount; ++i)
    {
        const char* name = model->GetBoneName( i );
        size_t szName = strlen( name );
        nw::ut::MemCpy( pBoneName, name, szName );
        pBoneName += szName;
        *pBoneName++ = ',';
    }
        *(pBoneName-1) = '\0';
  //  *pBoneName++ = '\0';
    //modelDataMessage->FlipEndian();
}

//---------------------------------------------------------------------------
//  ボーン情報バッファを生成し、EffectMakerに送信します。
//---------------------------------------------------------------------------
void G3dModelPreview::DumpBones()
{
    if ( mpModelDataMessage )
    {
        mNwAllocator->Free( mpModelDataMessage );
        mpModelDataMessage      = NULL;
        mpModelDataMessageSize  = 0;
    }

    // メッセージ送信ワークのサイズを計算します。
    mpModelDataMessageSize = sizeof( nw::eftcom::ModelDataMessage ) + sizeof(nw::eftcom::ModelHeaderMessage);
    size_t boneNameBufSize =  CalcSize_BoneNameBuff( &mModel );
    mpModelDataMessageSize += boneNameBufSize;

    // ワークを確保します。
    mpModelDataMessage = mNwAllocator->Alloc( mpModelDataMessageSize );
    if ( mpModelDataMessage )
    {
        memset( mpModelDataMessage, 0, mpModelDataMessageSize );

        nw::eftcom::ModelHeaderMessage* header = (nw::eftcom::ModelHeaderMessage*)mpModelDataMessage;
        header->type = nw::eftcom::MODEL_MESSAGE_TYPE_SEND_MODELINFO;
        header->size = boneNameBufSize + sizeof( nw::eftcom::ModelDataMessage);
        //header->FlipEndian();

        void* pPacket = (void *)( (u8*)mpModelDataMessage + sizeof(nw::eftcom::ModelHeaderMessage) );

        MakeModelDataMessage( (nw::eftcom::ModelDataMessage*)pPacket,
                              boneNameBufSize,
                              &mModel, mGuId );
    }
}
