﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <FrameBuffer.h>
#include <Texture.h>

//
// TODO : demoを利用している。
//
#if defined(NW_PLATFORM_WIN32)
#include <nw/demo/gl/demo_FrameBufferUtilGL.h>
#endif

namespace nw      {
namespace eftdemo {

const FrameBuffer* FrameBuffer::g_BoundFrameBuffer = NULL;



//---------------------------------------------------------------------------
//! @brief        初期化メソッドです。
//---------------------------------------------------------------------------
void FrameBuffer::Initialize( nw::ut::MemoryAllocator* allocator, s32 width, s32 height, FrameBufferType type )
{
    m_Size  = nw::math::VEC2( static_cast<f32>( width ), static_cast<f32>( height ) );

#if defined(NW_PLATFORM_WIN32)

    NW_UNUSED_VARIABLE( allocator );
    NW_UNUSED_VARIABLE( type );

    GLuint internalFormat = GL_RGBA;

    // フレームバッファフォーマット設定
    switch ( type )
    {
        case FRAMEBUFFER_TYPE_RGB16:    internalFormat = GL_RGBA;        break;
        case FRAMEBUFFER_TYPE_RGBA32:   internalFormat = GL_RGBA;        break;
        case FRAMEBUFFER_TYPE_SRGB:     internalFormat = GL_SRGB;        break;
        case FRAMEBUFFER_TYPE_FLOAT16:  internalFormat = GL_RGBA16F_ARB; break;
    }

    // カラーバッファ生成
    m_ColorTexture.CreateTextureImage2D(
        0,
        internalFormat,
        static_cast<GLsizei>( width ),
        static_cast<GLsizei>( height ),
        0,
        GL_RGBA,
        GL_UNSIGNED_BYTE,
        NULL
        );
    NW_GL_ASSERT();

    // デプスバッファ生成
    m_DepthTexture.CreateTextureImage2D(
        0,
        GL_DEPTH_COMPONENT32,
        static_cast<GLsizei>( width ),
        static_cast<GLsizei>( height ),
        0,
        GL_DEPTH_COMPONENT,
        GL_UNSIGNED_BYTE,
        NULL
        );
    NW_GL_ASSERT();

    // フレームバッファ生成
    glGenFramebuffersEXT( 1, &m_FBO );
    NW_ASSERT( m_FBO != 0 );

    glBindFramebufferEXT( GL_FRAMEBUFFER_EXT, m_FBO );
    NW_GL_ASSERT();

    // テクスチャとデプスバッファのバインド。
    glFramebufferTexture2DEXT( GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_2D, m_ColorTexture.GetID(), 0 );
    glFramebufferTexture2DEXT( GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, m_DepthTexture.GetID(), 0 );
    NW_GL_ASSERT();

#endif
#if defined(NW_PLATFORM_CAFE)

    GX2AAMode aamode = GX2_AA_MODE_1X;
    GX2SurfaceDim texSurfaceDim = GX2_SURFACE_DIM_2D;

    // カラーバッファ構造体を初期化する
    GX2InitColorBuffer( &m_Color, width, height, static_cast<GX2SurfaceFormat>( type ), aamode );
    // カラーバッファのメモリを取得する。
    GX2InitColorBufferPtr( &m_Color, allocator->Alloc( m_Color.surface.imageSize, m_Color.surface.alignment ) );

    // カラーバッファテクスチャ
    GX2InitTexture( &m_ColorTexture,
        m_Color.surface.width,
        m_Color.surface.height,
        m_Color.surface.depth,
        m_Color.surface.numMips,
        m_Color.surface.format,
        texSurfaceDim );

    m_ColorTexture.surface.tileMode = m_Color.surface.tileMode;
    GX2CalcSurfaceSizeAndAlignment( &m_ColorTexture.surface );
    GX2InitTextureRegs( &m_ColorTexture );
    GX2InitTexturePtrs( &m_ColorTexture, m_Color.surface.imagePtr, 0 );

    // デプスバッファ構造体を初期化する
    GX2InitDepthBuffer( &m_Depth, width, height, GX2_SURFACE_FORMAT_TCD_R32_FLOAT, aamode );

    // デプスバッファのメモリを取得する。
    GX2InitDepthBufferPtr( &m_Depth, allocator->Alloc( m_Depth.surface.imageSize, m_Depth.surface.alignment ) );
#endif
}


//---------------------------------------------------------------------------
//! @brief        終了処理メソッドです。
//---------------------------------------------------------------------------
void FrameBuffer::Finalize( nw::ut::MemoryAllocator* allocator )
{
#if defined(NW_PLATFORM_WIN32)
    NW_UNUSED_VARIABLE( allocator );
    glDeleteFramebuffersEXT( 1, &m_FBO );
#endif
}



//---------------------------------------------------------------------------
//! @brief        バッファをバインドします。
//---------------------------------------------------------------------------
void FrameBuffer::Bind()
{
#if defined(NW_PLATFORM_WIN32)

    glBindFramebufferEXT( GL_FRAMEBUFFER_EXT, m_FBO );
    NW_GL_ASSERT();

    nw::gfnd::Graphics::GetInstance()->SetViewport(
        0.0f,
        0.0f,
        static_cast<f32>( m_ColorTexture.GetWidth() ), static_cast<f32>( m_ColorTexture.GetHeight() ),
        0.0f,
        1.0f,
        static_cast<f32>( m_ColorTexture.GetWidth() ), static_cast<f32>( m_ColorTexture.GetHeight() )
        );

    nw::gfnd::Graphics::GetInstance()->SetScissor(
        0.0f,
        0.0f,
        static_cast<f32>( m_ColorTexture.GetWidth() ), static_cast<f32>( m_ColorTexture.GetHeight() ),
        static_cast<f32>( m_ColorTexture.GetWidth() ), static_cast<f32>( m_ColorTexture.GetHeight() ) );

    NW_GL_ASSERT();
#endif
#if defined(NW_PLATFORM_CAFE)
    GX2SetContextState(nw::gfnd::Graphics::GetInstance()->GetGX2ContextState());

    GX2SetViewport( 0, 0, (m_Color.surface.width >> m_Color.viewMip), (m_Color.surface.height >> m_Color.viewMip), 0.0f, 1.0f );
    GX2SetScissor( 0, 0,  (m_Color.surface.width >> m_Color.viewMip), (m_Color.surface.height >> m_Color.viewMip) );

    GX2SetViewport( 0, 0, m_Size.x, m_Size.y, 0.0f, 1.0f );
    GX2SetScissor( 0, 0,  m_Size.x, m_Size.y );

#if 0   // HalfZ Test
    GX2ViewportReg reg;
    GX2InitViewportRegHalfZ( &reg, 0, 0, (mColor.surface.width >> mColor.viewMip), (mColor.surface.height >> mColor.viewMip), 0.0f, 1.0f );
    GX2SetViewportReg( &reg );
#endif

    // カラーバッファを設定する(今のところターゲット0固定)
    GX2SetColorBuffer( &m_Color, GX2_RENDER_TARGET_0 );
    // デプスバッファを設定する
    GX2SetDepthBuffer( &m_Depth );
#endif

    g_BoundFrameBuffer = this;
}


//---------------------------------------------------------------------------
//! @brief        バッファをクリアします。
//---------------------------------------------------------------------------
void FrameBuffer::Clear( bool clearColor, bool clearDepth )
{
#if defined(NW_PLATFORM_WIN32)
    GLbitfield bit = GL_STENCIL_BUFFER_BIT;
    if ( clearColor ) bit |= GL_COLOR_BUFFER_BIT;
    if ( clearDepth ) bit |= GL_DEPTH_BUFFER_BIT;

    glClearColor( m_ClearColor.x, m_ClearColor.y, m_ClearColor.z, m_ClearColor.w );
    NW_GL_ASSERT();

    glClearDepth( 1.0f );
    glDepthMask( GL_TRUE );
    NW_GL_ASSERT();

    glClearStencil( 0 );
    NW_GL_ASSERT();

    glClear( bit );
    NW_GL_ASSERT();
#endif
#if defined(NW_PLATFORM_CAFE)

    if ( clearColor )
    {
        GX2ClearColor( &m_Color, m_ClearColor.x, m_ClearColor.y, m_ClearColor.z, m_ClearColor.w );
    }

    if ( clearDepth )
    {
        GX2SetClearDepthStencil( &m_Depth, 1.0f, 0 );
        GX2ClearDepthStencilEx( &m_Depth, 1.0f, 0, GX2_CLEAR_BOTH );
    }

    GX2SetContextState(nw::gfnd::Graphics::GetInstance()->GetGX2ContextState());
#endif
}


//---------------------------------------------------------------------------
//! @brief        カラーバッファテクスチャをコピーをします。
//---------------------------------------------------------------------------
void FrameBuffer::CopyColorBufferTexture( FrameBuffer* srcFrameBuffer )
{
#if defined(NW_PLATFORM_WIN32)
    NW_UNUSED_VARIABLE( srcFrameBuffer );

    // TODO : 現状では、カレントのフレームバッファのコピー動作となります。
    glBindTexture( GL_TEXTURE_2D, m_ColorTexture.GetID() );
    glCopyTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, 0, 0, m_ColorTexture.GetWidth(), m_ColorTexture.GetHeight(), 0 );
#endif
#if defined(NW_PLATFORM_CAFE)
    if ( srcFrameBuffer->m_Size.x == m_Size.x && srcFrameBuffer->m_Size.y == m_Size.y )
    {
        GX2CopySurface( &srcFrameBuffer->m_ColorTexture.surface, 0, 0, &m_ColorTexture.surface, 0, 0 );
    }
    else
    {
#if ( CAFE_OS_SDK_VERSION >= 21201 )
        GX2UTSetCopyState(GX2_ENABLE);
#else
        GX2UTSetCopyState();
#endif
        GX2UTCopySurface( &srcFrameBuffer->m_ColorTexture.surface, 0, 0, &m_ColorTexture.surface, 0, 0 );
    }

    GX2SetContextState(nw::gfnd::Graphics::GetInstance()->GetGX2ContextState());
#endif
}


//---------------------------------------------------------------------------
//! @brief        保持するColorバッファをテクスチャとしてコピーする。
//---------------------------------------------------------------------------
void FrameBuffer::CopyFrameBufferTexture( FrameBufferTexture* frameBufferTexture )
{
#if defined(NW_PLATFORM_WIN32)
    // TODO : 現状では、カレントのフレームバッファのコピー動作となります。
    glBindTexture( GL_TEXTURE_2D, frameBufferTexture->GetGLTextureID() );
    glCopyTexImage2D(   GL_TEXTURE_2D,
                        0,
                        GL_RGBA,
                        0,
                        0,
                        static_cast<GLsizei>( frameBufferTexture->GetSize().x ),
                        static_cast<GLsizei>( frameBufferTexture->GetSize().y ),
                        0 );
#endif
#if defined(NW_PLATFORM_CAFE)
    GX2Texture* gx2Texture = frameBufferTexture->GetGX2Texture();

    if ( m_UseAA )
    {
        GX2ResolveAAColorBuffer( &m_Color, &gx2Texture->surface, 0, 0);
    }
    else
    {
#if ( CAFE_OS_SDK_VERSION >= 21201 )
        GX2UTSetCopyState(GX2_ENABLE);
#else
        GX2UTSetCopyState();
#endif
        GX2UTCopySurface( &m_Color.surface, 0, 0, &gx2Texture->surface, 0, 0 );
        GX2SetContextState(nw::gfnd::Graphics::GetInstance()->GetGX2ContextState());
    }
#endif
}


//---------------------------------------------------------------------------
//! @brief        保持するDepthバッファをテクスチャとしてコピーする。
//---------------------------------------------------------------------------
void FrameBuffer::CopyDepthBufferTexture( DepthBufferTexture* depthBufferTexture )
{
#if defined(NW_PLATFORM_WIN32)
    // TODO : 現状では、カレントのフレームバッファのコピー動作となります。
    glBindTexture( GL_TEXTURE_2D, depthBufferTexture->GetGLTextureID() );
    //glCopyTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, 0, 0, mDepthTexture.GetWidth(), mDepthTexture.GetHeight(), 0 );
    glCopyTexSubImage2D( GL_TEXTURE_2D, 0, 0, 0, 0, 0, m_DepthTexture.GetWidth(), m_DepthTexture.GetHeight() );
#endif
#if defined(NW_PLATFORM_CAFE)
    GX2Texture* gx2Texture = depthBufferTexture->GetGX2Texture();

    GX2ConvertDepthBufferToTextureSurface( &m_Depth, &gx2Texture->surface, 0, 0 );
    GX2SetContextState(nw::gfnd::Graphics::GetInstance()->GetGX2ContextState());
    GX2Invalidate(GX2_INVALIDATE_DEPTH_BUFFER, m_Depth.surface.imagePtr, m_Depth.surface.imageSize);
    GX2Invalidate(GX2_INVALIDATE_TEXTURE, gx2Texture->surface.imagePtr, gx2Texture->surface.imageSize);
#endif
}


//---------------------------------------------------------------------------
//! @brief        CopyToScanBuffer
//---------------------------------------------------------------------------
void FrameBuffer::CopyToDisplayBuffer( FrameBuffer* dstFrameBuffer )
{
#if defined(NW_PLATFORM_WIN32)

    GLuint id = 0;

    if ( dstFrameBuffer )
    {
        id = dstFrameBuffer->m_FBO;
    }

    nw::demo::CopyToFrameBuffer(    &m_ColorTexture,
                                    id,
                                    static_cast<f32>( m_ColorTexture.GetWidth() ),
                                    static_cast<f32>( m_ColorTexture.GetHeight() ),
                                    nw::demo::DISPLAY_ROTATE_NONE );
#endif
#if defined(NW_PLATFORM_CAFE)
    NW_UNUSED_VARIABLE( dstFrameBuffer );

    // カラーバッファをスキャンバッファに転送して画面をスワップ
    GX2SwapBuffers( &m_Color );

    // context stateを戻す
    GX2SetContextState( nw::gfnd::Graphics::GetInstance()->GetGX2ContextState() );

    // カラーバッファを再設定、この処理はスワップの後に必ず行う必要がある
    Bind();

    // スワップコマンドが確実に実行されるようにコマンドをフラッシュする
    GX2Flush();

    // カラーバッファへの描画完了を待つ
    GX2DrawDone();
#endif
}


} // namespace nw::eftdemo
} // namespace nw
