﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe/demo/demoGfx.h>
#endif

namespace nw {
namespace eftdemo {

class FrameBufferTexture;
class DepthBufferTexture;

//---------------------------------------------------------------------------
//! @brief        フレームバッファクラスです。
//---------------------------------------------------------------------------
class FrameBuffer
{
public:
    enum FrameBufferType
    {
#if defined(NW_PLATFORM_WIN32)
        FRAMEBUFFER_TYPE_RGB16       = 0,
        FRAMEBUFFER_TYPE_RGBA32      = 1,
        FRAMEBUFFER_TYPE_SRGB        = 2,
        FRAMEBUFFER_TYPE_FLOAT16     = 3,
        FRAMEBUFFER_TYPE_RGB10A2     = 4,
        FRAMEBUFFER_TYPE_NONE
#endif
#if defined(NW_PLATFORM_CAFE)
        FRAMEBUFFER_TYPE_RGB16       = GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM,
        FRAMEBUFFER_TYPE_RGBA32      = GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM,
        FRAMEBUFFER_TYPE_FLOAT16     = GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_FLOAT,
        FRAMEBUFFER_TYPE_RGB10A2     = GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM,
        FRAMEBUFFER_TYPE_NONE
#endif
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    FrameBuffer()
    {
        m_ClearColor.Set( 0.0f, 0.0f, 0.0f, 0.0f );
    }

    //---------------------------------------------------------------------------
    //! @brief        初期化メソッドです。
    //---------------------------------------------------------------------------
    void Initialize( nw::ut::MemoryAllocator* allocator, s32 width, s32 height, FrameBufferType type );

    //---------------------------------------------------------------------------
    //! @brief        終了処理メソッドです。
    //---------------------------------------------------------------------------
    void Finalize( nw::ut::MemoryAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief        バッファをバインドします。
    //---------------------------------------------------------------------------
    void Bind();

    //---------------------------------------------------------------------------
    //! @brief        バッファをクリアします。
    //---------------------------------------------------------------------------
    void Clear( bool clearColor, bool clearDepth );

    //---------------------------------------------------------------------------
    //! @brief        保持するColorバッファをテクスチャとして利用可能にする。
    //---------------------------------------------------------------------------
    void CopyFrameBufferTexture( FrameBufferTexture* frameBufferTexture );

    //---------------------------------------------------------------------------
    //! @brief        保持するDepthバッファをテクスチャにコンバートする。
    //---------------------------------------------------------------------------
    void CopyDepthBufferTexture( DepthBufferTexture* depthBufferTexture );

    //---------------------------------------------------------------------------
    //! @brief        CopyToScanBuffer
    //---------------------------------------------------------------------------
    void CopyToDisplayBuffer( FrameBuffer* dstFrameBuffer );


#if defined(NW_PLATFORM_CAFE)
    //---------------------------------------------------------------------------
    //! @brief        カラーバッファを取得します。
    //---------------------------------------------------------------------------
    const GX2ColorBuffer* GetGX2ColorBuffer() const { return &m_Color; }

    //---------------------------------------------------------------------------
    //! @brief        デプスバッファを取得します。
    //---------------------------------------------------------------------------
    const GX2DepthBuffer* GetGX2DepthBuffer() const { return &m_Depth; }

    //---------------------------------------------------------------------------
    //! @brief        カラーバッファを取得します。
    //---------------------------------------------------------------------------
    GX2ColorBuffer* GetGX2ColorBuffer() { return &m_Color; }

    //---------------------------------------------------------------------------
    //! @brief        デプスバッファを取得します。
    //---------------------------------------------------------------------------
    GX2DepthBuffer* GetGX2DepthBuffer() { return &m_Depth; }

    //---------------------------------------------------------------------------
    //! @brief        カラーバッファテクスチャを取得します。
    //---------------------------------------------------------------------------
    GX2Texture* GetColorBufferTexture() { return &m_ColorTexture; }

    //---------------------------------------------------------------------------
    //! @brief        カラーバッファGfndテクスチャを取得します。
    //---------------------------------------------------------------------------
    const nw::gfnd::TextureCafeGX2& GetColorBufferGfndTexture()
    {
        m_GfndColorTexture.SetGX2Texture( &m_ColorTexture );
        return m_GfndColorTexture;
    }

    //---------------------------------------------------------------------------
    //! @brief        バッファのキャッシュをクリアします。
    //---------------------------------------------------------------------------
    void Invalidate( bool color, bool depth )
    {
        if ( color )
        {
            GX2InvalidateType type = GX2InvalidateType( GX2_INVALIDATE_COLOR_BUFFER | GX2_INVALIDATE_TEXTURE );
            GX2Invalidate( type, m_Color.surface.imagePtr, m_Color.surface.imageSize);
        }

        if ( depth )
        {
            GX2InvalidateType type = GX2InvalidateType( GX2_INVALIDATE_DEPTH_BUFFER | GX2_INVALIDATE_TEXTURE );
            GX2Invalidate( type, m_Depth.surface.imagePtr, m_Depth.surface.imageSize);
        }
    }
#endif


#if defined(NW_PLATFORM_WIN32)
    //---------------------------------------------------------------------------
    //! @brief        カラーバッファテクスチャを取得します。
    //---------------------------------------------------------------------------
    GLuint GetColorBufferTexture() const { return m_ColorTexture.GetID(); }

    //---------------------------------------------------------------------------
    //! @brief        Zテクスチャを取得します。
    //---------------------------------------------------------------------------
    GLuint GetDepthBufferTexture() const { return m_DepthTexture.GetID(); }

    //---------------------------------------------------------------------------
    //! @brief        カラーバッファGfndテクスチャを取得します。
    //---------------------------------------------------------------------------
    const nw::gfnd::TextureGL& GetColorBufferGfndTexture() const { return m_ColorTexture; }

    //---------------------------------------------------------------------------
    //! @brief        バッファのキャッシュをクリアします。
    //---------------------------------------------------------------------------
    void Invalidate( bool, bool ) { }
#endif



    //---------------------------------------------------------------------------
    //! @brief        クリアカラーを設定します。
    //---------------------------------------------------------------------------
    void SetClearColor( const nw::math::VEC4& color ) { m_ClearColor = color; }

    //---------------------------------------------------------------------------
    //! @brief        フレームバッファサイズを取得します。
    //!
    //! @return       フレームバッファサイズを返します。
    //---------------------------------------------------------------------------
    nw::math::VEC2 GetSize() const { return m_Size; }

    //---------------------------------------------------------------------------
    //! @brief        現在バインドされているフレームバッファを取得する static 関数です。
    //!
    //! @return       フレームバッファを返します。
    //---------------------------------------------------------------------------
    static const FrameBuffer* GetBoundFrameBuffer() { return g_BoundFrameBuffer; }

    //---------------------------------------------------------------------------
    //! @brief        カラーバッファテクスチャをコピーをします。
    //---------------------------------------------------------------------------
    void CopyColorBufferTexture( FrameBuffer* srcFrameBuffer );

private:
#if defined(NW_PLATFORM_WIN32)
    GLuint                          m_FBO;                      //!<
    nw::gfnd::TextureGL             m_ColorTexture;             //!<
    nw::gfnd::TextureGL             m_DepthTexture;             //!<
#endif
#if defined(NW_PLATFORM_CAFE)
    GX2ColorBuffer                  m_Color;                    //!<
    GX2DepthBuffer                  m_Depth;                    //!<
    GX2Texture                      m_ColorTexture;             //!<
    nw::gfnd::TextureCafeGX2        m_GfndColorTexture;         //!<
#endif

    nw::math::VEC4                  m_ClearColor;               //!< クリアカラー。
    nw::math::VEC2                  m_Size;                     //!< フレームバッファサイズです。
    static const FrameBuffer*       g_BoundFrameBuffer;         //!< 現在バインドされている FrameBuffer です。
    bool                            m_UseAA;                    //!< MSAA利用かどうか。
};


} // namespace eftdemo
} // namespace nw
