﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"
#include <windows.h>
#include <nw/types.h>
#include <dolphin/hio.h>

#define MUTEX_NAME           L"MUTEX_PC_MCS_SERVER"
#define MUTEX_PC_VIEWER_NAME L"MUTEX_PC_VIEWER"
#define MEMORY_NAME          L"MEMORY_PC_MCS_SERVER"
#define MEMORY_SIZE          0x20000

HANDLE m_hMapping;          // 共有メモリハンドル
HANDLE g_hMutex = NULL;     // 共有データへの排他アクセス用ミューテックスオブジェクトのハンドル

/*---------------------------------------------------------------------------*
  Name:         IsConnect_PC_Viewer

  Description:  PC_Viewer の起動確認

  Arguments:    なし。

  Returns:      PC_Viewer が起動していれば true、していなければ falseを返します。
 *---------------------------------------------------------------------------*/
static BOOL IsConnect_PC_Viewer()
{
    HANDLE hMutex = OpenMutex(MUTEX_ALL_ACCESS, FALSE, MUTEX_PC_VIEWER_NAME);
    if( hMutex != NULL )
    {
        CloseHandle(hMutex);
        return true;
    }
    return false;
}


/*---------------------------------------------------------------------------*
  Name:         spfHIOSearchDevices_

  Description:  PC_Viewer の起動確認

  Arguments:    なし。

  Returns:      PC_Viewer が起動していれば true、していなければ falseを返します。
 *---------------------------------------------------------------------------*/
BOOL WINAPI spfHIOSearchDevices_()
{
    if( IsConnect_PC_Viewer() == false )
        return false;
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         spfHIOInit_

  Description:  PC_Viewer が起動しているかどうか確認し、
                共有メモリを作成されていなければ、作成します。

  Arguments:    なし。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *---------------------------------------------------------------------------*/
BOOL WINAPI spfHIOInit_()
{
    if( IsConnect_PC_Viewer() == false )
        return false;
    // 共有メモリをオープン
    m_hMapping = ::OpenFileMapping( FILE_MAP_ALL_ACCESS, FALSE,
                                            MEMORY_NAME);
    // 既に作られている
    if( m_hMapping != NULL)
    {
        return true;
    }
    // 共有メモリを作成
    m_hMapping = ::CreateFileMapping( INVALID_HANDLE_VALUE,
                                NULL, PAGE_READWRITE,
                                0, MEMORY_SIZE, MEMORY_NAME);
    if( m_hMapping == NULL)
    {
        //AfxMessageBox( "共有メモリ作成失敗");
        return false;
    }
    // 共有メモリを初期化
    // マッピングする
    LPSTR lpBuf = (LPSTR)::MapViewOfFile( m_hMapping,
                                FILE_MAP_ALL_ACCESS, 0, 0, 0);
    if( lpBuf == NULL)
    {
        //AfxMessageBox( "共有メモリマッピング失敗");
        // 共有メモリも削除
        ::CloseHandle( m_hMapping);
        m_hMapping = NULL;
        return false;
    }
    // 初期化
    ::ZeroMemory( lpBuf, MEMORY_SIZE);
    // マッピングを解除
    ::UnmapViewOfFile( lpBuf);

    /* 排他制御用ミューテックスオブジェクトオープン */
    g_hMutex = OpenMutex(MUTEX_ALL_ACCESS, FALSE, MUTEX_NAME);
    if( g_hMutex == NULL )
        /* 排他制御用ミューテックスオブジェクト作成 */
        g_hMutex = CreateMutex(NULL, FALSE, MUTEX_NAME);

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         HIOReadStatus

  Description:  ステータスを読み込むます。

  Arguments:    status: 読み込む変数のポインタ

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *---------------------------------------------------------------------------*/
BOOL WINAPI spfHIOReadStatus_( u32* status )
{
    if( IsConnect_PC_Viewer() == false )
        return false;

    // 共有メモリをマッピング
    u32* lpBuf = (u32*)::MapViewOfFile( m_hMapping,
                                   FILE_MAP_READ, 0, 0, 0);
    if( lpBuf == NULL)
    {
        // マッピング失敗
        //AfxMessageBox( "共有メモリマッピング失敗。");
        return false;
    }
    //ミューテックスオブジェクトを使用して同期処理をおこなう
    WaitForSingleObject(g_hMutex, INFINITE);
    // 読込み
    *status = lpBuf[0];
    // マッピングを解除
    ::UnmapViewOfFile( lpBuf);

    ReleaseMutex(g_hMutex);

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         HIOReadMailbox

  Description:  メールボックスから読み込む

  Arguments:    word:   読み込む変数のポインタ

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *---------------------------------------------------------------------------*/
BOOL WINAPI spfHIOReadMailbox_( u32* word )
{
    if( IsConnect_PC_Viewer() == false )
        return false;

    // 共有メモリをマッピング
    u32* lpBuf = (u32*)::MapViewOfFile( m_hMapping,
                                   FILE_MAP_ALL_ACCESS, 0, 0, 0);
    if( lpBuf == NULL)
    {
        // マッピング失敗
        //AfxMessageBox( "共有メモリマッピング失敗。");
        return false;
    }
    //ミューテックスオブジェクトを使用して同期処理をおこなう
    WaitForSingleObject(g_hMutex, INFINITE);
    // 読込み
    *word = lpBuf[2];
    lpBuf[0] &= ~HIO_STATUS_RX;
    // マッピングを解除
    ::UnmapViewOfFile( lpBuf);

    ReleaseMutex(g_hMutex);

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         HIOWriteMailbox

  Description:  メールボックスに書き込みます。

  Arguments:    word:   書き込む変数

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *---------------------------------------------------------------------------*/
BOOL WINAPI spfHIOWriteMailbox_( u32  word )
{
    if( IsConnect_PC_Viewer() == false )
        return false;

    // 共有メモリをマッピング
    u32* lpBuf = (u32*)::MapViewOfFile( m_hMapping,
                              FILE_MAP_WRITE,
                                0, 0, 0);
    if( lpBuf == NULL)
    {
        // マッピング失敗
        //AfxMessageBox( "共有メモリマッピング失敗。");
        return false;
    }
    //ミューテックスオブジェクトを使用して同期処理をおこなう
    WaitForSingleObject(g_hMutex, INFINITE);
    // 書き込み
    lpBuf[1] = word;
    lpBuf[0] |= HIO_STATUS_TX;
    // マッピングを解除
    ::UnmapViewOfFile( lpBuf);

    ReleaseMutex(g_hMutex);

    return true;
}

/*-------------------------------------------------------------------------*
  Name:         spfHIORead_

  Description:  共有メモリからの読み込みを行います。
                MemoryBlockSizeの境界をまたいで読み込むことは出来ません。

  Arguments:    addr:    共有メモリの読み込み開始アドレス。
                buffer:  読み込むデータを格納するバッファへのポインタ。
                size:    読み込みデータサイズ。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *-------------------------------------------------------------------------*/
BOOL WINAPI spfHIORead_( u32 addr, void* buffer, s32 size )
{
    if( IsConnect_PC_Viewer() == false )
        return false;
    // 共有メモリをマッピング
    u8* lpBuf = (u8*)::MapViewOfFile( m_hMapping,
                                   FILE_MAP_READ, 0, 0, 0);
    if( lpBuf == NULL)
    {
        // マッピング失敗
        // MessageBox( NULL, L"1", L"共有メモリマッピング失敗。", MB_OK);
        return false;
    }
    //ミューテックスオブジェクトを使用して同期処理をおこなう
    WaitForSingleObject(g_hMutex, INFINITE);
    // 読込み
    (void)memcpy(buffer, lpBuf + addr, size);
    // マッピングを解除
    ::UnmapViewOfFile( lpBuf);

    ReleaseMutex(g_hMutex);

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         spfHIOWrite_

  Description:  共有メモリへの書き込みを行います。
                MemoryBlockSizeの境界をまたいで書き込むことは出来ません。

  Arguments:    addr:    共有メモリの書き込み開始アドレス。
                buffera: 書き込むデータを格納するバッファへのポインタ。
                size:    書き込みデータサイズ。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *---------------------------------------------------------------------------*/
BOOL WINAPI spfHIOWrite_( u32 addr, const void* buffer, s32 size )
{
    if( IsConnect_PC_Viewer() == false )
        return false;
    // 共有メモリをマッピング
    u8* lpBuf = (u8*)::MapViewOfFile( m_hMapping,
                              FILE_MAP_WRITE,
                                0, 0, 0);
    if( lpBuf == NULL)
    {
        // マッピング失敗
        //AfxMessageBox( "共有メモリマッピング失敗。");
        return false;
    }
    //ミューテックスオブジェクトを使用して同期処理をおこなう
    WaitForSingleObject(g_hMutex, INFINITE);
    // 書き込み
    (void)memcpy(lpBuf + addr, buffer, size);
    // マッピングを解除
    ::UnmapViewOfFile( lpBuf);

    ReleaseMutex(g_hMutex);

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         spfHIOExit_

  Description:  共有メモリハンドルをクローズします。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void WINAPI spfHIOExit_()
{
    // 共有メモリハンドルをクローズ
    ::CloseHandle( m_hMapping);
}

/*
BOOL WINAPI spfHIOReadAsync_( u32 addr, void* buffer, s32 size, HIOCallback callback )
{
    return true;
}

BOOL WINAPI spfHIOWriteAsync_( u32 addr, const void* buffer, s32 size, HIOCallback callback )
{
    return true;
}

BOOL WINAPI spfHIOInit2_( s32 chan, HIOCallback callback, HIONotify notify, void* param)
{
    return true;
}

BOOL WINAPI spfHIOEnumDevices_( HIOEnumCallback callback )
{
    return true;
}
*/
