﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "StdAfx.h"

#if defined(WIN32) || defined(NW4R_GC)
#if !defined(NW4R_FINAL) || defined(NW4R_MCS_ENABLE)

#if defined(WIN32)

    #pragma unmanaged

    #include <windows.h>
    #include <stdio.h>
    #include <assert.h>

    #define NW4R_ASSERT             assert
    #define NW4R_WARNING(exp, msg)  ((void) 0)

    #define MCS_HIOREAD             mspfHIORead
    #define MCS_HIOWRITE            mspfHIOWrite

#else   // #if defined(WIN32)

    #include <string.h>
    #include <stdarg.h>

    #include <dolphin.h>
    #include <dolphin/hio.h>
    #include <nw4r/misc.h>

    #define MCS_HIOREAD             HIORead
    #define MCS_HIOWRITE            HIOWrite

#endif  // #if defined(WIN32)

#include <nw4r/mcs/hioRingBuffer.h>
#include <nw4r/mcs/common.h>

const u32   CheckCode       = ('M' << 24) + ('C' << 16) + ('H' << 8) + 'I';     // バッファの先頭に書き込むチェックコード(起動チェックにも使用)

const int   HIOFuncRetryCountMax   = 30;
const int   HIOFuncRetrySleepTime  = 8;   // ms

namespace
{

struct DataPointer
{
    u32         pointer;
    u8          reserved[24];
    u32         pointerCopy;
};


struct BufferHeader
{
    u32         signature;
    u8          reserved[28];

    DataPointer read;
    DataPointer write;
};

struct MessageHeader
{
    u32     channel;
    u32     size;
    u32     reserve1;
    u32     reserve2;
};

const u16   ReadInfoOffset  = 0x20;         // 読み込みポイントが入っている場所のOffset
const u16   WriteInfoOffset = 0x40;         // 書き込みポイントが入っている場所のOffset
const u16   HeaderSize      = sizeof(BufferHeader);     // Header : 全ヘッダのサイズ

const int   UpdateReadWritePointRetryCountMax   = 30;   // mReadPoint, mWritePoint を更新する際のリトライ回数

/* ------------------------------------------------------------------------
        デバッグメッセージ出力
   ------------------------------------------------------------------------ */

#if defined(ENABLE_MCS_REPORT)

    #if defined(WIN32)

        void
        McsReport(const char* format, ...)
        {
            static HANDLE shConsole = NULL;

            va_list vargs;
            va_start(vargs, format);

            char tString[256];
            s32 tStringNum = _vsnprintf(tString, 256, format, vargs);

            if (shConsole == NULL)
            {
                AllocConsole();
                shConsole = CreateFile("CONOUT$", GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, NULL);
            }
            DWORD   tStringDoneNum;
            WriteConsole( shConsole , tString , tStringNum , &tStringDoneNum , NULL );

            va_end(vargs);
        }

        #define MCS_REPORT0(str)                    McsReport(str)
        #define MCS_REPORT1(fmt, arg1)              McsReport(fmt, arg1)
        #define MCS_REPORT2(fmt, arg1, arg2)        McsReport(fmt, arg1, arg2)
        #define MCS_REPORT3(fmt, arg1, arg2, arg3)  McsReport(fmt, arg1, arg2, arg3)

    #else   // #if defined(WIN32)

        #define MCS_REPORT0(str)                    OSReport(str)
        #define MCS_REPORT1(fmt, arg1)              OSReport(fmt, arg1)
        #define MCS_REPORT2(fmt, arg1, arg2)        OSReport(fmt, arg1, arg2)
        #define MCS_REPORT3(fmt, arg1, arg2, arg3)  OSReport(fmt, arg1, arg2, arg3)

    #endif  // #if defined(WIN32)

#else   // #if defined(ENABLE_MCS_REPORT)

    #define MCS_REPORT0(str)                    ((void) 0)
    #define MCS_REPORT1(fmt, arg1)              ((void) 0)
    #define MCS_REPORT2(fmt, arg1, arg2)        ((void) 0)
    #define MCS_REPORT3(fmt, arg1, arg2, arg3)  ((void) 0)

#endif  // #if defined(ENABLE_MCS_REPORT)

u32
HostToNet(u32 val)
{
    #if defined(WIN32)
        return htonl(val);
    #else
        return val;
    #endif
}

u32
NetToHost(u32 val)
{
    #if defined(WIN32)
        return ntohl(val);
    #else
        return val;
    #endif
}

void
DataPointer_Init(DataPointer* pPt, u32 pointer)
{
    pPt->pointer = HostToNet(pointer);
    (void)memset(pPt->reserved, 0, sizeof(pPt->reserved));
    pPt->pointerCopy = pPt->pointer;
}


#if !defined(WIN32)
    // const void*で通るようにする。
    void    DCFlushRange(const void* addr, u32 nBytes)
    {
        ::DCFlushRange(const_cast<void*>(addr), nBytes);
    }
#endif

}   // namespace

namespace nw4r
{
namespace mcs
{

using namespace detail;

#if defined(WIN32)

    HIOReadType     HioRingBuffer::mspfHIORead = 0;
    HIOWriteType    HioRingBuffer::mspfHIOWrite = 0;

    void
    HioRingBuffer::SetHIOFuncAddress(
        HIOReadType     readFunc,
        HIOWriteType    writeFunc
    )
    {
        mspfHIORead = readFunc;
        mspfHIOWrite = writeFunc;
    }

#endif // #if defined(WIN32)

/*---------------------------------------------------------------------------*
  Name:         HioRingBuffer::Init

  Description:  HioRingBufferオブジェクトを初期化します。

  Arguments:    baseOffset:  共有メモリの開始オフセット。
                size:        共有メモリのサイズ
                tempBuf:     HioRingBufferオブジェクトが使用する作業メモリ

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
HioRingBuffer::Init(
    u32     baseOffset,
    u32     size,
    void*   tempBuf
)
{
    NW4R_ASSERT(RoundDown(baseOffset, CopyAlignment) == baseOffset);
    NW4R_ASSERT(RoundDown(size, CopyAlignment) == size);

    #if !defined(WIN32)
        NW4R_ASSERT(RoundDown(tempBuf, CopyAlignment) == tempBuf);
    #endif

    mBaseOffset = baseOffset;
    mBufferSize = size - HeaderSize;
    mTempBuf = tempBuf;

    mReadPoint = mWritePoint = 0;

    mEnablePort = false;
}

/*---------------------------------------------------------------------------*
  Name:         HioRingBuffer::InitBuffer

  Description:  共有メモリを初期化します。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
HioRingBuffer::InitBuffer()
{
    BufferHeader* bufHeader = static_cast<BufferHeader*>(mTempBuf);
    bufHeader->signature = HostToNet(CheckCode);
    (void)memset(bufHeader->reserved, 0, sizeof(bufHeader->reserved));

    DataPointer_Init(&bufHeader->read, 0);
    DataPointer_Init(&bufHeader->write, 0);

    if (!WriteSection_(mBaseOffset + 0, bufHeader, sizeof(*bufHeader)))
    {
        return;
    }

    mReadPoint = mWritePoint = 0;

    MCS_REPORT2("DEBUG: Write to Offset:%05x , Size: %d\n" , 0, 0x60 );
}


/*-------------------------------------------------------------------------*
  Name:         ReadSection_

  Description:  共有メモリからの読み込みを行います。
                MemoryBlockSizeの境界をまたいで読み込むことは出来ません。

  Arguments:    offset:  共有メモリの読み込み開始アドレス。
                data:    読み込むデータを格納するバッファへのポインタ。
                size:    読み込みデータサイズ。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *-------------------------------------------------------------------------*/
bool
HioRingBuffer::ReadSection_(
    u32     offset,
    void*   data,
    u32     size
)
{
    NW4R_ASSERT(RoundDown(offset, 4) == offset);  // 4バイトアライメントされていること

    #if !defined(WIN32)
        NW4R_ASSERT(RoundDown(data, 32) == data);     // 32バイトアライメントされていること
    #endif

    NW4R_ASSERT(RoundUp(size, 32) == size);       // 32バイトアライメントされていること

    MCS_REPORT3("DEBUG:      HioRingBuffer::ReadSection_(%06x, %08x, %d);\n", offset, data, size );

    if (!IsPort())
    {
        return false;
    }

    #if !defined(WIN32)
        DCInvalidateRange(data , size);
    #endif

    for (int retry_count = 0; retry_count < HIOFuncRetryCountMax; ++retry_count)
    {
        BOOL bSuccess = MCS_HIOREAD(offset, data, (s32)size);
        if (bSuccess)
        {
            return true;
        }
        SleepThread(HIOFuncRetrySleepTime);
    }

    NW4R_WARNING(false, "ERROR: Communication Error. -HIORead()\n");
    DisablePort();
    return false;
}

/*-------------------------------------------------------------------------*
  Name:         WriteSection_

  Description:  共有メモリへの書き込みを行います。
                MemoryBlockSizeの境界をまたいで書き込むことは出来ません。

  Arguments:    offset:  共有メモリの書き込み開始アドレス。
                data:    書き込むデータを格納するバッファへのポインタ。
                size:    書き込みデータサイズ。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *-------------------------------------------------------------------------*/
bool
HioRingBuffer::WriteSection_(
    u32         offset,
    const void* data,
    u32         size
)
{
    NW4R_ASSERT(RoundDown(offset, 4) == offset);  // 4バイトアライメントされていること

    #if !defined(WIN32)
        NW4R_ASSERT(RoundDown(data, CopyAlignment) == data);  // 32バイトアライメントされていること
    #endif

    NW4R_ASSERT(RoundUp(size, CopyAlignment) == size);        // 32バイトアライメントされていること

    MCS_REPORT3("DEBUG:      HioRingBuffer::WriteSection_(0x%06x, %08x, %d);\n", offset, data, size );

    if (!IsPort())
    {
        return false;
    }

    #if !defined(WIN32)
        DCFlushRange(data, size);
    #endif

    for (int retryCount = 0; retryCount < HIOFuncRetryCountMax; ++retryCount)
    {
        BOOL bSuccess = MCS_HIOWRITE(offset, data, s32(size));
        if (bSuccess)
        {
            return true;
        }
        SleepThread(HIOFuncRetrySleepTime);
    }

    MCS_REPORT0("ERROR: Communication Error. -HIOWrite()\n");
    DisablePort();
    return false;
}

/*-------------------------------------------------------------------------*
  Name:         UpdateReadWritePoint_

  Description:  読み込み位置/書き込み位置を最新の状態に更新します。

  Arguments:    なし。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *-------------------------------------------------------------------------*/
bool
HioRingBuffer::UpdateReadWritePoint_()
{
    if (!UpdateReadPoint_())
    {
        return false;
    }
    return UpdateWritePoint_();
}

/*-------------------------------------------------------------------------*
  Name:         UpdateReadPoint_

  Description:  読み込み位置を最新の状態に更新します。

  Arguments:    なし。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *-------------------------------------------------------------------------*/
bool
HioRingBuffer::UpdateReadPoint_()
{
    if (!IsPort())
    {
        return false;
    }

    DataPointer* dataPtr = static_cast<DataPointer*>(mTempBuf);

    int retryCount = 0;
    for (; retryCount < UpdateReadWritePointRetryCountMax; ++retryCount)
    {
        if (!ReadSection_(mBaseOffset + ReadInfoOffset, dataPtr, sizeof(*dataPtr)))
        {
            return false;
        }
        if (dataPtr->pointer == dataPtr->pointerCopy)
        {
            break;
        }
    }
    if (retryCount >= UpdateReadWritePointRetryCountMax)
    {
        MCS_REPORT0("ReadPoint read fail.\n");
        return false;
    }

    mReadPoint = NetToHost(dataPtr->pointer);
    if (RoundDown(mReadPoint, 32) != mReadPoint)
    {
        MCS_REPORT1("mReadPoint %08X\n", mReadPoint);
    }

    return true;
}

/*-------------------------------------------------------------------------*
  Name:         UpdateWritePoint_

  Description:  書き込み位置を最新の状態に更新します。

  Arguments:    なし。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *-------------------------------------------------------------------------*/
bool
HioRingBuffer::UpdateWritePoint_()
{
    if (!IsPort())
    {
        return false;
    }

    DataPointer* dataPtr = static_cast<DataPointer*>(mTempBuf);

    int retryCount = 0;
    for (; retryCount < UpdateReadWritePointRetryCountMax; ++retryCount)
    {
        if (!ReadSection_(mBaseOffset + WriteInfoOffset, dataPtr, sizeof(*dataPtr)))
        {
            return false;
        }
        if (dataPtr->pointer == dataPtr->pointerCopy)
        {
            break;
        }
    }
    if (retryCount >= UpdateReadWritePointRetryCountMax)
    {
        MCS_REPORT0("WritePoint read fail.\n");
        return false;
    }

    mWritePoint = NetToHost(dataPtr->pointer);
    if (RoundDown(mWritePoint, 32) != mWritePoint)
    {
        MCS_REPORT1("mWritePoint %08X\n", mWritePoint);
    }

    return true;
}

/*-------------------------------------------------------------------------*
  Name:         HioRingBuffer::Read

  Description:  共有メモリからデータを読み込みます。

  Arguments:    なし。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *-------------------------------------------------------------------------*/
bool
HioRingBuffer::Read(bool* pbAvailable)
{
    NW4R_ASSERT(pbAvailable != 0);

    if (! UpdateWritePoint_())
    {
        return false;
    }

    *pbAvailable = mReadPoint != mWritePoint;

    if (! *pbAvailable)
    {
        return true;
    }

    mProcBytes = 0;
    mTransBytes = (mWritePoint + mBufferSize - mReadPoint) % mBufferSize;

    u32 currReadPoint = mReadPoint;
    u8 *const msgBuf = static_cast<u8*>(mTempBuf) + HeaderSize;
    u32 tmpOfs = 0;

    for (u32 restBytes = mTransBytes; restBytes > 0;)
    {
        const u32 readBytes = mWritePoint < currReadPoint ? mBufferSize - currReadPoint: restBytes;
        if (!ReadSection_(mBaseOffset + HeaderSize + currReadPoint, msgBuf + tmpOfs, readBytes))
        {
            return false;
        }
        currReadPoint = (currReadPoint + readBytes) % mBufferSize;
        tmpOfs += readBytes;
        restBytes -= readBytes;
    }

    NW4R_ASSERT(currReadPoint == mWritePoint);

    DataPointer* dataPtr = static_cast<DataPointer*>(mTempBuf);
    DataPointer_Init(dataPtr, currReadPoint);

    if (!WriteSection_(mBaseOffset + ReadInfoOffset, dataPtr, sizeof(*dataPtr)))
    {
        return false;
    }

    mReadPoint = currReadPoint;
    MCS_REPORT1("DEBUG: Read: Point:%08x\n" , mReadPoint);

    return true;
}

/*-------------------------------------------------------------------------*
  Name:         HioRingBuffer::GetMessage

  Description:  メッセージのデータを取得します。
                取得するたびに次のメッセージ位置へ移動します。
                取得できるメッセージが無いときは NULL を返します。

  Arguments:    pChannel:    メッセージのチャンネル値を格納する変数への
                             ポインタ。
                pTotalSize:  メッセージの総サイズを格納する変数へのポインタ。

  Returns:      取得できるメッセージが無いときは NULL、
                そうでない場合は、メッセージデータへのポインタを返します。
 *-------------------------------------------------------------------------*/
void*
HioRingBuffer::GetMessage(
    u32*    pChannel,
    u32*    pTotalSize
)
{
    if (mProcBytes >= mTransBytes)
    {
        return NULL;
    }

    u8 *const msgStart = static_cast<u8*>(mTempBuf) + HeaderSize + mProcBytes;
    const MessageHeader *const pMsgHeader = reinterpret_cast<MessageHeader*>(msgStart);

    *pChannel = NetToHost(pMsgHeader->channel);
    *pTotalSize = NetToHost(pMsgHeader->size);

    mProcBytes += RoundUp(sizeof(*pMsgHeader) + *pTotalSize, CopyAlignment);
    return msgStart + sizeof(*pMsgHeader);
}

/*-------------------------------------------------------------------------*
  Name:         HioRingBuffer::GetWritableBytes

  Description:  書き込み可能なバイト数を取得します。

  Arguments:    pBytes:     書き込み可能なバイト数を取得する変数へのポインタ。
                withUpdate: trueならばアップデートを行います。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *-------------------------------------------------------------------------*/
bool
HioRingBuffer::GetWritableBytes(u32* pBytes, bool withUpdate)
{
    if (withUpdate)
    {
        if (!UpdateReadPoint_())
        {
            return false;
        }
    }

    u32 writeEndPoint = (mReadPoint + mBufferSize - CopyAlignment) % mBufferSize;
    u32 writableBytes = (writeEndPoint + mBufferSize - mWritePoint) % mBufferSize;

    *pBytes = writableBytes >= sizeof(MessageHeader) ? writableBytes - sizeof(MessageHeader): 0;
    return true;
}

/*-------------------------------------------------------------------------*
  Name:         HioRingBuffer::Write

  Description:  共有メモリへデータを書き込みます。

  Arguments:    channel:  チャンネル値。
                buf:      書き込むデータ。
                size:     書き込むデータのサイズ。

  Returns:      関数が成功すれば true、失敗すれば falseを返します。
 *-------------------------------------------------------------------------*/
bool
HioRingBuffer::Write(
    u32         channel,
    const void* buf,
    u32         size
)
{
    if (!IsPort())
    {
        return false;
    }

    {
        u32 writableBytes;
        // 更新なしで書き込み可能バイト数を取得
        if (! GetWritableBytes(&writableBytes, false))
        {
            return false;
        }

        if (size > writableBytes)
        {
            // 更新ありで書き込み可能バイト数を取得
            if (! GetWritableBytes(&writableBytes, true))
            {
                return false;
            }
            if (size > writableBytes)
            {
                // 書き込めるサイズを超えていたら書き込みを拒否します。
                // この関数を呼び出す側でサイズを調整しなければなりません。
                return false;
            }
        }
    }

    u8 *const tempBuf = static_cast<u8*>(mTempBuf);
    bool bOutMessageHeader = false;
    const u8* pSrc = static_cast<const u8*>(buf);
    const u32 writeEndPoint = (mReadPoint + mBufferSize - CopyAlignment) % mBufferSize;

    u32 currWritePoint = mWritePoint;
    for (u32 restBytes = size; restBytes > 0;)
    {
        u32 tmpOfs = 0;

        // メッセージヘッダを含まないサイズ
        u32 writableBytes = writeEndPoint >= currWritePoint ? writeEndPoint - currWritePoint: mBufferSize - currWritePoint;

        if (! bOutMessageHeader)
        {
            MessageHeader *const pHeader = reinterpret_cast<MessageHeader*>(tempBuf + tmpOfs);
            pHeader->channel = HostToNet(channel);
            pHeader->size = HostToNet(size);
            pHeader->reserve1 = 0;
            pHeader->reserve2 = 0;

            tmpOfs += sizeof(MessageHeader);
            // NW4R_COMPILER_ASSERT(sizeof(MessageHeader) < CopyAlignment);
            writableBytes -= sizeof(MessageHeader);

            bOutMessageHeader = true;
        }

        const u32 msgWriteBytes = GetMin(restBytes, writableBytes);
        (void)memcpy(tempBuf + tmpOfs, pSrc, msgWriteBytes);       // テンポラリバッファに転送
        tmpOfs += msgWriteBytes;
        pSrc += msgWriteBytes;
        restBytes -= msgWriteBytes;

        if (restBytes == 0)
        {
            const u32 oldTempOfs = tmpOfs;
            tmpOfs = RoundUp(tmpOfs, CopyAlignment);

            if (tmpOfs > oldTempOfs)
            {
                (void)memset(tempBuf + oldTempOfs, 0, tmpOfs - oldTempOfs);
            }
        }

        if (!WriteSection_(mBaseOffset + HeaderSize + currWritePoint, tempBuf, tmpOfs))
        {
            return false;
        }
        currWritePoint = (currWritePoint + tmpOfs) % mBufferSize;
    }

    DataPointer* dataPtr = static_cast<DataPointer*>(mTempBuf);
    DataPointer_Init(dataPtr, currWritePoint);

    MCS_REPORT1("DEBUG: EndWrite: Point:%08x\n" , currWritePoint);
    if (!WriteSection_(mBaseOffset + WriteInfoOffset, dataPtr, sizeof(*dataPtr)))
    {
        return false;
    }

    mWritePoint = currWritePoint;

    return true;
}

}   // namespace mcs
}   // namespace nw4r

#endif  // #if !defined(NW4R_FINAL) || defined(NW4R_MCS_ENABLE)
#endif  // #if defined(WIN32) || defined(NW4R_GC)
