﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Net.Sockets;
using System.Diagnostics;
using System.IO;
using System.Threading;
using Nintendo.McsServer.McsUtil;

namespace Nintendo.McsServer.PCIO2CommDevice
{
    enum ChunkID : uint
    {
        Unknown = 0,
        RegistName,
        Reboot,
        InitBuffer,
        Ack,
        Disconnect,
        MessageToServer,
        MessageToAppication,
        ServerTime,
        CheckAlive,
    }

    enum ResultCode : uint
    {
        Success, // 成功。
        ProtocolError, // 通信プロトコルのエラーです。
        AlreadyRegisted, // 既に登録されています。
    }

    /// <summary>
    /// 検出されたデバイスごとに呼ばれるコールバックです。
    /// </summary>
    /// <param name="id">デバイスを識別するIDです。</param>
    /// <returns>デバイスの検出を完了（中断）したい場合は true を返します。</returns>
    public delegate bool EnumDeviceCallback(object id);

    /// <summary>
    /// ソケットを使ったバージョンのPC版実機アプリ用通信モジュールです。
    /// </summary>
    public sealed class PCIO2CommDevice : ICommDevice, ICheckAlive
    {
        const string _sCommDeviceName = "PC";

        /// <summary>
        /// ソケットを監視しPC版実機アプリからの接続を受け付けます。
        /// </summary>
        static SocketListener _sSocketListner;

        /// <summary>
        ///  実機アプリからの接続。
        /// </summary>
        static Queue<Socket> _sSockets = new Queue<Socket>();

        /// <summary>
        /// PC版実機アプリとMCSサーバ間の通信に利用するソケット。
        /// </summary>
        McsSocket _mcsSocket = null;

        /// <summary>
        /// 同期用オブジェクト。
        /// </summary>
        readonly object _syncObj = new object();

        /// <summary>
        /// スレッドへの終了通知。
        /// </summary>
        ManualResetEvent _DoneEvent = new ManualResetEvent(false);

        /// <summary>
        /// 実機アプリからの接続がある。
        /// </summary>
        static ManualResetEvent _sSocketExistEvent = new ManualResetEvent(false);

        /// <summary>
        /// メッセージ受信用スレッド。
        /// </summary>
        private Thread _readThread;

        /// <summary>
        /// (コマンド)パケットヘッダサイズ。
        /// </summary>
        const int ChunkHeaderSize = 8;

        /// <summary>
        /// メッセージヘッダサイズ。
        /// UInt32 channel
        /// </summary>
        const int MessageHeaderSize = 8;

        /// <summary>
        /// 実機アプリ側の制限による最大メッセージサイズ。
        /// </summary>
        const int WritableBytes = 32 * 1024 - ChunkHeaderSize - MessageHeaderSize;

        /// <summary>
        /// リードスレッドからメインスレッドへの要求キュー。
        /// </summary>
        Queue<NegoRequest> _NegotiateQueue = new Queue<NegoRequest>();

        /// <summary>
        /// Negotiate() で認知されたメッセージ。
        /// </summary>
        Queue<MessageData> _ReadQueue = new Queue<MessageData>();

        /// <summary>
        /// 接続状況。
        /// 通信が可能な場合に設定されます。
        /// Negociate() で更新されます。
        /// </summary>
        bool _bTargetConnect = false;

        /// <summary>
        /// ターゲットプログラムが停止したと判断する上限の時間[秒]。
        /// 0 の場合はタイムアウトによる生存判定は無効。
        /// </summary>
        uint _CheckAliveTimeout = 0;

        /// <summary>
        /// 生存確認の失敗によりターゲットとの接続を切った場合に設定されます。
        /// </summary>
        bool _DisconnectByCheckAlive = false;

        /// <summary>
        /// 最後にターゲットプログラムからの受信データを処理した時間。
        /// </summary>
        DateTime _LastNegotiateTime = DateTime.Now;

        /// <summary>
        /// ターゲットプログラムに生存確認要求をしたか。
        /// （要求を繰り返し送信しないようにするため）
        /// </summary>
        bool _CheckAliveWaiting;

        #region CommDevice取得

        /// <summary>
        /// ソケットの接続要求の監視を始めます。
        /// </summary>
        /// <param name="portNo">ポート番号です。</param>
        public static void Start(int portNo)
        {
            Debug.Assert(_sSocketListner == null);

            _sSocketListner = new SocketListener(SocketListenerAccepted);

            _sSocketListner.Start(portNo);
        }

        public static void Stop()
        {
            Debug.Assert(_sSocketListner != null);

            _sSocketListner.Stop();
            _sSocketListner = null;
        }

        public static void EnumDevices(EnumDeviceCallback callback)
        {
            if (_sSockets.Count > 0)
            {
                callback(_sCommDeviceName);
            }
        }

        /// <summary>
        /// 通信デバイスを取得します。
        /// </summary>
        /// <param name="id">EnumDevices() で取得したデバイス ID のうちの１つを指定します。</param>
        /// <returns>通信デバイスを返します。引数 id がこの通信デバイスのもので無い場合は null を返します。</returns>
        public static PCIO2CommDevice GetDevice(object id)
        {
            if (_sCommDeviceName.Equals(id))
            {
                PCIO2CommDevice device = new PCIO2CommDevice();
                if (device.StartDevice())
                {
                    return device;
                }
            }

            return null;
        }

        /// <summary>
        /// 通信状態のデバイスを１つ取得します。
        /// </summary>
        /// <remarks>
        /// デバイスが明示的に指定された場合に使用されます。
        /// 呼び出しの時点でアプリが未接続でもデバイスを生成します。
        /// </remarks>
        /// <returns></returns>
        public static PCIO2CommDevice GetDevice()
        {
            PCIO2CommDevice device = new PCIO2CommDevice();
            if (device.StartDevice())
            {
                return device;
            }

            return null;
        }

        #endregion

        /// <summary>
        /// 接続要求があると呼ばれるコールバックです。
        /// </summary>
        /// <param name="result"></param>
        private static void SocketListenerAccepted(SocketListenerResult result)
        {
            if (result.Exception == null)
            {
                lock ((_sSockets as ICollection).SyncRoot)
                {
                    if (_sSockets.Count == 0)
                    {
                        _sSocketExistEvent.Set();
                    }

                    _sSockets.Enqueue(result.Socket);
                }
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PCIO2CommDevice()
        {
        }

        /// <summary>
        /// 非同期処理の開始。
        /// </summary>
        /// <returns></returns>
        private bool StartDevice()
        {
            _DoneEvent.Reset();
            _readThread = new Thread(ReadThreadFunc);
            _readThread.Start();
            return true;
        }

        private void ReadThreadFunc()
        {
            byte[] _buff = new byte[256];
            MemoryStream _stream = new MemoryStream(_buff);
            BinaryReader _reader = new NetBinaryReader(_stream);
            BinaryWriter _writer = new NetBinaryWriter(_stream);

            WaitHandle[] waitHandles = new WaitHandle[] { _DoneEvent, _sSocketExistEvent };

            try
            {
                while (true)
                {
                    int waitIndex = WaitHandle.WaitAny(waitHandles);
                    if (waitIndex == 0)
                    {
                        throw new OperationCanceledException();
                    }

                    Socket socket;
                    lock ((_sSockets as ICollection).SyncRoot)
                    {
                        socket = _sSockets.Dequeue();
                        if (_sSockets.Count == 0)
                        {
                            _sSocketExistEvent.Reset();
                        }
                    }

                    Log(_sCommDeviceName + " : socket start.");

                    McsSocket mcsSocket = new McsSocket(socket);
                    mcsSocket.CancelWaitHandle = _DoneEvent;

                    try
                    {
                        while (true)
                        {
                            mcsSocket.Read(_buff, 0, ChunkHeaderSize);
                            _stream.Position = 0;

                            ChunkID chunkID = (ChunkID)_reader.ReadUInt32();
                            int chunkBodySize = (int)_reader.ReadUInt32();

                            switch (chunkID)
                            {
                                case ChunkID.RegistName:
                                    this.DoIgnore(mcsSocket, chunkBodySize);
                                    break;

                                case ChunkID.MessageToServer:
                                    // メッセージを受信し、キューに登録します。
                                    {
                                        mcsSocket.Read(_buff, 0, MessageHeaderSize);
                                        _stream.Position = 0;

                                        int messageChannel = _reader.ReadInt32();
                                        int messageSize = _reader.ReadInt32();

                                        if (MessageHeaderSize + messageSize > chunkBodySize)
                                        {
                                            throw new ChunkException(chunkID);
                                        }

                                        byte[] data = new byte[chunkBodySize - MessageHeaderSize];
                                        mcsSocket.Read(data, 0, data.Length);

                                        MessageData messageData = new MessageData(messageChannel, data, 0, messageSize);

                                        lock (_syncObj)
                                        {
                                            _NegotiateQueue.Enqueue(new NegoRequest(ChunkID.MessageToServer, messageData));
                                        }
                                    }
                                    break;

                                case ChunkID.Reboot:
                                    lock (_syncObj)
                                    {
                                        _NegotiateQueue.Enqueue(new NegoRequest(ChunkID.Reboot, mcsSocket));
                                    }

                                    this.DoIgnore(mcsSocket, chunkBodySize);
                                    break;

                                case ChunkID.Ack:
                                    lock (_syncObj)
                                    {
                                        _NegotiateQueue.Enqueue(new NegoRequest(ChunkID.Ack, mcsSocket));
                                    }

                                    this.DoIgnore(mcsSocket, chunkBodySize);
                                    break;

                                case ChunkID.CheckAlive:
                                    lock (_syncObj)
                                    {
                                        _NegotiateQueue.Enqueue(new NegoRequest(ChunkID.CheckAlive));
                                    }

                                    this.DoIgnore(mcsSocket, chunkBodySize);
                                    break;

                                default:
                                    this.DoIgnore(mcsSocket, chunkBodySize);
                                    break;
                            }
                        }
                    }
                    catch (EndOfStreamException)
                    {
                    }
                    catch (ObjectDisposedException)
                    {
                    }
                    catch (Exception ex)
                    {
                        ex.Data["socket"] = socket;
                        throw;
                    }
                    finally
                    {
                        mcsSocket.Close();
                        lock (_syncObj)
                        {
                            _NegotiateQueue.Enqueue(new NegoRequest(ChunkID.Disconnect));
                        }
                    }
                }
            }
            catch (OperationCanceledException)
            {
            }
            catch (Exception ex)
            {
                this.Dispose(ex);
            }
        }

        /// <summary>
        /// 読み飛ばします。
        /// </summary>
        void DoIgnore(McsSocket mcsSocket, int ignoreBytes)
        {
            byte[] buff = new byte[1024];

            while (0 < ignoreBytes)
            {
                int readBytes = Math.Min(buff.Length, ignoreBytes);
                mcsSocket.Read(buff, 0, readBytes);
                ignoreBytes -= readBytes;
            }
        }

        void Response(McsSocket mcsSocket, ChunkID chunkID)
        {
            const UInt32 MessageSize = 0;
            byte[] resBuf = new byte[ChunkHeaderSize + MessageSize];
            var bw = new NetBinaryWriter(new MemoryStream(resBuf));
            bw.Write((UInt32)chunkID);
            bw.Write(MessageSize);
            bw.Flush();

            try
            {
                mcsSocket.Write(resBuf, 0, resBuf.Length);
            }
            catch
            {
            }
        }

        void Response(McsSocket mcsSocket, ChunkID chunkID, ResultCode result)
        {
            const UInt32 MessageSize = 4;
            byte[] resBuf = new byte[ChunkHeaderSize + MessageSize];
            var bw = new NetBinaryWriter(new MemoryStream(resBuf));
            bw.Write((UInt32)chunkID);
            bw.Write(MessageSize);
            bw.Write((UInt32)result);
            bw.Flush();

            try
            {
                mcsSocket.Write(resBuf, 0, resBuf.Length);
            }
            catch
            {
            }
        }

        void SendServerTime(McsSocket mcsSocket)
        {
            const UInt32 MessageSize = 8;
            byte[] resBuf = new byte[ChunkHeaderSize + MessageSize];
            var bw = new NetBinaryWriter(new MemoryStream(resBuf));
            bw.Write((UInt32)ChunkID.ServerTime);
            bw.Write(MessageSize);

            // 2000年 1月 1日 午前0時からの経過時間(msec)。
            long msec = (long)(DateTime.Now - new DateTime(2000, 1, 1)).TotalMilliseconds;
            bw.Write((Int64)msec);
            bw.Flush();

            try
            {
                mcsSocket.Write(resBuf, 0, resBuf.Length);
            }
            catch
            {
            }
        }

        public void Negotiate()
        {
            bool bNegoProcessed = false;

            while (_NegotiateQueue.Count > 0)
            {
                bNegoProcessed = true;
                NegoRequest negoRequest = null;

                lock (_syncObj)
                {
                    if (_NegotiateQueue.Count > 0)
                    {
                        negoRequest = _NegotiateQueue.Dequeue();
                    }
                }

                if (negoRequest == null)
                {
                    break;
                }

                switch (negoRequest.ChunkID)
                {
                    case ChunkID.Reboot:
                        {
                            McsSocket mcsSocket = (McsSocket)negoRequest.Data;
                            this.Response(mcsSocket, ChunkID.InitBuffer);
                            _ReadQueue.Clear();
                        }
                        break;

                    case ChunkID.Ack:
                        _bTargetConnect = true;
                        _DisconnectByCheckAlive = false;
                        lock (_syncObj)
                        {
                            _mcsSocket = (McsSocket)negoRequest.Data;
                        }
                        // Negotiate 中に状態は１度しか変化させない。
                        goto Pass;

                    case ChunkID.Disconnect:
                        _bTargetConnect = false;
                        lock (_syncObj)
                        {
                            _mcsSocket = null;
                        }
                        // Negotiate 中に状態は１度しか変化させない。
                        goto Pass;

                    case ChunkID.MessageToServer:
                        _ReadQueue.Enqueue((MessageData)negoRequest.Data);
                        break;

                    case ChunkID.CheckAlive:
                        break;

                    default:
                        break;
                }
            }

        Pass:

            // ターゲットプログラムの生存確認
            if (_CheckAliveTimeout > 0)
            {
                if (bNegoProcessed)
                {
                    _LastNegotiateTime = DateTime.Now;
                    _CheckAliveWaiting = false;
                }

                if (_bTargetConnect)
                {
                    TimeSpan timeSpan = DateTime.Now - _LastNegotiateTime;

                    if (!_CheckAliveWaiting)
                    {
                        // しばらくターゲットからのデータの受信がなかったら生存確認を要求する。
                        if (timeSpan > TimeSpan.FromSeconds(0.5 * _CheckAliveTimeout))
                        {
                            this.Response(_mcsSocket, ChunkID.CheckAlive);
                            _CheckAliveWaiting = true;
                        }
                    }
                    else
                    {
                        // 規定時間までにターゲットから何らかのデータの受信がなかったら停止したと判断する。
                        if (timeSpan > TimeSpan.FromSeconds(_CheckAliveTimeout))
                        {
                            this.Response(_mcsSocket, ChunkID.Disconnect);
                            _bTargetConnect = false;
                            _DisconnectByCheckAlive = true;
                        }
                    }
                }
            }

            if (_bTargetConnect)
            {
#if false // FIXME! Writeがブロックされてデッドロックする
                try
                {
                    this.SendServerTime(_mcsSocket);
                }
                catch
                {
                }
#endif
            }
        }

        public string Name
        {
            get { return _sCommDeviceName; }
        }

        public bool IsAttach
        {
            get { return true; }
        }

        public bool IsTargetConnect
        {
            get { return _bTargetConnect; }
        }

        uint ICheckAlive.CheckAliveTimeout
        {
            get
            {
                return _CheckAliveTimeout;
            }
            set
            {
                _CheckAliveTimeout = value;
            }
        }

        bool ICheckAlive.IsDisconnectByCheckAlive
        {
            get
            {
                return _DisconnectByCheckAlive;
            }
        }

        public bool Read()
        {
            return _ReadQueue.Count > 0;
        }

        public MessageData GetMessage()
        {
            if (_ReadQueue.Count == 0)
            {
                return null;
            }
            else
            {
                return _ReadQueue.Dequeue();
            }
        }

        public int GetWritableBytes(bool withUpdate)
        {
            return WritableBytes;
        }

        public uint GetWriteBufferSize()
        {
            return (uint)WritableBytes;
        }

        public void Write(int channel, byte[] buf, int offset, int size)
        {
            try
            {
                var chunkBodySize = MessageHeaderSize + size;
                var header = new byte[ChunkHeaderSize + MessageHeaderSize];
                var bw = new NetBinaryWriter(new MemoryStream(header));
                bw.Write((UInt32)ChunkID.MessageToAppication);
                bw.Write((UInt32)chunkBodySize);
                bw.Write((UInt32)channel);
                bw.Write((UInt32)size);

                _mcsSocket.Write(header, 0, header.Length);
                _mcsSocket.Write(buf, offset, size);
            }
            catch
            {
            }
        }

        private void Dispose(Exception ex)
        {
            ReportException(ex);
            Dispose();
        }

        public void Dispose()
        {
            _DoneEvent.Set();
        }

        private void Log(string message)
        {
            Debug.WriteLine(message);
        }

        private void Log(string format, object[] args)
        {
            Debug.Print(format, args);
            Debug.WriteLine("");
        }

        private void InternalError(string message)
        {
            Debug.WriteLine(message);
        }

        private void InternalError(string format, object[] args)
        {
            Debug.Write("internal error : ");
            Debug.Print(format, args);
            Debug.WriteLine("");
        }

        private void ReportException(Exception e)
        {
            if (e == null)
            {
                return;
            }

            Socket socket = null;

            if (e.Data.Contains("socket"))
            {
                socket = (Socket)e.Data["socket"];
            }

            Debug.WriteLine(GetInfoStr(socket) + " : " + e);
        }

        /// <summary>
        /// 説明文字列を取得します。
        /// </summary>
        /// <returns>文字列を返します。</returns>
        private string GetInfoStr(Socket socket)
        {
            lock (_syncObj)
            {
                if (socket != null)
                {
                    return string.Format("{0} socket [{1}]", _sCommDeviceName, socket.RemoteEndPoint);
                }
                else
                {
                    return string.Format("{0} socket [Unknown]", _sCommDeviceName);
                }
            }
        }

        public override string ToString()
        {
            return Name;
        }

        /// <summary>
        /// リードスレッドからメインスレッドへの要求データ。
        /// </summary>
        class NegoRequest
        {
            public ChunkID ChunkID
            {
                get;
                set;
            }

            public object Data
            {
                get;
                set;
            }

            public NegoRequest(ChunkID chunkID)
                : this(chunkID, null)
            {
            }

            public NegoRequest(ChunkID chunkID, object data)
            {
                this.ChunkID = chunkID;
                this.Data = data;
            }
        }

        /// <summary>
        /// ソケットがShutdownされたと判断した場合にスローする例外。
        /// コネクションが切れる。
        /// </summary>
        class ShutdownException : Exception
        {
        }

        /// <summary>
        /// チャンクの内容に不正があった場合にスローする例外。
        /// チャンクの受付は継続する。
        /// </summary>
        class ChunkException : Exception
        {
            readonly ChunkID _chunkID;

            public ChunkID ChunkID { get { return _chunkID; } }

            public ChunkException(ChunkID id)
            {
                _chunkID = id;
            }
        }

        public class PCIO2Exception : Exception
        {
            public PCIO2Exception(string message)
                : base(message)
            {
            }
        }
    }
}
