﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Nintendo.McsServer.McsUtil
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.IO;
    using System.Diagnostics;
    using System.Net.Sockets;
    using System.Threading;

    /// <summary>
    /// 基となるコネクションに対して以下の操作を提供するラッパークラスです。
    /// <list>
    /// <item>指定されたバイト数までの完了が保障された Read / Write 処理。</item>
    /// <item>基となるコネクションの操作の相互排除。</item>
    /// <item>処理の中断。</item>
    /// </list>
    /// </summary>
    public class McsSocket : IDisposable
    {
        /// <summary>
        /// 非同期の入出力処理のための状態オブジェクトです。
        /// </summary>
        private class IOAsyncState
        {
            public AsyncResult AsyncResult { get; set; }
            public byte[] Buffer { get; set; }
            public int Offset { get; set; }
            public int Count { get; set; }
            public AsyncCallback Callback { get; set; }
            public int DoneBytes { get; set; }

            public IOAsyncState(
                byte[] buffer,
                int offset,
                int count,
                AsyncCallback callback)
            {
                this.AsyncResult = new AsyncResult();
                this.Buffer = buffer;
                this.Offset = offset;
                this.Count = count;
                this.Callback = callback;
                this.DoneBytes = 0;
            }
        }

        /// <summary>
        /// 基になるコネクションへの操作を同期するために使用するオブジェクトです。
        /// </summary>
        public object SyncRoot
        {
            get;
            private set;
        }

        /// <summary>
        /// 基になるコネクションです。
        /// </summary>
        public Socket BaseConnection
        {
            get;
            private set;
        }

        /// <summary>
        /// イベントが設定されると処理を中断します。
        /// </summary>
        public WaitHandle CancelWaitHandle
        {
            get;
            set;
        }

        public int SendBufferSize
        {
            get
            {
                lock (this.SyncRoot)
                {
                    if (this.BaseConnection != null)
                    {
                        return this.BaseConnection.SendBufferSize;
                    }
                    else
                    {
                        return 0;
                    }
                }
            }
        }

        public System.Net.EndPoint RemoteEndPoint
        {
            get
            {
                lock (this.SyncRoot)
                {
                    if (this.BaseConnection != null)
                    {
                        return this.BaseConnection.RemoteEndPoint;
                    }
                    else
                    {
                        return null;
                    }
                }
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="connection">基になるコネクション。</param>
        public McsSocket(Socket connection)
            : this(connection, new object())
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="connection">基になるコネクション。</param>
        /// <param name="syncObj">基になるコネクションへの操作を同期するために使用するオブジェクト。</param>
        public McsSocket(Socket connection, object syncObj)
        {
            if (connection == null)
            {
                throw new ArgumentNullException("connection");
            }

            if (syncObj == null)
            {
                throw new ArgumentNullException("syncObj");
            }

            this.BaseConnection = connection;
            this.SyncRoot = syncObj;
        }

        ~McsSocket()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// 基となるコネクションを取得します。
        /// </summary>
        /// <returns>基となるコネクションのオブジェクト。</returns>
        /// <exception cref="ObjectDisposedException">
        /// コネクションは閉じられています。
        /// </exception>
        protected Socket GetBaseConnection()
        {
            if (this.BaseConnection == null)
            {
                throw new ObjectDisposedException("BaseConnection");
            }

            return this.BaseConnection;
        }

        /// <summary>
        /// 非同期処理の完了を待ちます。
        /// </summary>
        /// <param name="ar">非同期処理の開始時に取得した IAsyncResult オブジェクト。</param>
        /// <exception cref="OperationCanceledException">
        /// キャンセルイベントが設定されました。
        /// </exception>
        protected void WaitAsync(IAsyncResult ar)
        {
            if (!ar.IsCompleted)
            {
                try
                {
                    if (this.CancelWaitHandle != null)
                    {
                        WaitHandle.WaitAny(new WaitHandle[] { this.CancelWaitHandle, ar.AsyncWaitHandle });
                    }
                    else
                    {
                        ar.AsyncWaitHandle.WaitOne();
                    }
                }
                finally
                {
                    ar.AsyncWaitHandle.Close();
                }
            }

            this.CheckCancel();
        }

        /// <summary>
        /// CancelWaitHandle にイベントが設定されると OperationCanceledException 例外をスローします。
        /// </summary>
        private void CheckCancel()
        {
            if (this.CancelWaitHandle != null && this.CancelWaitHandle.WaitOne(0))
            {
                throw new OperationCanceledException();
            }
        }

        /// <summary>
        /// 元になるコネクションでの送受信を無効にします。
        /// </summary>
        /// <param name="how">無効化する送受信のタイプ。</param>
        /// <exception cref="ObjectDisposedException">
        /// コネクションは閉じられています。
        /// </exception>
        public void Shutdown(SocketShutdown how)
        {
            lock (this.SyncRoot)
            {
                this.GetBaseConnection().Shutdown(how);
            }
        }

        /// <summary>
        /// 基となるコネクションを閉じます。
        /// </summary>
        public void Close()
        {
            // NetworkStream を Close できる場合が良く分からないので、
            // とにかく Close してみて、例外は無視します。
            try
            {
                lock (this.SyncRoot)
                {
                    if (this.BaseConnection != null)
                    {
                        this.BaseConnection.Close();
                        this.BaseConnection = null;
                    }
                }
            }
            catch (Exception)
            {
            }
        }

        /// <summary>
        /// コネクションから指定したバイト数のデータを同期的に読み出します。
        /// </summary>
        /// <exception cref="EndOfStreamException">
        /// 指定されたバイト数を読み出す前にコネクションが閉じられました。
        /// </exception>
        /// <exception cref="ObjectDisposedException">
        /// コネクションは閉じられています。
        /// </exception>
        /// <exception cref="OperationCanceledException">
        /// キャンセルイベントが設定されました。
        /// </exception>
        /// <exception cref="IOException">
        /// 予期しない例外です。
        /// </exception>
        public void Read(byte[] buffer, int offset, int count)
        {
            IAsyncResult ar = this.BeginRead(buffer, offset, count, null);

            this.EndRead(ar);
        }

        /// <summary>
        /// コネクションから指定したバイト数のデータの非同期読み出しを開始します。
        /// </summary>
        /// <exception cref="EndOfStreamException">
        /// 指定されたバイト数を読み出す前にコネクションが閉じられました。
        /// </exception>
        /// <exception cref="ObjectDisposedException">
        /// コネクションは閉じられています。
        /// </exception>
        /// <exception cref="OperationCanceledException">
        /// キャンセルイベントが設定されました。
        /// </exception>
        /// <exception cref="IOException">
        /// 予期しない例外です。
        /// </exception>
        public IAsyncResult BeginRead(
            byte[] buffer,
            int offset,
            int count,
            AsyncCallback callback)
        {
            this.CheckCancel();

            //
            // 非同期読み出しは繰り返し呼び出される可能性があるので、
            // デリゲート生成のコストを避けるために状態オブジェクトを使います。
            //
            IOAsyncState state = new IOAsyncState(buffer, offset, count, callback);

            lock (this.SyncRoot)
            {
                try
                {
                    this.GetBaseConnection().BeginReceive(
                        buffer,
                        offset,
                        count,
                        SocketFlags.None,
                        ReadAsyncCallback_,
                        state);
                }
                // NOTE:
                // ObjectDisposedException は InvalidOperationException の導出クラスなので順番に注意。
                catch (ObjectDisposedException)
                {
                    throw;
                }
                catch (SocketException ex)
                {
                    throw new EndOfStreamException("", ex);
                }
                catch (InvalidOperationException ex)
                {
                    throw new EndOfStreamException("", ex);
                }
                catch (Exception ex)
                {
                    throw new IOException("", ex);
                }
            }

            return state.AsyncResult;
        }

        private void ReadAsyncCallback_(IAsyncResult ar)
        {
            IOAsyncState state = (IOAsyncState)ar.AsyncState;

            try
            {
                this.CheckCancel();

                int c;

                lock (this.SyncRoot)
                {
                    c = this.GetBaseConnection().EndReceive(ar);
                }

                if (c == 0)
                {
                    throw new EndOfStreamException();
                }

                state.DoneBytes += c;

                if (state.DoneBytes < state.Count)
                {
                    lock (this.SyncRoot)
                    {
                        this.GetBaseConnection().BeginReceive(
                            state.Buffer,
                            state.Offset + state.DoneBytes,
                            state.Count - state.DoneBytes,
                            SocketFlags.None,
                            ReadAsyncCallback_,
                            state);

                        return;
                    }
                }
                else
                {
                    state.AsyncResult.Complete();
                }
            }
            catch (Exception ex)
            {
                state.AsyncResult.Complete(ex);
            }

            if (state.Callback != null)
            {
                state.Callback(state.AsyncResult);
            }
        }

        /// <summary>
        /// 読み出し操作を非同期的に完了します。
        /// </summary>
        /// <param name="ar">BeginRead の呼び出しによって返される IAsyncResult オブジェクト。</param>
        /// <exception cref="EndOfStreamException">
        /// 指定されたバイト数を読み出す前にコネクションが閉じられました。
        /// </exception>
        /// <exception cref="ObjectDisposedException">
        /// コネクションは閉じられています。
        /// </exception>
        /// <exception cref="OperationCanceledException">
        /// キャンセルイベントが設定されました。
        /// </exception>
        /// <exception cref="IOException">
        /// 予期しない例外です。
        /// </exception>
        public void EndRead(IAsyncResult ar)
        {
            this.WaitAsync(ar);

            Exception ex = ((AsyncResult)ar).Exception;

            if (ex != null)
            {
                // StackTrace を保持するため、InnerException を使用します。

                // NOTE:
                // ObjectDisposedException は InvalidOperationException の導出クラスなので順番に注意。
                if (ex is ObjectDisposedException)
                {
                    throw new ObjectDisposedException("", ex);
                }

                if (ex is OperationCanceledException)
                {
                    throw new OperationCanceledException("", ex);
                }

                if (ex is EndOfStreamException ||
                    ex is SocketException ||
                    ex is IOException ||
                    ex is InvalidOperationException)
                {
                    throw new EndOfStreamException("", ex);
                }

                throw new IOException("", ex);
            }
        }

        /// <summary>
        /// 読み出しデータを指定バイト無視します。
        /// </summary>
        /// <param name="count">無視するバイト数です。</param>
        public void Skip(int count)
        {
            byte[] buff = new byte[1024];

            while (0 < count)
            {
                int readBytes = Math.Min(buff.Length, count);
                this.Read(buff, 0, readBytes);
                count -= readBytes;
            }
        }

        /// <summary>
        /// コネクションへ指定したバイト数のデータを同期的に書き出します。
        /// </summary>
        /// <exception cref="EndOfStreamException">
        /// 指定されたバイト数を書き出す前にコネクションが閉じられました。
        /// </exception>
        /// <exception cref="ObjectDisposedException">
        /// コネクションは閉じられています。
        /// </exception>
        /// <exception cref="OperationCanceledException">
        /// キャンセルイベントが設定されました。
        /// </exception>
        /// <exception cref="IOException">
        /// 予期しない例外です。
        /// </exception>
        public void Write(
            byte[] buffer,
            int offset,
            int count)
        {
            IAsyncResult ar = this.BeginWrite(buffer, offset, count, null);

            this.EndWrite(ar);
        }

        /// <summary>
        /// コネクションへ指定したバイト数のデータの非同期書き出しを開始します。
        /// </summary>
        /// <exception cref="EndOfStreamException">
        /// 指定されたバイト数を書き出す前にコネクションが閉じられました。
        /// </exception>
        /// <exception cref="ObjectDisposedException">
        /// コネクションは閉じられています。
        /// </exception>
        /// <exception cref="OperationCanceledException">
        /// キャンセルイベントが設定されました。
        /// </exception>
        /// <exception cref="IOException">
        /// 予期しない例外です。
        /// </exception>
        public IAsyncResult BeginWrite(
            byte[] buffer,
            int offset,
            int count,
            AsyncCallback callback)
        {
            this.CheckCancel();

            lock (this.SyncRoot)
            {
                try
                {
                    return this.GetBaseConnection().BeginSend(buffer, offset, count, SocketFlags.None, callback, null);
                }
                // NOTE:
                // ObjectDisposedException は InvalidOperationException の導出クラスなので順番に注意。
                catch (ObjectDisposedException)
                {
                    throw;
                }
                catch (SocketException ex)
                {
                    throw new EndOfStreamException("", ex);
                }
                catch (InvalidOperationException ex)
                {
                    throw new EndOfStreamException("", ex);
                }
                catch (Exception ex)
                {
                    throw new IOException("", ex);
                }
            }
        }

        /// <summary>
        /// 書き出し操作を非同期的に完了します。
        /// </summary>
        /// <param name="ar">BeginWrite の呼び出しによって返される IAsyncResult オブジェクト。</param>
        /// <exception cref="EndOfStreamException">
        /// 指定されたバイト数を書き出す前にコネクションが閉じられました。
        /// </exception>
        /// <exception cref="ObjectDisposedException">
        /// コネクションは閉じられています。
        /// </exception>
        /// <exception cref="OperationCanceledException">
        /// キャンセルイベントが設定されました。
        /// </exception>
        /// <exception cref="IOException">
        /// 予期しない例外です。
        /// </exception>
        public void EndWrite(IAsyncResult ar)
        {
            this.WaitAsync(ar);

            lock (this.SyncRoot)
            {
                try
                {
                    this.GetBaseConnection().EndSend(ar);
                }
                // NOTE:
                // ObjectDisposedException は InvalidOperationException の導出クラスなので順番に注意。
                catch (ObjectDisposedException)
                {
                    throw;
                }
                catch (SocketException ex)
                {
                    throw new ObjectDisposedException("", ex);
                }
                catch (InvalidOperationException ex)
                {
                    throw new ObjectDisposedException("", ex);
                }
                catch (Exception ex)
                {
                    throw new IOException("", ex);
                }
            }
        }

        /// <summary>
        /// アンマネージ リソースを開放します。
        ///
        /// このメソッドは public ではありません。
        /// 基になるコネクションの破棄には Close メソッドを使用します。
        /// </summary>
        void IDisposable.Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// メンバーを解放します。
        /// </summary>
        /// <param name="disposing">
        /// マネージ リソースとアンマネージ リソースの両方を解放する場合は true。
        /// アンマネージ リソースだけを解放する場合は false。
        /// </param>
        protected virtual void Dispose(bool disposing)
        {
            if (this.BaseConnection != null)
            {
                if (disposing)
                {
                    (this.BaseConnection as IDisposable).Dispose();
                }

                this.BaseConnection = null;
            }
        }
    }
}
