﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Nintendo.McsServer.McsUtil
{
    using System;
    using System.Diagnostics;
    using System.Threading;

    /// <summary>
    /// 非同期操作のステータスを表わす IAsyncResult の実装です。
    /// </summary>
    public class AsyncResult : IAsyncResult
    {
        private object _SyncObj = new object();
        private ManualResetEvent _AsyncWaitHandle = null;
        private bool _IsCompleted = false;

        public AsyncResult()
        {
        }

        public AsyncResult(object asyncState)
        {
            this.AsyncState = asyncState;
        }

        public object AsyncState
        {
            get;
            set;
        }

        public WaitHandle AsyncWaitHandle
        {
            get
            {
                lock (_SyncObj)
                {
                    if (_AsyncWaitHandle == null)
                    {
                        _AsyncWaitHandle = new ManualResetEvent(_IsCompleted);
                    }

                    return _AsyncWaitHandle;
                }
            }
        }

        public bool CompletedSynchronously
        {
            get { return false; }
        }

        public bool IsCompleted
        {
            get
            {
                lock (_SyncObj)
                {
                    return _IsCompleted;
                }
            }
        }

        /// <summary>
        /// 非同期処理中に発生した例外です。
        /// </summary>
        public Exception Exception
        {
            get;
            set;
        }

        /// <summary>
        /// 非同期処理が完了したことを宣言します。
        /// </summary>
        public void Complete()
        {
            Debug.Assert(!_IsCompleted);

            lock (_SyncObj)
            {
                if (_AsyncWaitHandle != null)
                {
                    _AsyncWaitHandle.Set();
                }

                _IsCompleted = true;
            }
        }

        /// <summary>
        /// 例外により非同期処理が完了したことを宣言します。
        /// </summary>
        public void Complete(Exception ex)
        {
            Debug.Assert(!_IsCompleted);

            lock (_SyncObj)
            {
                if (_AsyncWaitHandle != null)
                {
                    try
                    {
                        _AsyncWaitHandle.Set();
                    }
                    catch (ObjectDisposedException)
                    {
                        // コールバックがキャンセルを検出する前に、呼び出し側スレッドが
                        // コールバックの完了待ちでキャンセルを認識すると、その時点で
                        // _AsyncWaitHandle.Close()が行われる。
                    }
                }

                this.Exception = ex;
                _IsCompleted = true;
            }
        }
    }

    /// <summary>
    /// 非同期操作のステータスを表わす IAsyncResult の実装です。
    /// </summary>
    /// <typeparam name="T">非同期処理の完了時の戻り値型。</typeparam>
    public class AsyncResult<T> : IAsyncResult
    {
        private object _SyncObj = new object();
        private ManualResetEvent _AsyncWaitHandle = null;
        private bool _IsCompleted = false;

        public AsyncResult()
        {
        }

        public AsyncResult(object asyncState)
        {
            this.AsyncState = asyncState;
        }

        public object AsyncState
        {
            get;
            set;
        }

        public WaitHandle AsyncWaitHandle
        {
            get
            {
                lock (_SyncObj)
                {
                    if (_AsyncWaitHandle == null)
                    {
                        _AsyncWaitHandle = new ManualResetEvent(_IsCompleted);
                    }

                    return _AsyncWaitHandle;
                }
            }
        }

        public bool CompletedSynchronously
        {
            get { return false; }
        }

        public bool IsCompleted
        {
            get
            {
                lock (_SyncObj)
                {
                    return _IsCompleted;
                }
            }
        }

        /// <summary>
        /// 非同期処理の完了時の戻り値です。
        /// </summary>
        public T Result
        {
            get;
            private set;
        }

        /// <summary>
        /// 非同期処理中に発生した例外です。
        /// </summary>
        public Exception Exception
        {
            get;
            set;
        }

        /// <summary>
        /// 非同期処理が完了したことを宣言します。
        /// </summary>
        /// <param name="result">非同期処理の完了時の戻り値。</param>
        public void Complete(T result)
        {
            Debug.Assert(!_IsCompleted);

            lock (_SyncObj)
            {
                if (_AsyncWaitHandle != null)
                {
                    _AsyncWaitHandle.Set();
                }

                this.Result = result;
                _IsCompleted = true;
            }
        }

        /// <summary>
        /// 例外により非同期処理が完了したことを宣言します。
        /// </summary>
        /// <param name="exception">非同期処理中に生じた例外オブジェクト。</param>
        public void Complete(Exception exception)
        {
            Debug.Assert(!_IsCompleted);

            lock (_SyncObj)
            {
                if (_AsyncWaitHandle != null)
                {
                    _AsyncWaitHandle.Set();
                }

                this.Exception = exception;
                _IsCompleted = true;
            }
        }
    }
}
