﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Threading;
using System.Diagnostics;

using System.Runtime.Remoting;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Channels.Ipc;

namespace Nintendo.McsServer
{
    static class Program
    {
        const string GuidStr = "{3A8E47FF-C117-4078-951D-3930285DBAEF}";
        const string IpcChannelPortName = GuidStr;
        const string ObjectUriName = "ServerControl";
        const bool EnsureSecurity = true;

        internal static readonly RuntimeSettings RuntimeSettings = new RuntimeSettings();

        /// <summary>
        /// cygwin の仕様変更によって、1.7.10 以降の bash からツールが実行された場合、
        /// tmp と TMP など大文字・小文字違いの同名の環境変数が定義され、
        /// XmlSerializer内部で例外が発生してしまう問題があります。
        /// 対策として、小文字の環境変数を削除します。
        /// </summary>
        static void RemoveMultipleEnvironmentVariableWorkaround_(string envName)
        {
            // 環境変数を取得する。
            string envVal = Environment.GetEnvironmentVariable(envName);
            if (string.IsNullOrEmpty(envVal))
            {
                return;
            }

            // 手始めに一つ削除して、その後再度取得してみる。
            Environment.SetEnvironmentVariable(envName, null);
            string envVal2 = Environment.GetEnvironmentVariable(envName);
            if (!string.IsNullOrEmpty(envVal2))
            {
                // 2つ目が取得できた場合は、多重定義されている状況。
                // POSIX スタイルの環境変数を削除するようにする。
                if (envVal2.StartsWith("/"))
                {
                    Environment.SetEnvironmentVariable(envName, envVal);
                }
                return;
            }
            else
            {
                // 多重定義がなかったので、削除した環境変数を元に元しておく。
                Environment.SetEnvironmentVariable(envName, envVal);
            }
        }

        /// <summary>
        /// アプリケーションのメイン エントリ ポイントです。
        /// </summary>
        [STAThread]
        static void Main(string[] args)
        {
            // cygwin 挙動変更による環境変数多重定義の回避コード
            RemoveMultipleEnvironmentVariableWorkaround_("tmp");
            RemoveMultipleEnvironmentVariableWorkaround_("temp");

            // 強制的に en にする
            Thread.CurrentThread.CurrentUICulture = new System.Globalization.CultureInfo("en");

#if false
            // 規定の言語で無い場合は英語のリソースにする。(大文字小文字を無視し、カルチャに依存しない比較を行う)
            if (0 != string.Compare(Thread.CurrentThread.CurrentUICulture.Name, "ja-JP", true, System.Globalization.CultureInfo.InvariantCulture))
            {
                Thread.CurrentThread.CurrentUICulture = new System.Globalization.CultureInfo("en", false);
            }
#endif

            // 2重起動の禁止
            // 異なるポート番号で待機できるようにすれば、複数のmcsサーバを起動することは可能だ。
            // その時、異なるポート番号の設定をどのように保存するのか、自分が何番目のインスタンスで何番目の設定のポート番号を選択すべきなのかを
            // 取得する必要がある。
            bool mutexWasCreated;
            using (Mutex m = new Mutex(true, GuidStr, out mutexWasCreated))
            {
                if (mutexWasCreated)
                {
                    Application.EnableVisualStyles();
                    Application.SetCompatibleTextRenderingDefault(false);
                    MainForm mainForm = new MainForm();

                    ServerControl serverControl = RegisterServerObject(mainForm);
                    GC.KeepAlive(serverControl);    // serverControlがガベージコレクトされないようにする。

                    // 起動時ウィンドウ非表示に対応するために、
                    // Application.Run() の引数に mainForm を渡さないようにしています。
                    Application.Run();
                }
                else
                {
                    Debug.WriteLine("mutex was created.");

                    ServerControl serverControl = RegisterClientObject();
                    try
                    {
                        serverControl.SendAppArgs(args);
                    }
                    catch (RemotingException ex)
                    {
                        MessageBox.Show(ex.ToString(), Application.ProductName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }

        static ServerControl RegisterServerObject(MainForm mainForm)
        {
            ServerControl serverControl = new ServerControl();
            serverControl.SetMainForm(mainForm);

            // Register the server channel.
            ChannelServices.RegisterChannel(new IpcChannel(IpcChannelPortName), EnsureSecurity);

            // Creates the single instance of ServiceClass. All clients will use this instance.
            ObjRef ref1 = RemotingServices.Marshal(serverControl, ObjectUriName);
            Debug.WriteLine("ObjRef.URI: " + ref1.URI);

            return serverControl;
        }

        static ServerControl RegisterClientObject()
        {
            // Register the channel.
            ChannelServices.RegisterChannel(new IpcChannel(), EnsureSecurity);

            // Register as client for remote object.
            WellKnownClientTypeEntry remoteType = new WellKnownClientTypeEntry(
                typeof(ServerControl),
                "ipc://" + IpcChannelPortName + "/" + ObjectUriName);
            RemotingConfiguration.RegisterWellKnownClientType(remoteType);

            // Create an instance of the remote object.
            return new ServerControl();
        }
    }
}
