﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Threading;
using System.Net;
using System.Net.Sockets;
using System.IO;
using System.Diagnostics;
using System.Text;
using System.Runtime.InteropServices;
using System.Collections;
using System.Collections.Generic;

using System.Windows.Forms;

namespace Nintendo.McsServer
{
    enum FileIOCommandType : int
    {
        FILEOPEN        = 0,
        FILECLOSE,
        FILEREAD,
        FILEWRITE,
        FILESEEK,

        FINDFIRST       = 16,
        FINDNEXT,
        FINDCLOSE,

        SHOWFILEDLG     = 32,
        EXPANDPATH
}

    [Flags()]
    enum FileIOFlagType : uint
    {
        Read        = 1 << 0,
        Write       = 1 << 1,
        ReadWrite   = Read | Write,

        NoOverwrite = 1 << 2,
        NoCreate    = 1 << 3,
        IncEnvVar   = 1 << 4,
        CreateDir   = 1 << 5,

        ShareRead   = 1 << 16,
        ShareWrite  = 1 << 17
    }

    enum FileIOFileDialogFlagType : uint
    {
        Open                = 0,
        Save                = 0x40000000,

        ReadOnlyChecked     = 0x00000001,
        NoOverwritePrompt   = 0x00000002,
        ShowReadOnly        = 0x00000004,
        MultiSelect         = 0x00000200,
        UncheckPathExists   = 0x00000800,
        UncheckFileExists   = 0x00001000,
        CreatePrompt        = 0x00002000,

        IncludeFileName     = 0x80000000,

    }

    enum FileIOErrorType : int
    {
        Success             = 0,
        ComError,                   // 通信エラー
        NotConnect,                 // 未接続

        Continue            =100,   // 実行を継続中
        ProtocolError,              // バージョン違いなどによるプロトコルエラー
        FileNotFound,
        DirectoryNotFound,
        FileAlreadyExist,
        SecurityError,
        UnauthorizedAccess,
        IOError,                    // 未分類のIO関連のエラー
        InvalidFileHandle,
        NoMoreFiles,
        Canceled
    }

    enum FileIOSeekOrigin : int
    {
        Begin,
        Current,
        End
    };

    class ErrorResponseException : MCSException
    {
        public readonly FileIOErrorType     ErrorCode;

        public ErrorResponseException(FileIOErrorType errorCode)
            : base()
        {
            ErrorCode = errorCode;
        }
    }

    class FileIOChunkHeader
    {
        int         _id;
        uint        _dataSize;

        public FileIOChunkHeader(int id, uint dataSize)
        {
            _id = id;
            _dataSize = dataSize;
        }

        public FileIOChunkHeader(Stream stream)
        {
            NetBinaryReader br = new NetBinaryReader(stream);
            _id = br.ReadInt32();
            _dataSize = br.ReadUInt32();
        }

#if false
        public void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            bw.Write(_id);
            bw.Write(_dataSize);
        }
#endif

        public FileIOCommandType ID
        {
            get
            {
                return (FileIOCommandType)_id;
            }
        }

        public uint DataSize
        {
            get
            {
                return _dataSize;
            }
        }
    }

    class FileIOCommandHeader
    {
        public readonly uint        ClnFileInfoPtr;
        public readonly IntPtr      Handle;

        public FileIOCommandHeader(NetBinaryReader br, uint dataSize)
        {
            if ( dataSize > FileIOServer.FILEIO_CHUNK_DATASIZE_MAX  // 最大チャンクデータサイズを超えている
              || (IsExtend ? dataSize < HeaderSize: dataSize != HeaderSize)
            )
            {
                throw new ErrorResponseException(FileIOErrorType.ProtocolError);
            }

            ClnFileInfoPtr = br.ReadUInt32();
            Handle = (IntPtr)br.ReadInt32();
        }

        public virtual uint HeaderSize
        {
            get
            {
                return 4 + 4;
            }
        }

        public virtual bool IsExtend
        {
            get
            {
                return false;
            }
        }
    }

    class FileIOExpandPathCommandHeader
    {
        public readonly string         FileName;

        public FileIOExpandPathCommandHeader(NetBinaryReader br, uint dataSize)
        {
            if ( dataSize > FileIOServer.FILEIO_CHUNK_DATASIZE_MAX  // 最大チャンクデータサイズを超えている
                || (dataSize != HeaderSize)
                )
            {
                throw new ErrorResponseException(FileIOErrorType.ProtocolError);
            }

            FileName = FileIOServer.ReadFileNameByteArray(br);
        }

        public virtual uint HeaderSize
        {
            get
            {
                return FileIOServer.FILEIO_PATH_BYTES;
            }
        }

        public virtual bool IsExtend
        {
            get
            {
                return false;
            }
        }
    }

    class FileIOOpenCommandHeader : FileIOCommandHeader
    {
        public readonly uint            Mode;
        public readonly string          FileName;

        public FileIOOpenCommandHeader(NetBinaryReader br, uint dataSize)
            : base(br, dataSize)
        {
            Mode = br.ReadUInt32();
            FileName = FileIOServer.ReadFileNameByteArray(br);
        }

        public override uint HeaderSize
        {
            get
            {
                return base.HeaderSize + 4 + FileIOServer.FILEIO_PATH_BYTES;
            }
        }

        public FileIOFlagType Flag
        {
            get
            {
                return (FileIOFlagType)(uint)Handle;
            }
        }
    }

    class FileIOReadCommandHeader : FileIOCommandHeader
    {
        public readonly uint            ClnBufPtr;
        public readonly uint            ReadSize;

        public FileIOReadCommandHeader(NetBinaryReader br, uint dataSize)
            : base(br, dataSize)
        {
            ClnBufPtr = br.ReadUInt32();
            ReadSize = br.ReadUInt32();
        }

        public override uint HeaderSize
        {
            get
            {
                return base.HeaderSize + 4 + 4;
            }
        }
    }

    class FileIOWriteCommandHeader : FileIOCommandHeader
    {
        public readonly uint        WriteSize;

        public FileIOWriteCommandHeader(NetBinaryReader br, uint dataSize)
            : base(br, dataSize)
        {
            WriteSize = br.ReadUInt32();

            if (dataSize != HeaderSize + WriteSize
              || WriteSize + HeaderSize > FileIOServer.FILEIO_CHUNK_DATASIZE_MAX     // 最大チャンクデータサイズを超えている
            )
            {
                throw new ErrorResponseException(FileIOErrorType.ProtocolError);
            }
        }

        public override uint HeaderSize
        {
            get
            {
                return base.HeaderSize + 4;
            }
        }

        public override bool IsExtend
        {
            get
            {
                return true;
            }
        }
    }

    class FileIOSeekCommandHeader : FileIOCommandHeader
    {
        public readonly int                 DistanceToMove;
        public readonly FileIOSeekOrigin    SeekOrigin;

        public FileIOSeekCommandHeader(NetBinaryReader br, uint dataSize)
            : base(br, dataSize)
        {
            DistanceToMove = br.ReadInt32();
            SeekOrigin = (FileIOSeekOrigin)br.ReadInt32();
        }

        public override uint HeaderSize
        {
            get
            {
                return base.HeaderSize + 4 + 4;
            }
        }
    }

    class FileIOFindFirstCommandHeader : FileIOCommandHeader
    {
        public readonly uint            Mode;
        public readonly string          FileName;

        public FileIOFindFirstCommandHeader(NetBinaryReader br, uint dataSize)
            : base(br, dataSize)
        {
            Mode = br.ReadUInt32();
            FileName = FileIOServer.ReadFileNameByteArray(br);
        }

        public override uint HeaderSize
        {
            get
            {
                return base.HeaderSize + 4 + FileIOServer.FILEIO_PATH_BYTES;
            }
        }

        public uint ClnFindDataPtr
        {
            get
            {
                return (uint)Handle;
            }
        }
    }

    class FileIOFindNextCommandHeader : FileIOCommandHeader
    {
        public readonly uint            ClnFindDataPtr;

        public FileIOFindNextCommandHeader(NetBinaryReader br, uint dataSize)
            : base(br, dataSize)
        {
            ClnFindDataPtr = br.ReadUInt32();
        }

        public override uint HeaderSize
        {
            get
            {
                return base.HeaderSize + 4;
            }
        }
    }

    class FileIOShowFileDlgCommandHeader
    {
        public readonly FileIOFileDialogFlagType    Flag;
        public readonly uint        Mode;
        public readonly string      InitialPath;
        public readonly string      Filter;
        public readonly string      DefaultExt;
        public readonly string      Title;

        public FileIOShowFileDlgCommandHeader(NetBinaryReader br, uint dataSize)
        {
            if ( dataSize > FileIOServer.FILEIO_CHUNK_DATASIZE_MAX  // 最大チャンクデータサイズを超えている
                || (dataSize != HeaderSize)
                )
            {
                throw new ErrorResponseException(FileIOErrorType.ProtocolError);
            }

            Flag = (FileIOFileDialogFlagType)br.ReadUInt32();
            Mode = br.ReadUInt32();
            br.ReadInt32(); // reserved
            InitialPath = FileIOServer.ReadFileNameByteArray(br);
            Filter = FileIOServer.ReadStringFromByteArray(br, 256 * 2);
            DefaultExt = FileIOServer.ReadStringFromByteArray(br, 16 * 2);
            Title = FileIOServer.ReadStringFromByteArray(br, 256 * 2);
        }

        public virtual uint HeaderSize
        {
            get
            {
                return 4 + 4 + 4 + FileIOServer.FILEIO_PATH_BYTES + 256*2 + 16*2 + 256*2;
            }
        }

        public virtual bool IsExtend
        {
            get
            {
                return false;
            }
        }
    }

    class FileIOResponseHeader
    {
        public readonly FileIOCommandType   ChunkID;

        public FileIOErrorType      ErrorCode       = FileIOErrorType.Success;
        public uint                 ClnFileInfoPtr;

        public FileIOResponseHeader(FileIOCommandType chunkID)
        {
            ChunkID = chunkID;
        }

        public void ToStream(Stream stream)
        {
            MemoryStream oms = new MemoryStream();
            NetBinaryWriter bw = new NetBinaryWriter(oms);
            ToWriter(bw);
            bw.Flush();

            stream.Write(oms.GetBuffer(), 0, (int)oms.Length);
        }

        protected virtual void ToWriter(BinaryWriter bw)
        {
            bw.Write((int)ChunkID);
            bw.Write(HeaderSize + DataSize);

            bw.Write((int)ErrorCode);
            bw.Write(ClnFileInfoPtr);
        }

        public virtual uint HeaderSize
        {
            get
            {
                return 4 + 4;
            }
        }

        public virtual uint DataSize
        {
            get
            {
                return 0;
            }
            set
            {
            }
        }
    }

    class FileIOOpenResponseHeader : FileIOResponseHeader
    {
        public IntPtr               Handle;
        public long                 FileSize;

        public FileIOOpenResponseHeader()
            : base(FileIOCommandType.FILEOPEN)
        {
        }

        protected override void ToWriter(BinaryWriter bw)
        {
            base.ToWriter(bw);

            bw.Write((int)Handle);
            bw.Write((uint)FileSize);
        }

        public override uint HeaderSize
        {
            get
            {
                return base.HeaderSize + 4 + 4;
            }
        }
    }

    class FileIOReadResponseHeader : FileIOResponseHeader
    {
        public uint                 ClnBufPtr;
        uint                        _dataSize;
        public uint                 TotalSentSize;

        public FileIOReadResponseHeader()
            : base(FileIOCommandType.FILEREAD)
        {
        }

        protected override void ToWriter(BinaryWriter bw)
        {
            base.ToWriter(bw);

            bw.Write(ClnBufPtr);
            bw.Write(DataSize);
            bw.Write(TotalSentSize);
        }

        public override uint HeaderSize
        {
            get
            {
                return base.HeaderSize + 4 + 4 + 4;
            }
        }

        public override uint DataSize
        {
            get
            {
                return _dataSize;
            }
            set
            {
                _dataSize = value;
            }
        }
    }

    class FileIOSeekResponseHeader : FileIOResponseHeader
    {
        public uint                 FilePointer;

        public FileIOSeekResponseHeader()
            : base(FileIOCommandType.FILESEEK)
        {
        }

        protected override void ToWriter(BinaryWriter bw)
        {
            base.ToWriter(bw);

            bw.Write(FilePointer);
        }

        public override uint HeaderSize
        {
            get
            {
                return base.HeaderSize + 4;
            }
        }
    }

    class FileIOFindResponseHeader : FileIOResponseHeader
    {
        public uint                 ClnFindDataPtr;
        public IntPtr               Handle;
        public long                 FileSize;
        public FileAttributes       FileAttribute;
        public DateTime             FileDate;
        public string               FileName;

        public FileIOFindResponseHeader(FileIOCommandType chunkID)
            : base(chunkID)
        {
        }

        protected override void ToWriter(BinaryWriter bw)
        {
            base.ToWriter(bw);

            bw.Write(ClnFindDataPtr);
            bw.Write((int)Handle);
            bw.Write((uint)FileSize);
            bw.Write((int)FileAttribute);
            TimeSpan span = FileDate - new DateTime(1970, 1, 1, 0, 0, 0);   // ローカル時間 1970/1/1 00:00:00 からの経過時間
            bw.Write((uint)span.TotalSeconds);
            bw.Write((uint)0);  // reserved1
            bw.Write((uint)0);  // reserved2

            byte[] writeBuf = new byte[FileIOServer.FILEIO_PATH_BYTES];
            // ファイル名
            if (FileName != null)
            {
                int fileNameLength = Math.Min(FileName.Length, FileIOServer.FILEIO_PATH_MAX - 1);

                Encoding.Default.GetBytes(FileName, 0, fileNameLength, writeBuf, 0);
                writeBuf[FileIOServer.FILEIO_PATH_BYTES - 1] = 0;     // Cの終端文字列をセットしておく
                writeBuf[FileIOServer.FILEIO_PATH_BYTES - 2] = 0;     // Cの終端文字列をセットしておく
            }

            bw.Write(writeBuf, 0, FileIOServer.FILEIO_PATH_BYTES);
        }

        public override uint HeaderSize
        {
            get
            {
                return base.HeaderSize + 4 + 4 + 4 + 4 + 4   + 4 + 4   + FileIOServer.FILEIO_PATH_BYTES;
            }
        }

        public void SetFileSystemInfo(FileSystemInfo info)
        {
            FileSize = info is FileInfo ? ((FileInfo)info).Length: 0;
            FileAttribute = info.Attributes;
            FileDate = info.LastWriteTime;
            FileName = info.Name;
        }
    }

    class FileIOShowFileDlgResponseHeader
    {
        public readonly FileIOCommandType   ChunkID = FileIOCommandType.SHOWFILEDLG;

        public FileIOErrorType      ErrorCode       = FileIOErrorType.Success;
        uint                        _dataSize;

        public FileIOShowFileDlgResponseHeader()
        {
        }

        public void ToStream(Stream stream)
        {
            MemoryStream oms = new MemoryStream();
            NetBinaryWriter bw = new NetBinaryWriter(oms);
            ToWriter(bw);
            bw.Flush();

            stream.Write(oms.GetBuffer(), 0, (int)oms.Length);
        }

        protected void ToWriter(BinaryWriter bw)
        {
            bw.Write((int)ChunkID);
            bw.Write(HeaderSize + DataSize);

            bw.Write((int)ErrorCode);
            bw.Write(DataSize);
        }

        public uint HeaderSize
        {
            get
            {
                return 4 + 4;
            }
        }

        public uint DataSize
        {
            get
            {
                return _dataSize;
            }
            set
            {
                _dataSize = value;
            }
        }
    }

    class FileIOExpandPathResponseHeader
    {
        public readonly FileIOCommandType ChunkID = FileIOCommandType.EXPANDPATH;

        public FileIOErrorType ErrorCode = FileIOErrorType.Success;
        uint _dataSize;

        public FileIOExpandPathResponseHeader()
        {
        }

        public void ToStream(Stream stream)
        {
            MemoryStream oms = new MemoryStream();
            NetBinaryWriter bw = new NetBinaryWriter(oms);
            ToWriter(bw);
            bw.Flush();

            stream.Write(oms.GetBuffer(), 0, (int)oms.Length);
        }

        protected void ToWriter(BinaryWriter bw)
        {
            bw.Write((int)ChunkID);
            bw.Write(HeaderSize + DataSize);

            bw.Write((int)ErrorCode);
            bw.Write(DataSize);
        }

        public uint HeaderSize
        {
            get
            {
                return 4 + 4;
            }
        }

        public uint DataSize
        {
            get
            {
                return _dataSize;
            }
            set
            {
                _dataSize = value;
            }
        }
    }

    /// <summary>
    /// FileIOを処理するクラス。
    /// FileIO用のチャンネルをRouterに登録して動作する。
    ///
    /// 実機ライブラリは、FileIO用チャンネルに対してFileIOコマンド
    /// を送信する。
    ///
    /// 本クラスは実機側が送信する、コマンドを別スレッドで
    /// 読み取り待機しており、(ThreadStartFunc())
    /// コマンドが受信され次第、コマンド種類に応じた処理をおこなって、
    /// 処理結果をレスポンスデータという形式で、実機側に送り返す。
    ///
    /// 実機側は、コマンド送信後、レスポンスデータの受信を待機して
    /// 待っている。
    /// 無事に処理が完了し、レスポンスを受信されれば実機側は次の処理に移る。
    ///
    /// 何らかの理由で一定時間にレスポンスが受信されなかった場合は、
    /// タイムアウトエラーとして処理する。
    /// </summary>
    public class FileIOServer
    {
        public const int        FILEIO_PATH_MAX = 260;
        public const int        FILEIO_PATH_BYTES = FILEIO_PATH_MAX * 2;

        public const uint       FILEIO_CHANNEL      = 0xFF7F;

        internal const int      FILEIO_CHUNK_SIZE_MAX       = 16 * 1024;
        internal const int      FILEIO_CHUNKHEAD_SIZE       = 4 + 4;
        /// <summary>
        /// File I/O 送受信のデータの最大サイズ
        /// </summary>
        internal const int      FILEIO_CHUNK_DATASIZE_MAX   = FILEIO_CHUNK_SIZE_MAX - FILEIO_CHUNKHEAD_SIZE;

        /// <summary>
        /// スレッド同期用変数
        /// </summary>
        readonly object         _lockObj = new object();
        volatile bool           _vbStart;
        Thread                  _thread;
        readonly Form           _form;

        /// <summary>
        /// 開いたファイルストリームのセット
        /// </summary>
        readonly List<IntPtr>   _fileStreamArray = new List<IntPtr>();
        /// <summary>
        /// 開いたディレクトリのセット
        /// </summary>
        readonly List<IntPtr>   _filesEnumArray = new List<IntPtr>();

        /// <summary>
        /// MCS Serverの待機ポート番号
        /// </summary>
        int                     _portNo         = UserSettings.DefaultPortNo;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FileIOServer(Form form)
        {
            _form = form;
        }

        /// <summary>
        /// 動作開始
        /// </summary>
        public void Start(int portNo)
        {
            if (_vbStart)
            {
                return ;
            }

            lock (_lockObj)
            {
                _portNo = portNo;

                _thread = new Thread(new ThreadStart(ThreadStartFunc));
                _thread.IsBackground = true;    // バックグラウンドスレッドとする。

                _vbStart = true;

                _thread.Start();
            }
        }

        /// <summary>
        /// 動作停止
        /// </summary>
        public void Stop()
        {
            if (!_vbStart)
            {
                return ;
            }

            lock (_lockObj)
            {
                _vbStart = false;

                _thread.Abort();
                // _thread.Join();

                _thread = null;
            }
        }

        /// <summary>
        /// FileIO用チャンネルを登録し、
        /// 実機側から送信されるコマンドパケットの受信を待ち受けます。
        ///
        /// コマンドが受信されたら、種類に応じて処理をおこない、
        /// 結果をレスポンスパケットとして返信します。
        /// </summary>
        void ThreadStartFunc()
        {
            IPHostEntry entry = Dns.GetHostEntry("localhost");
            IPEndPoint remoteEndPoint = new IPEndPoint(entry.AddressList[0], _portNo);

            while (_vbStart)    // Stop()が呼ばれるまでは、Socketが切断されても繰り返す。
            {
                try
                {
                    //----------------------------------------------------
                    // FileIO用チャンネルを登録します。
                    Socket socket = new Socket(remoteEndPoint.AddressFamily, SocketType.Stream, ProtocolType.Tcp);
                    socket.Connect(remoteEndPoint);

                    ChannelRequestResult result = ChannelRequest.Request(socket, FILEIO_CHANNEL, (uint)ChannelRequestFlags.NoCheckConnect);
                    if (result != ChannelRequestResult.Success)
                    {
                        socket.Close();
                        Debug.WriteLine(string.Format("File IO Sever : fail channel request. - {0}", (uint)result));
                        Thread.Sleep(10 * 1000);
                        continue;
                    }

                    const bool bOwnSocket = true;
                    using (NetworkStream netsm = new NetworkStream(socket, bOwnSocket))
                    {
                        //----------------------------------------------------
                        // 接続が切断されるまで、永久に
                        // FileIO用チャンネルでコマンドを待ち受けます。
                        while( true )
                        {
                            FileIOChunkHeader header = new FileIOChunkHeader(netsm);

                            // サイズ分をきっちり読み込む処理をどうするか...(???なぞのつぶやき)
                            // コマンドの種類に応じた処理を行います。
                            switch (header.ID)
                            {
                            case FileIOCommandType.FILEOPEN:    DoFileOpen(netsm, header.DataSize);         break;
                            case FileIOCommandType.FILECLOSE:   DoFileClose(netsm, header.DataSize);        break;
                            case FileIOCommandType.FILEREAD:    DoFileRead(netsm, header.DataSize);         break;
                            case FileIOCommandType.FILEWRITE:   DoFileWrite(netsm, header.DataSize);        break;
                            case FileIOCommandType.FILESEEK:    DoFileSeek(netsm, header.DataSize);         break;
                            case FileIOCommandType.FINDFIRST:   DoFindFirst(netsm, header.DataSize);        break;
                            case FileIOCommandType.FINDNEXT:    DoFindNext(netsm, header.DataSize);         break;
                            case FileIOCommandType.FINDCLOSE:   DoFindClose(netsm, header.DataSize);        break;
                            case FileIOCommandType.SHOWFILEDLG: DoShowFileDialog(netsm, header.DataSize);   break;
                            case FileIOCommandType.EXPANDPATH:  DoExpandPath(netsm, header.DataSize);       break;
                            default:
                                // 知らないコマンドのときは、読み飛ばしてunknownレスポンスを返す
                                Debug.Assert(false);
                                break;
                            }
                        }
                    }
                }
                catch (EndOfStreamException)
                {
                    // ネットワークからデータを読み取っているうちに、リモート側が終了していた場合
                    Debug.WriteLine("File IO Server : remote shutdown.");
                }
                catch (SocketException e)
                {
                    // ソケットに関する例外が発生した場合
                    switch (e.SocketErrorCode)
                    {
                    case SocketError.ConnectionReset:
                        RouterLog.ServerReport("File IO Server : closed by remote");
                        break;

                    default:
                        RouterLog.ServerReport(ServerReportType.InternalError, "File IO Server : SocketException Error code : {0}, {1}", e.ErrorCode, e.ToString());
                        break;
                    }
                }
                finally
                {
                    // すべてのファイルハンドルを開放します。
                    ReleaseAllFileHandle();
                }
            }
        }

        /// <summary>
        /// すべてのファイルハンドルを開放します。
        /// </summary>
        void ReleaseAllFileHandle()
        {
            foreach (IntPtr handle in _fileStreamArray)
            {
                CloseFileHandle(handle);
            }
            _fileStreamArray.Clear();

            foreach (IntPtr handle in _filesEnumArray)
            {
                CloseEnumHandle(handle);
            }
            _filesEnumArray.Clear();
        }

        /// <summary>
        /// ファイルを開きます。
        /// </summary>
        void DoFileOpen(Stream netSm, uint dataSize)
        {
            FileIOOpenResponseHeader resHead = new FileIOOpenResponseHeader();
            FileStream fs = null;

            try
            {
                FileIOOpenCommandHeader cmdHead = new FileIOOpenCommandHeader(new NetBinaryReader(netSm), dataSize);

                resHead.ClnFileInfoPtr = cmdHead.ClnFileInfoPtr;

                try
                {
                    FileIOFlagType openFlag = cmdHead.Flag;

                    FileInfo fileInfo = null;
                    try
                    {
                        string fileName = cmdHead.FileName;

                        if (0 != (openFlag & FileIOFlagType.IncEnvVar))    // 環境変数を含む?
                        {
                            fileName = Environment.ExpandEnvironmentVariables(fileName);
                        }

                        fileInfo = new FileInfo(fileName);
                    }
                    catch (ArgumentException)   // path が空の文字列 ("") か、空白しか含んでいないか、無効な文字を 1 つ以上含んでいます。
                    {
                        throw new ErrorResponseException(FileIOErrorType.FileNotFound);
                    }

                    FileShare fileShare = FileShare.None;
                    if (0 != (openFlag & FileIOFlagType.ShareRead))
                    {
                         fileShare = FileShare.Read;
                    }
                    if (0 != (openFlag & FileIOFlagType.ShareWrite))
                    {
                        fileShare = (fileShare == FileShare.Read) ? FileShare.ReadWrite : FileShare.Write;
                    }

                    if (0 == (openFlag & FileIOFlagType.Write))
                    {
                        fs = new FileStream(fileInfo.FullName, FileMode.Open, FileAccess.Read, fileShare);
                    }
                    else
                    {
                        FileMode fileMode = FileMode.Create;
                        FileAccess fileAccess = FileAccess.Write;

                        if (0 != (openFlag & FileIOFlagType.Read))  // ReadWrite
                        {
                            fileMode = 0 != (openFlag & FileIOFlagType.NoCreate) ?
                                FileMode.Open:
                                FileMode.OpenOrCreate;
                            fileAccess = FileAccess.ReadWrite;
                        }
                        else    // write only
                        {
                            // FileModeをTruncateにする方法も考えられますが、例外がIOExceptionでしか捕らえられないので、
                            // 事前に判断するようにしています。
                            if (0 != (openFlag & FileIOFlagType.NoOverwrite) && fileInfo.Exists)
                            {
                                throw new ErrorResponseException(FileIOErrorType.FileAlreadyExist);
                            }
                        }

                        if (0 != (openFlag & FileIOFlagType.CreateDir))
                        {
                            /*
                                環境変数の展開を指定しつつ、変換されない環境変数がある場合は
                                安全のためにディレクトリの作成を試みません。
                                ファイル名の一部として'%'文字を使用しているのか、環境変数を囲むために使用しているのかの
                                判断がつきにくいので、どのような場合でも%文字が残っていたら失敗とします。
                            */
                            if ( 0 != (openFlag & FileIOFlagType.IncEnvVar)     // 環境変数を含む?
                                && fileInfo.FullName.IndexOf('%') != -1         // 展開されていない環境変数がある
                            )
                            {
                                throw new ErrorResponseException(FileIOErrorType.DirectoryNotFound);
                            }

                            if (! fileInfo.Directory.Exists)
                            {
                                fileInfo.Directory.Create();
                            }
                        }

                        fs = new FileStream(fileInfo.FullName, fileMode, fileAccess, fileShare);
                    }
                }
                catch (System.Security.SecurityException)
                {
                    throw new ErrorResponseException(FileIOErrorType.SecurityError);
                }
                catch (UnauthorizedAccessException)
                {
                    throw new ErrorResponseException(FileIOErrorType.UnauthorizedAccess);
                }
                catch (FileNotFoundException)
                {
                    throw new ErrorResponseException(FileIOErrorType.FileNotFound);
                }
                catch (DirectoryNotFoundException)
                {
                    throw new ErrorResponseException(FileIOErrorType.DirectoryNotFound);
                }
                catch (IOException)
                {
                    throw new ErrorResponseException(FileIOErrorType.IOError);
                }
            }
            catch (ErrorResponseException e)
            {
                // エラーのレスポンスを送って抜ける
                resHead.ErrorCode = e.ErrorCode;
                resHead.Handle = IntPtr.Zero;
                resHead.FileSize = 0;

                resHead.ToStream(netSm);
                netSm.Flush();
                return;
            }

            resHead.Handle = (IntPtr)GCHandle.Alloc(fs);
            resHead.FileSize = fs.Length;

            try
            {
                resHead.ToStream(netSm);    // レスポンスを送る
                netSm.Flush();
            }
            catch
            {
                // 何らかの例外が発生した場合、ストリームを閉じておく
                ((GCHandle)resHead.Handle).Free();
                fs.Close();
                throw;  // 例外を再送出
            }

            // 開いたファイルのハンドルを管理しておく
            _fileStreamArray.Add(resHead.Handle);
            RouterLog.ServerReport(string.Format("file open \"{0}\"", fs.Name));
        }

        /// <summary>
        /// ファイルを閉じます。
        /// </summary>
        void DoFileClose(Stream netSm, uint dataSize)
        {
            FileIOResponseHeader resHead = new FileIOResponseHeader(FileIOCommandType.FILECLOSE);

            try
            {
                FileIOCommandHeader cmdHead = new FileIOCommandHeader(new NetBinaryReader(netSm), dataSize);

                resHead.ClnFileInfoPtr = cmdHead.ClnFileInfoPtr;

                // このオブジェクトが開いたファイルであるなら閉じる
                if (_fileStreamArray.Contains(cmdHead.Handle))
                {
                    CloseFileHandle(cmdHead.Handle);
                    _fileStreamArray.Remove(cmdHead.Handle);
                }
                else
                {
                    throw new ErrorResponseException(FileIOErrorType.InvalidFileHandle);
                }
            }
            catch (ErrorResponseException e)
            {
                resHead.ErrorCode = e.ErrorCode;
            }

            resHead.ToStream(netSm);    // レスポンスを送る
            netSm.Flush();
        }

        /// <summary>
        /// ファイルを読み込ます。
        /// </summary>
        void DoFileRead(Stream netSm, uint dataSize)
        {
            FileIOReadResponseHeader resHead = new FileIOReadResponseHeader();

            try
            {
                FileIOReadCommandHeader cmdHead = new FileIOReadCommandHeader (new NetBinaryReader(netSm), dataSize);

                resHead.ClnFileInfoPtr = cmdHead.ClnFileInfoPtr;
                resHead.ClnBufPtr = cmdHead.ClnBufPtr;

                FileStream fs = GetFileStream(cmdHead.Handle);
                if (fs == null)
                {
                    throw new ErrorResponseException(FileIOErrorType.InvalidFileHandle);
                }

                uint sendBytesMax = FILEIO_CHUNK_DATASIZE_MAX - resHead.HeaderSize;
                byte[] transBuf = new byte[sendBytesMax];
                resHead.TotalSentSize = 0;

                // ターゲットへの書き込み
                do
                {
                    uint restBytes = cmdHead.ReadSize - resHead.TotalSentSize;

                    resHead.DataSize = Math.Min(restBytes, sendBytesMax);

                    try
                    {
                        resHead.DataSize = (uint)fs.Read(transBuf, 0, (int)resHead.DataSize);
                    }
                    catch (IOException)
                    {
                        throw new ErrorResponseException(FileIOErrorType.IOError);
                    }


                    Debug.WriteLine(string.Format("FileIOServer : send bytes {0:d}", resHead.DataSize));
#if HORYUU // #if ! defined(NDEBUG)
                    NNS_MCS_TRACE1("file data check sum %08X\n", CalcCheckSum(&readBuf[0], fileReadBytes));
#endif

                    // 今回の送信で要求したバイト数を転送するか、あるいはEOFに達したか
                    resHead.TotalSentSize += resHead.DataSize;
                    resHead.ErrorCode = resHead.TotalSentSize == cmdHead.ReadSize || fs.Position == fs.Length ?
                        FileIOErrorType.Success:    // 最後
                        FileIOErrorType.Continue;   // 途中結果を送信

                    // レスポンスを送る
                    resHead.ToStream(netSm);
                    netSm.Write(transBuf, 0, (int)resHead.DataSize);
                    netSm.Flush();

                    resHead.ClnBufPtr += resHead.DataSize;  // ターゲット側のバッファ位置の更新
                }while (resHead.ErrorCode == FileIOErrorType.Continue);
            }
            catch (ErrorResponseException e)
            {
                // レスポンスの作成
                resHead.ErrorCode = e.ErrorCode;
                resHead.DataSize = 0;
                resHead.TotalSentSize = 0;

                resHead.ToStream(netSm);    // レスポンスを送る
                netSm.Flush();
            }
        }
    　
        /// <summary>
        /// ファイルに書き込みます。
        /// </summary>
        void DoFileWrite(Stream netSm, uint chunkSize)
        {
            FileIOResponseHeader resHead = new FileIOResponseHeader(FileIOCommandType.FILEWRITE);

            try
            {
                FileIOWriteCommandHeader cmdHead = new FileIOWriteCommandHeader(new NetBinaryReader(netSm), chunkSize);

                // データの読み込み
                //     実機側コードの FileIO_Write() では、コマンドヘッダの書き込みのあと、
                //     すぐにデータを書き込んでおり、この2つで1つのコマンドとみなす。
                byte[] buf = new byte[cmdHead.WriteSize];
                for (int readBytes = 0; readBytes < buf.Length; )
                {
                    readBytes += netSm.Read(buf, readBytes, buf.Length - readBytes);
                }

                resHead.ClnFileInfoPtr = cmdHead.ClnFileInfoPtr;

                FileStream fs = GetFileStream(cmdHead.Handle);
                if (fs == null)
                {
                    throw new ErrorResponseException(FileIOErrorType.InvalidFileHandle);
                }

                try
                {
                    fs.Write(buf, 0, buf.Length);
                }
                catch (IOException)
                {
                    throw new ErrorResponseException(FileIOErrorType.IOError);
                }
            }
            catch (ErrorResponseException e)
            {
                resHead.ErrorCode = e.ErrorCode;
            }

            // レスポンスを送る
            resHead.ToStream(netSm);
            netSm.Flush();
        }

        /// <summary>
        /// ファイルをシークします。
        /// </summary>
        void DoFileSeek(Stream netSm, uint chunkSize)
        {
            FileIOSeekResponseHeader resHead = new FileIOSeekResponseHeader();

            try
            {
                FileIOSeekCommandHeader cmdHead = new FileIOSeekCommandHeader(new NetBinaryReader(netSm), chunkSize);

                resHead.ClnFileInfoPtr = cmdHead.ClnFileInfoPtr;

                FileStream fs = GetFileStream(cmdHead.Handle);
                if (fs == null)
                {
                    throw new ErrorResponseException(FileIOErrorType.InvalidFileHandle);
                }

                try
                {
                    SeekOrigin seekOrigin;

                    switch (cmdHead.SeekOrigin)
                    {
                    case FileIOSeekOrigin.Begin:
                    default:                        seekOrigin = SeekOrigin.Begin;      break;
                    case FileIOSeekOrigin.Current:  seekOrigin = SeekOrigin.Current;    break;
                    case FileIOSeekOrigin.End:      seekOrigin = SeekOrigin.End;        break;
                    }

                    resHead.FilePointer = (uint)fs.Seek(cmdHead.DistanceToMove, seekOrigin);
                }
                catch (IOException)
                {
                    throw new ErrorResponseException(FileIOErrorType.IOError);
                }
            }
            catch (ErrorResponseException e)
            {
                resHead.ErrorCode = e.ErrorCode;
            }

            // レスポンスを送る
            resHead.ToStream(netSm);
            netSm.Flush();
        }

        /// <summary>
        /// ファイル、フォルダ検索を開始します。
        /// </summary>
        void DoFindFirst(Stream netSm, uint chunkSize)
        {
            FileIOFindResponseHeader resHead = new FileIOFindResponseHeader(FileIOCommandType.FINDFIRST);
            FileSystemInfo[] fsi = null;
            IEnumerator en = null;

            try
            {
                FileIOFindFirstCommandHeader cmdHead = new FileIOFindFirstCommandHeader(new NetBinaryReader(netSm), chunkSize);

                resHead.ClnFileInfoPtr = cmdHead.ClnFileInfoPtr;
                resHead.ClnFindDataPtr = cmdHead.ClnFindDataPtr;

                string fileName = Environment.ExpandEnvironmentVariables(cmdHead.FileName);

                try
                {
                    DirectoryInfo dirInfo = null;
                    try
                    {
                        dirInfo = new DirectoryInfo(Path.GetDirectoryName(fileName));
                    }
                    catch (ArgumentException)   // パスに無効な文字列が含まれているなど。
                    {
                        throw new ErrorResponseException(FileIOErrorType.NoMoreFiles);
                    }

                    if (! dirInfo.Exists)   // ディレクトリが存在しない
                    {
                        throw new ErrorResponseException(FileIOErrorType.DirectoryNotFound);
                    }

                    fsi = dirInfo.GetFileSystemInfos(Path.GetFileName(fileName));

                    if (fsi.Length == 0) // パターンに一致するものが1つもない場合
                    {
                        throw new ErrorResponseException(FileIOErrorType.NoMoreFiles);
                    }

                    en = fsi.GetEnumerator();
                    if (! en.MoveNext())
                    {
                        throw new ErrorResponseException(FileIOErrorType.NoMoreFiles);
                    }
                    RouterLog.ServerReport(string.Format("directory open \"{0}\", patten \"{1}\" ", dirInfo.FullName, Path.GetFileName(fileName)));
                }
                catch (System.Security.SecurityException)
                {
                    throw new ErrorResponseException(FileIOErrorType.SecurityError);
                }
                catch (DirectoryNotFoundException)
                {
                    throw new ErrorResponseException(FileIOErrorType.DirectoryNotFound);
                }
            }
            catch (ErrorResponseException e)
            {
                // エラーのレスポンスを送って抜ける
                resHead.ErrorCode = e.ErrorCode;
                resHead.Handle = IntPtr.Zero;

                resHead.ToStream(netSm);
                netSm.Flush();
                return;
            }

            resHead.ErrorCode = FileIOErrorType.Success;
            resHead.Handle = (IntPtr)GCHandle.Alloc(en);
            resHead.SetFileSystemInfo(fsi[0]);

            try
            {
                resHead.ToStream(netSm);    // レスポンスを送る
                netSm.Flush();
            }
            catch
            {
                // 何らかの例外が発生した場合、固定したハンドルを解放しておく
                ((GCHandle)resHead.Handle).Free();
                throw;  // 例外を再送出
            }

            // 開いたファイルのハンドルを管理しておく
            _filesEnumArray.Add(resHead.Handle);
        }

        /// <summary>
        /// 次のファイル、フォルダを検索します。
        /// </summary>
        void DoFindNext( Stream netSm, uint chunkSize )
        {
            FileIOFindResponseHeader resHead = new FileIOFindResponseHeader(FileIOCommandType.FINDNEXT);
            IEnumerator en = null;

            try
            {
                FileIOFindNextCommandHeader cmdHead = new FileIOFindNextCommandHeader(new NetBinaryReader(netSm), chunkSize);

                resHead.ClnFileInfoPtr = cmdHead.ClnFileInfoPtr;
                resHead.Handle = cmdHead.Handle;
                resHead.ClnFindDataPtr = cmdHead.ClnFindDataPtr;

                if (!_filesEnumArray.Contains(cmdHead.Handle))
                {
                    throw new ErrorResponseException(FileIOErrorType.InvalidFileHandle);
                }

                en = (IEnumerator)((GCHandle)cmdHead.Handle).Target;
                if (! en.MoveNext())
                {
                    throw new ErrorResponseException(FileIOErrorType.NoMoreFiles);
                }
            }
            catch (ErrorResponseException e)
            {
                // エラーのレスポンスを送って抜ける
                resHead.ErrorCode = e.ErrorCode;

                resHead.ToStream(netSm);
                netSm.Flush();
                return;
            }

            FileSystemInfo fileSystemInfo = (FileSystemInfo)en.Current;
            resHead.SetFileSystemInfo(fileSystemInfo);

            resHead.ErrorCode = FileIOErrorType.Success;
            resHead.ToStream(netSm);    // レスポンスを送る
            netSm.Flush();
        }

        /// <summary>
        /// ファイル、フォルダ検索を終了します。
        /// </summary>
        void
        DoFindClose(Stream netSm, uint chunkSize)
        {
            FileIOResponseHeader resHead = new FileIOResponseHeader(FileIOCommandType.FINDCLOSE);

            try
            {
                FileIOCommandHeader cmdHead = new FileIOCommandHeader(new NetBinaryReader(netSm), chunkSize);

                resHead.ClnFileInfoPtr = cmdHead.ClnFileInfoPtr;

                if (!_filesEnumArray.Contains(cmdHead.Handle))
                {
                    throw new ErrorResponseException(FileIOErrorType.InvalidFileHandle);
                }

                CloseEnumHandle(cmdHead.Handle);
                _filesEnumArray.Remove(cmdHead.Handle);
            }
            catch (ErrorResponseException e)
            {
                resHead.ErrorCode = e.ErrorCode;
            }

            // レスポンスを送る
            resHead.ToStream(netSm);
            netSm.Flush();
        }

        /// <summary>
        /// ダイアログを表示するデリゲート用オブジェクト
        /// </summary>
        internal class DoShowDialog
        {
            readonly FileIOShowFileDlgCommandHeader _cmdHead;
            string _fileName = null;

            internal DoShowDialog(FileIOShowFileDlgCommandHeader cmdHead)
            {
                _cmdHead = cmdHead;
            }

            internal void DoShow(object sender, EventArgs e)
            {
                Form form = (Form)sender;
                if (form.IsDisposed)
                {
                    return;
                }

                FileDialog dlg = null;
                if (0 != (_cmdHead.Flag & FileIOFileDialogFlagType.Save))
                {
                    SaveFileDialog saveDlg = new SaveFileDialog();
                    dlg = saveDlg;
                    saveDlg.OverwritePrompt = 0 == (_cmdHead.Flag & FileIOFileDialogFlagType.NoOverwritePrompt);
                    saveDlg.CreatePrompt = 0 != (_cmdHead.Flag & FileIOFileDialogFlagType.CreatePrompt);
                }
                else
                {
                    dlg = new OpenFileDialog();
                }

                if (_cmdHead.InitialPath != string.Empty)
                {
                    string initPath = Environment.ExpandEnvironmentVariables(_cmdHead.InitialPath);
                    if (0 != (_cmdHead.Flag & FileIOFileDialogFlagType.IncludeFileName))
                    {
                        dlg.FileName = initPath;
                    }
                    else
                    {
                        dlg.InitialDirectory = initPath;
                    }
                }

                try
                {
                    dlg.Filter = _cmdHead.Filter;
                }
                catch (ArgumentException ex)
                {
                    RouterLog.ServerReport(ServerReportType.Error, string.Format("Filter error. - [{0}]\n{1}", _cmdHead.Filter, ex.Message));
                }
                dlg.DefaultExt = _cmdHead.DefaultExt;
                dlg.Title = _cmdHead.Title;

                // タスクトレイに入っていてタスクバーに表示されていないときは、
                // ファイルダイアログを表示する間だけタスクバーに復活させる
                bool oldShowInTaskbar = form.ShowInTaskbar;
                if (! form.ShowInTaskbar)
                {
                    form.ShowInTaskbar = true;
                }
                form.Activate();
                form.TopMost = true;
                if (DialogResult.OK == dlg.ShowDialog(form))
                {
                    _fileName = dlg.FileName;
                }

                form.TopMost = false;

                if (!oldShowInTaskbar)
                {
                    form.ShowInTaskbar = false;
                }
            }

            internal string FileName
            {
                get { return _fileName; }
            }
        }

        /// <summary>
        /// 環境変数を展開します。
        /// </summary>
        void DoExpandPath(Stream netSm, uint dataSize)
        {
            FileIOExpandPathResponseHeader resHead = new FileIOExpandPathResponseHeader();

            string fileName;

            try
            {
                FileIOExpandPathCommandHeader cmdHead = new FileIOExpandPathCommandHeader(new NetBinaryReader(netSm), dataSize);

                try
                {
                    fileName = cmdHead.FileName;
                    fileName = Environment.ExpandEnvironmentVariables(fileName);
                }
                catch (IOException)
                {
                    throw new ErrorResponseException(FileIOErrorType.FileNotFound);
                }
            }
            catch (ErrorResponseException e)
            {
                // エラーのレスポンスを送って抜ける
                resHead.ErrorCode = e.ErrorCode;
                resHead.DataSize = 0;

                resHead.ToStream(netSm);
                netSm.Flush();
                return;
            }

            if (fileName == null)
            {
                resHead.ErrorCode = FileIOErrorType.FileNotFound;
                resHead.DataSize = 0;
                resHead.ToStream(netSm);
                netSm.Flush();
                return;
            }

            // ファイル名文字列をバイト配列にして書き込む
            int fileNameLength = Math.Min(fileName.Length, FileIOServer.FILEIO_PATH_MAX - 1);
            Encoding enc;
            int charBytes = 0;

            if (false)
            {
                // UTF16文字列への変換
                // enc = Encoding.BigEndianUnicode;
                // charBytes = 2;
            }
            else
            {
                // ANSI文字列への変換
                enc = Encoding.Default;
                charBytes = 1;
            }

            byte[] writeBuf = new byte[enc.GetMaxByteCount(fileNameLength) + charBytes];   // Cの終端文字列分余分に確保しておく。デフォルトで0がセットされている。
            int writeSize = enc.GetBytes(fileName, 0, fileNameLength, writeBuf, 0);
            writeSize += charBytes;

            Debug.WriteLine(string.Format("expand path response data size {0}", writeSize));
            resHead.DataSize = (uint)writeSize;
            resHead.ToStream(netSm);

            netSm.Write(writeBuf, 0, writeSize);
            netSm.Flush();
        }

        /// <summary>
        /// FileDialogを表示します。
        /// </summary>
        void DoShowFileDialog( Stream netSm, uint dataSize )
        {
            FileIOShowFileDlgResponseHeader resHead = new FileIOShowFileDlgResponseHeader();

            string fileName;

            try
            {
                FileIOShowFileDlgCommandHeader cmdHead = new FileIOShowFileDlgCommandHeader(new NetBinaryReader(netSm), dataSize);

                try
                {
                    // formのスレッドでダイアログを表示させる
                    DoShowDialog doShowDialog = new DoShowDialog(cmdHead);
                    _form.Invoke(new EventHandler(doShowDialog.DoShow));
                    fileName = doShowDialog.FileName;
                }
                catch (IOException)
                {
                    throw new ErrorResponseException(FileIOErrorType.IOError);
                }
            }
            catch (ErrorResponseException e)
            {
                // エラーのレスポンスを送って抜ける
                resHead.ErrorCode = e.ErrorCode;
                resHead.DataSize = 0;

                resHead.ToStream(netSm);
                netSm.Flush();
                return;
            }

            if (fileName == null)
            {
                resHead.ErrorCode = FileIOErrorType.Canceled;
                resHead.DataSize = 0;
                resHead.ToStream(netSm);
                netSm.Flush();
                return;
            }

            // ファイル名文字列をバイト配列にして書き込む
            int fileNameLength = Math.Min(fileName.Length, FileIOServer.FILEIO_PATH_MAX - 1);
            Encoding enc;
            int charBytes = 0;

            if (false)
            {
                // UTF16文字列への変換
                // enc = Encoding.BigEndianUnicode;
                // charBytes = 2;
            }
            else
            {
                // ANSI文字列への変換
                enc = Encoding.Default;
                charBytes = 1;
            }

            byte[] writeBuf = new byte[enc.GetMaxByteCount(fileNameLength) + charBytes];   // Cの終端文字列分余分に確保しておく。デフォルトで0がセットされている。
            int writeSize = enc.GetBytes(fileName, 0, fileNameLength, writeBuf, 0);
            writeSize += charBytes;

            Debug.WriteLine(string.Format("show dialog response data size {0}", writeSize));
            resHead.DataSize = (uint)writeSize;
            resHead.ToStream(netSm);

            netSm.Write(writeBuf, 0, writeSize);
            netSm.Flush();
        }

        /// <summary>
        ///
        /// </summary>
        public static string ReadFileNameByteArray( BinaryReader br )
        {
            byte[] fileNameBuf = br.ReadBytes(FILEIO_PATH_BYTES);
            if (fileNameBuf.Length != FILEIO_PATH_BYTES)
            {
                throw new EndOfStreamException();   // 途中でEOFに達した
            }

            // C文字列の終端コード'\0'の位置を探す
            int fileNameLen;
            for (fileNameLen = 0; fileNameLen < fileNameBuf.Length; ++fileNameLen)
            {
                if (fileNameBuf[fileNameLen] == 0)
                {
                    break;
                }
            }
            return Encoding.Default.GetString(fileNameBuf, 0, fileNameLen);
        }

        /// <summary>
        ///
        /// </summary>
        public static string ReadStringFromByteArray( BinaryReader br, int size )
        {
            byte[] strBuf = br.ReadBytes(size);
            if (strBuf.Length != size)
            {
                throw new EndOfStreamException();   // 途中でEOFに達した
            }

            // C文字列の終端コード'\0'の位置を探す
            int strLen;
            for (strLen = 0; strLen < strBuf.Length; ++strLen)
            {
                if (strBuf[strLen] == 0)
                {
                    break;
                }
            }
            return Encoding.Default.GetString(strBuf, 0, strLen);
        }

        /// <summary>
        ///
        /// </summary>
        FileStream GetFileStream( IntPtr gcHandleVal )
        {
            if (_fileStreamArray.Contains(gcHandleVal))
            {
                GCHandle gcHandle = (GCHandle)gcHandleVal;
                return (FileStream)gcHandle.Target;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        void CloseFileHandle( IntPtr handle )
        {
            GCHandle gcHandle = (GCHandle)handle;
            FileStream fs = (FileStream)gcHandle.Target;
            try
            {
                fs.Close();
                RouterLog.ServerReport(string.Format("file close \"{0}\"", fs.Name));
            }
            catch (IOException) // ストリームを閉じようとしている間にエラーが発生
            {
                RouterLog.ServerReport("File IO Server : IOException - FileStream.Close()");
            }
            gcHandle.Free();
        }

        /// <summary>
        ///
        /// </summary>
        void CloseEnumHandle( IntPtr handle )
        {
            ((GCHandle)handle).Free();
            RouterLog.ServerReport("directry close.");
        }
    }
}
