﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Threading;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Diagnostics;

namespace Nintendo.McsServer
{
    /// <summary>
    /// ChannelManager
    /// McsServerが管理する複数のチャンネルを管理するクラス。
    /// 主に、チャンネルの登録、削除を処理する。
    /// </summary>
    public class ChannelManager
    {
        /// <summary>
        /// ChannelInfoリスト
        /// </summary>
        readonly List<ChannelInfo> _channelList = new List<ChannelInfo>();

        /// <summary>
        /// チャンネル値をキーとするChannelInfo辞書
        /// </summary>
        readonly Dictionary<ushort, ChannelInfo> _channelDic = new Dictionary<ushort, ChannelInfo>();

        /// <summary>
        /// 接続状況を確認するための特殊チャンネルのチャンネル番号。
        /// </summary>
        const int WATCHDOG_CHANNEL = 0xFF7D;

        /// <summary>
        /// FileIOServer用チャンネルが登録されているかどうか
        /// </summary>
        bool _registFileIOServer = false;

        /// <summary>
        /// デバイスと接続していたら真。
        /// </summary>
        bool _bTargetConnect = false;

        /// <summary>
        /// 同期用オブジェクト。
        /// </summary>
        readonly object _syncObj = new object();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ChannelManager()
        {}

        /// <summary>
        /// FileIOServerが登録されているかどうか取得します。
        /// </summary>
        public bool IsRegistFileIOServer
        {
            get
            {
                lock (_syncObj)
                {
                    return _registFileIOServer;
                }
            }
        }

        /// <summary>
        /// ChannelInfoの追加
        /// </summary>
        public void Add(ChannelInfo channelInfo)
        {
            lock (_syncObj)
            {
                _channelList.Add(channelInfo);
            }
        }

        /// <summary>
        /// チャンネルの登録
        /// </summary>
        public ChannelRequestResult RegistChannel( ChannelInfo channelInfo, ushort channel, uint flag )
        {
            // Debug.WriteLine(string.Format("channel regist request {0:X4}, {1:X8}", channel, flag));

            if (0xFF80 <= channel && channel <= 0xFFFF)  // システム予約チャンネル
            {
                return ChannelRequestResult.SystemReserved;
            }

            lock (_syncObj)
            {
                if (flag == 0)
                {
                    if (! _bTargetConnect)
                    {
                        return ChannelRequestResult.NotConnect;
                    }
                }

                if (channel == WATCHDOG_CHANNEL)
                {
                    return ChannelRequestResult.Success;
                }

                if (_channelDic.ContainsKey(channel))   // 既に同じチャンネル値を持つChannelInfoが登録されている
                {
                    return ChannelRequestResult.AlreadyRegisted;
                }

                _channelDic.Add(channel, channelInfo);
                if (channel == FileIOServer.FILEIO_CHANNEL)
                {
                    _registFileIOServer = true;
                }
            }

            RouterLog.ServerReport("Channel {0:X4}({0:d}) : registered.", channel);
            return ChannelRequestResult.Success;
        }

        /// <summary>
        /// チャンネルの削除
        /// </summary>
        public void Remove( ChannelInfo channelInfo )
        {
            lock (_syncObj)
            {
                // オブジェクトが存在しない場合はそのまま抜ける。
                // このメソッドが呼ばれるときに、同じチャンネル値を持つ新しいChannelInfoインスタンスが登録されている可能性があるため、
                // キーではなくオブジェクトで判断すること。

                if (! _channelList.Remove(channelInfo))
                {
                    return;
                }

                if (ChannelInfo.InvalidChannel != channelInfo.ChannelNo && _channelDic.ContainsValue(channelInfo))
                {
                    _channelDic.Remove(channelInfo.ChannelNo);
                    if (channelInfo.ChannelNo == FileIOServer.FILEIO_CHANNEL)
                    {
                        _registFileIOServer = false;
                    }
                }
            }

            RouterLog.ServerReport("Channel {0:X4}({0:d}) : unregistered.", channelInfo.ChannelNo);
        }

        /// <summary>
        /// チャンネルの取得
        /// </summary>
        public ChannelInfo GetChannelInfo( ushort channel )
        {
            ChannelInfo channelInfo = null;
            lock (_syncObj)
            {
                _channelDic.TryGetValue(channel, out channelInfo);
            }
            return channelInfo;
        }

        /// <summary>
        /// 通信状態を変更する。
        /// </summary>
        public void SetTargetConnect( bool value )
        {
            lock (_syncObj)
            {
                if (_bTargetConnect != value)    // 値が変化するとき
                {
                    if (! value)    // 切断された場合
                    {
                        DisconnectAll();
                    }

                    _bTargetConnect = value;
                }
            }
        }

        /// <summary>
        /// すべてのチャンネルの通信を切断する。
        /// </summary>
        void DisconnectAll()
        {
            lock (_syncObj)
            {
                foreach (ChannelInfo channelInfo in _channelList)
                {
                    channelInfo.Shutdown();
                }

                _channelList.Clear();
                _channelDic.Clear();
                _registFileIOServer = false;
            }

            RouterLog.ServerReport("all channel unregistered.");
        }
    }
}
