﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <windows.h>
#include <revolution/hio2.h>
#include <nw4r/mcs/common.h>

namespace nw4r
{

namespace mcs
{

class Hio2RingBuffer;

}   // namespace mcs

}   // namespace nw4r

namespace Nintendo
{

namespace McsServer
{

//--------------------------------------------------
// HIO2を利用する通信デバイス
public ref class HIO2CommDevice : public ICommDevice
{
public:
    #if _DEBUG
            literal System::String^ DllName = "hio2D.dll";
    #else
            literal System::String^ DllName = "hio2.dll";
    #endif

// staticメンバ
public:
    static bool                 SearchDevice(
                                    [System::Runtime::InteropServices::Out] System::String^ % devicePathName);

private:
    static HMODULE              LoadLibrarySub(System::String^ dllPath);

    static bool                 IsFindHIO2Module();

    static void                 InitHIO2();

    static HIO2Handle           OpenDevice(System::String^ devPathName);

    static void                 CloseDevice(HIO2Handle devHandle);

    static u32                  ReadMailbox(HIO2Handle devHandle);

    static void                 WriteMailbox(HIO2Handle devHandle, u32 mailWord);

    static u32                  ReadStatus(HIO2Handle devHandle);

    static System::IntPtr       RoundUp(
                                    System::IntPtr  value,
                                    u32             alignment);

    static HMODULE              shHIO2Module_               = 0;    // HIO2.dllモジュールハンドル。ロードに成功した場合、明示的に解放していません。
    static HIO2InitType         spfHIO2Init_;
    static HIO2EnumDevicesType  spfHIO2EnumDevices_;
    static HIO2OpenType         spfHIO2Open_;
    static HIO2CloseType        spfHIO2Close_;
    static HIO2ReadMailboxType  spfHIO2ReadMailbox_;
    static HIO2WriteMailboxType spfHIO2WriteMailbox_;
    static HIO2ReadType         spfHIO2Read_;
    static HIO2WriteType        spfHIO2Write_;
    static HIO2ReadAsyncType    spfHIO2ReadAsync_;
    static HIO2WriteAsyncType   spfHIO2WriteAsync_;
    static HIO2ReadStatusType   spfHIO2ReadStatus_;
    static HIO2ExitType         spfHIO2Exit_;
    static HIO2GetLastErrorType spfHIO2GetLastError_;

// インスタンスメンバ
public:
                                HIO2CommDevice(System::String^ devPathName);

                                ~HIO2CommDevice();

                                !HIO2CommDevice();

    virtual void                Negotiate();

    property System::String^    Name
    {
        virtual System::String^ get();
    }

    property bool               IsAttach
    {
        virtual bool get();
    }

    property bool               IsTargetConnect
    {
        virtual bool get();
    }

    virtual bool                Read();

    virtual System::IntPtr      GetMessage(
                                    [System::Runtime::InteropServices::Out] int%    channel,
                                    [System::Runtime::InteropServices::Out] int%    size);

    virtual int                 GetWritableBytes(bool withUpdate);

    virtual void                Write(
                                    int                     channel,
                                    array<System::Byte>^    buf,
                                    int                     offset,
                                    int                     size);

    virtual System::UInt32      GetWriteBufferSize();

    property bool               IsPort
    {
        bool get()  { return pReader_->IsPort() && pWriter_->IsPort(); }
    }

private:

    //-----------------------------------------------------
    // 送受信用内部バッファのサイズ
    // HIOモジュールの内部バッファのサイズと相互関係する数値です。
    //
    // 現在は、
    // 送信(受信)バッファサイズ = (HIOモジュールの内部バッファのサイズ) / 2
    // となっています。
    //
    static const u32            READ_TRANSBUF_SIZE  = nw4r::mcs::detail::MCS_INTERNAL_BUFFER_SIZE;
    static const u32            WRITE_TRANSBUF_SIZE = nw4r::mcs::detail::MCS_INTERNAL_BUFFER_SIZE;


    void                        DeviceFinalize();

    bool                        bTargetConnect_;

    // アンマネージリソースの関連
    bool                        bInitHIO2_;
    HIO2Handle                  devHandle_;
    nw4r::mcs::detail::Hio2RingBuffer*  pReader_;
    nw4r::mcs::detail::Hio2RingBuffer*  pWriter_;
    System::IntPtr              hTempBuf_;

private:
    /// <summary>
    /// HIO2関数でエラーが起こった場合の例外クラス
    /// この例外オブジェクトは外にthrowしないこと。
    /// </summary>
    ref class HIO2FunctionException : public System::Exception
    {
    public:
        HIO2FunctionException();
        HIO2FunctionException(System::String^ description);

        property int                ErrorCode
        {
            int get()   { return errorCode_; }
        }

    private:
        int                         errorCode_;
    };
};

}   // namespace McsServer

}   // namespace Nintendo
