﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "StdAfx.h"
#include <vcclr.h>
#include <nw4r/mcs/hioNegotiate.h>
#include <nw4r/mcs/hio2RingBuffer.h>
#include "HIO2CommDevice.h"
#include "HIO2CommDeviceException.h"

using namespace System;
using namespace System::Runtime::InteropServices;
using namespace System::Diagnostics;
using namespace System::Collections::Generic;
using namespace System::IO;

namespace
{
// typedef LPVOID	HIO2Handle;
// typedef	LPCSTR	HIO2DevicePath;

// typedef BOOL	(*HIO2EnumCallback)( HIO2DevicePath path, void* param );
[UnmanagedFunctionPointer(CallingConvention::Cdecl)]
delegate BOOL DelegateHIO2EnumCallback(HIO2DevicePath path, void* param);

// typedef void	(*HIO2ReceiveCallback)( HIO2Handle h );
[UnmanagedFunctionPointer(CallingConvention::Cdecl)]
delegate void DelegateHIO2ReceiveCallback(System::IntPtr h);

// typedef int		(*HIO2NotifyCallback)( HIO2NotifyEvent event, void* param );
[UnmanagedFunctionPointer(CallingConvention::Cdecl)]
delegate int DelegateHIO2NotifyCallback(HIO2NotifyEvent aEvent, System::IntPtr param);

ref class DeviceEnumerator
{
public:
    DeviceEnumerator(List<String^>^ findChannelList)
        : findChannelList_(findChannelList)
    {}

    BOOL                EnumCallback(HIO2DevicePath path, void* param);

private:
    List<String^>^      findChannelList_;

};

/// <summary>
/// HIO2EnumDevicesに渡すコールバック関数
/// </summary>
/// <param name="chan"></param>
/// <returns></returns>
BOOL
DeviceEnumerator::EnumCallback(HIO2DevicePath path, void* /* param */)
{
    String^ devPathName = gcnew String(path);
    findChannelList_->Add(devPathName);
    Debug::WriteLine(String::Format("find device path - {0}", devPathName));
    return true;    // USB I/Fの列挙を継続
}

}   // namespace

namespace Nintendo
{

namespace McsServer
{

//----------------------------------------------------------------------------------------------------
bool
HIO2CommDevice::SearchDevice([Out] String^ % devicePathName)
{
    devicePathName = nullptr;
    if (! IsFindHIO2Module())
    {
        throw gcnew HIO2DllNotFoundException();
    }

    List<String^>^ findChannelList = gcnew List<String^>();

    if (! spfHIO2Init_())
    {
        throw gcnew HIO2CommDeviceException("HIO2Init() failed.");
    }

    int result = 0;
    try
    {
            DelegateHIO2EnumCallback^ enumDeviceCallback = gcnew DelegateHIO2EnumCallback(gcnew DeviceEnumerator(findChannelList), &DeviceEnumerator::EnumCallback);
            result = spfHIO2EnumDevices_(HIO2EnumCallback(Marshal::GetFunctionPointerForDelegate(enumDeviceCallback).ToPointer()), 0);
            GC::KeepAlive(enumDeviceCallback);
            if (result == -1)
            {
                throw gcnew HIO2CommDeviceException("HIO2EnumDevices() failed.");
            }
    }
    finally
    {
        spfHIO2Exit_();
    }

    Debug::WriteLine(String::Format("HIO2EnumDevices() result - {0}", result));

    if (findChannelList->Count == 0)    // デバイスが1つも見つからなかった場合
    {
        return false;
    }

    devicePathName = findChannelList[0];
    return true;
}

//----------------------------------------------------------------------------------------------------
HMODULE
HIO2CommDevice::LoadLibrarySub(String^ dllPath)
{
    pin_ptr<const wchar_t> wchDllPath = PtrToStringChars(dllPath);
    HMODULE hModule = LoadLibrary(wchDllPath);
    if (! hModule)
    {
        Debug::WriteLine(String::Format("LoadLibrary({0}) - {1}", dllPath, GetLastError()));
    }
    return hModule;
}

//----------------------------------------------------------------------------------------------------
// DLLをロードして、必要とする関数すべてが利用可能か
// 確認しておく。
//
// DLLを明示的にロードせずに、
// 関数が実際に呼び出された時点で、動的リンクを行うようにすると、
// エラー報告のメッセージがわかりづらく、不具合の原因が特定しづらくなると
// 考えてこのような処理を行っています。
//
bool
HIO2CommDevice::IsFindHIO2Module()
{
    if (shHIO2Module_)
    {
        return true;
    }

    // パス指定なしでアクセス可能か?
    shHIO2Module_ = LoadLibrarySub(DllName);
    if (! shHIO2Module_)
    {
        // Revolution SDKの環境変数が設定されていてそこから辿れるか?
        String^ dllPath = Path::Combine(Path::Combine(Environment::ExpandEnvironmentVariables("%REVOLUTION_SDK_ROOT%"), "X86\\bin"), DllName);
        shHIO2Module_ = LoadLibrarySub(dllPath);
    }

    if (! shHIO2Module_)
    {
        return false;
    }

    if ( 0 == (spfHIO2Init_         = reinterpret_cast<HIO2InitType        >(GetProcAddress(shHIO2Module_, HIO2InitStr        )))
      || 0 == (spfHIO2EnumDevices_  = reinterpret_cast<HIO2EnumDevicesType >(GetProcAddress(shHIO2Module_, HIO2EnumDevicesStr )))
      || 0 == (spfHIO2Open_         = reinterpret_cast<HIO2OpenType        >(GetProcAddress(shHIO2Module_, HIO2OpenStr        )))
      || 0 == (spfHIO2Close_        = reinterpret_cast<HIO2CloseType       >(GetProcAddress(shHIO2Module_, HIO2CloseStr       )))
      || 0 == (spfHIO2ReadMailbox_  = reinterpret_cast<HIO2ReadMailboxType >(GetProcAddress(shHIO2Module_, HIO2ReadMailboxStr )))
      || 0 == (spfHIO2WriteMailbox_ = reinterpret_cast<HIO2WriteMailboxType>(GetProcAddress(shHIO2Module_, HIO2WriteMailboxStr)))
      || 0 == (spfHIO2Read_         = reinterpret_cast<HIO2ReadType        >(GetProcAddress(shHIO2Module_, HIO2ReadStr        )))
      || 0 == (spfHIO2Write_        = reinterpret_cast<HIO2WriteType       >(GetProcAddress(shHIO2Module_, HIO2WriteStr       )))
      || 0 == (spfHIO2ReadAsync_    = reinterpret_cast<HIO2ReadAsyncType   >(GetProcAddress(shHIO2Module_, HIO2ReadAsyncStr   )))
      || 0 == (spfHIO2WriteAsync_   = reinterpret_cast<HIO2WriteAsyncType  >(GetProcAddress(shHIO2Module_, HIO2WriteAsyncStr  )))
      || 0 == (spfHIO2ReadStatus_   = reinterpret_cast<HIO2ReadStatusType  >(GetProcAddress(shHIO2Module_, HIO2ReadStatusStr  )))
      || 0 == (spfHIO2Exit_         = reinterpret_cast<HIO2ExitType        >(GetProcAddress(shHIO2Module_, HIO2ExitStr        )))
      || 0 == (spfHIO2GetLastError_ = reinterpret_cast<HIO2GetLastErrorType>(GetProcAddress(shHIO2Module_, HIO2GetLastErrorStr)))
    )
    {
        FreeLibrary(shHIO2Module_);
        shHIO2Module_ = 0;
        return false;
    }

    nw4r::mcs::detail::Hio2RingBuffer::SetHIO2FuncAddress(spfHIO2Read_, spfHIO2Write_, spfHIO2GetLastError_);
    return true;
}

//----------------------------------------------------------------------------------------------------
void
HIO2CommDevice::InitHIO2()
{
    if (! spfHIO2Init_())
    {
        throw gcnew HIO2FunctionException();
    }
}

//----------------------------------------------------------------------------------------------------
HIO2Handle
HIO2CommDevice::OpenDevice(String^ devPathName)
{
    array<Byte>^ devPathNameBytes = System::Text::Encoding::Default->GetBytes(devPathName);
    pin_ptr<Byte> pDevpPathName = &devPathNameBytes[0];
    HIO2Handle devHandle = spfHIO2Open_(reinterpret_cast<HIO2DevicePath>(pDevpPathName), 0, 0, 0);
    if (devHandle == HIO2_INVALID_HANDLE_VALUE)
    {
        throw gcnew HIO2FunctionException("HIO2Open()");
    }
    return devHandle;
}

//----------------------------------------------------------------------------------------------------
void
HIO2CommDevice::CloseDevice(HIO2Handle devHandle)
{
    Debug::Assert(devHandle != HIO2_INVALID_HANDLE_VALUE);

    if (! spfHIO2Close_(devHandle))
    {
        throw gcnew HIO2FunctionException("HIO2Close()");
    }
}

//----------------------------------------------------------------------------------------------------
u32
HIO2CommDevice::ReadMailbox(HIO2Handle devHandle)
{
    Debug::Assert(devHandle != HIO2_INVALID_HANDLE_VALUE);

    u32 mailWord;
    if (! spfHIO2ReadMailbox_(devHandle, &mailWord))
    {
        throw gcnew HIO2FunctionException("HIO2ReadMailbox()");
    }

    return mailWord;
}

//----------------------------------------------------------------------------------------------------
void
HIO2CommDevice::WriteMailbox(HIO2Handle devHandle, u32 mailWord)
{
    Debug::Assert(devHandle != HIO2_INVALID_HANDLE_VALUE);

    if (! spfHIO2WriteMailbox_(devHandle, mailWord))
    {
        throw gcnew HIO2FunctionException("HIO2WriteMailbox()");
    }
}

//----------------------------------------------------------------------------------------------------
u32
HIO2CommDevice::ReadStatus(HIO2Handle devHandle)
{
    Debug::Assert(devHandle != HIO2_INVALID_HANDLE_VALUE);

    u32 hioStatus;
    if (! spfHIO2ReadStatus_(devHandle, &hioStatus))
    {
        throw gcnew HIO2FunctionException("HIO2ReadStatus()");
    }
    return hioStatus;
}

//----------------------------------------------------------------------------------------------------
IntPtr
HIO2CommDevice::RoundUp(
    IntPtr value,
    u32    alignment
)
{
    Int64 mask = alignment - 1;
    return IntPtr((value.ToInt64() + mask) & ~mask);
}

//----------------------------------------------------------------------------------------------------
HIO2CommDevice::HIO2CommDevice(String^ devPathName)
:   bTargetConnect_(false),
    bInitHIO2_(false),
    devHandle_(HIO2_INVALID_HANDLE_VALUE),
    pReader_(0),
    pWriter_(0),
    hTempBuf_(IntPtr::Zero)
{
    using namespace nw4r::mcs;

    try
    {
        InitHIO2();
        bInitHIO2_ = true;

        devHandle_  = OpenDevice(devPathName);
        Debug::WriteLine(String::Format("HIO2Open() device path name {0}.", devPathName));

        if (0 != (ReadStatus(devHandle_) & HIO2_STATUS_RX))   // 未読メールがあれば呼んでおく
        {
            u32 mailWord = ReadMailbox(devHandle_);
            Debug::WriteLine(String::Format("clear mail box. drop mail {0:X8}", mailWord));
        }

        WriteMailbox(devHandle_, detail::HioNegotiate::RebootCode);
        Debug::WriteLine("Write reboot mail.");
    }
    catch (HIO2FunctionException^ ex)
    {
        // コンストラクタが失敗するので、元に戻す
        DeviceFinalize();
        // ファイナライザの呼び出しを禁止します。
        GC::SuppressFinalize(this);
        throw gcnew HIO2CommDeviceException(ex->Message, ex);
    }

    hTempBuf_ = Marshal::AllocHGlobal(detail::Hio2RingBuffer::CopyAlignment + READ_TRANSBUF_SIZE + WRITE_TRANSBUF_SIZE);
    IntPtr readBuffer = RoundUp(hTempBuf_, detail::Hio2RingBuffer::CopyAlignment);
    IntPtr writeBuffer = IntPtr(readBuffer.ToInt64() + READ_TRANSBUF_SIZE);

    pReader_ = new detail::Hio2RingBuffer(devHandle_, 0                 , READ_TRANSBUF_SIZE , readBuffer.ToPointer());
    pWriter_ = new detail::Hio2RingBuffer(devHandle_, READ_TRANSBUF_SIZE, WRITE_TRANSBUF_SIZE, writeBuffer.ToPointer());
}

//----------------------------------------------------------------------------------------------------
HIO2CommDevice::~HIO2CommDevice()
{
    if (bTargetConnect_)
    {
        try
        {
            WriteMailbox(devHandle_, nw4r::mcs::detail::HioNegotiate::Disconnect);
        }
        catch (HIO2FunctionException^ e)
        {
            Debug::WriteLine(String::Format("HIO2WriteMailbox() failed. - ({0})", e->ErrorCode));
        }
        bTargetConnect_ = false;
    }

    // アンマネージリソースの解放
    this->!HIO2CommDevice();
}

//----------------------------------------------------------------------------------------------------
HIO2CommDevice::!HIO2CommDevice()
{
    delete pWriter_;
    pWriter_ = 0;

    delete pReader_;
    pReader_ = 0;

    if (hTempBuf_ != IntPtr::Zero)
    {
        Marshal::FreeHGlobal(hTempBuf_);
        hTempBuf_ = IntPtr::Zero;
    }

    DeviceFinalize();
}

//----------------------------------------------------------------------------------------------------
void
HIO2CommDevice::Negotiate()
{
    using namespace nw4r::mcs;

    if (! IsAttach)
    {
        return;
    }

    try
    {
        if (0 != (ReadStatus(devHandle_) & HIO2_STATUS_RX))   // 未読メールがある
        {
            switch (ReadMailbox(devHandle_))
            {
            case detail::HioNegotiate::RebootCode:
                if (bTargetConnect_)
                {
                    Debug::WriteLine("client reset.");
                }
                bTargetConnect_ = false;    // 一旦ネゴシエーション状態を解除

                // 共有メモリ上にリングバッファを構築
                pReader_->EnablePort();
                pWriter_->EnablePort();
                pReader_->InitBuffer();
                pWriter_->InitBuffer();

                // 共有メモリを初期化したことを伝える
                WriteMailbox(devHandle_, detail::HioNegotiate::InitBufferCode);
                break;

            case detail::HioNegotiate::AckCode:
                Debug::WriteLine("Negotiation end.");
                bTargetConnect_ = true;
                break;
            }
        }
    }
    catch (HIO2FunctionException^ ex)
    {
        throw gcnew HIO2CommDeviceException(ex->Message, ex);
    }
}

//----------------------------------------------------------------------------------------------------
String^
HIO2CommDevice::Name::get()
{
    return "NDEV";
}

//----------------------------------------------------------------------------------------------------
bool
HIO2CommDevice::IsAttach::get()
{
    return devHandle_ != HIO2_INVALID_HANDLE_VALUE;
}

//----------------------------------------------------------------------------------------------------
bool
HIO2CommDevice::IsTargetConnect::get()
{
    return bTargetConnect_;
}

//----------------------------------------------------------------------------------------------------
bool
HIO2CommDevice::Read()
{
    bool bAvailable;
    if (! pReader_->Read(&bAvailable))
    {
        throw gcnew HIO2CommDeviceException("Read() failed.");
    }
    return bAvailable;
}

//----------------------------------------------------------------------------------------------------
IntPtr
HIO2CommDevice::GetMessage([Out] int% channel, [Out] int% size)
{
    u32 wkChannel, wkSize;
    void *const msgPtr = pReader_->GetMessage(&wkChannel, &wkSize);

    channel = wkChannel;
    size = wkSize;

    return IntPtr(msgPtr);
}

//----------------------------------------------------------------------------------------------------
int
HIO2CommDevice::GetWritableBytes(bool withUpdate)
{
    u32 writableBytes;
    if (pWriter_->GetWritableBytes(&writableBytes, withUpdate))
    {
        return writableBytes;
    }
    else
    {
        throw gcnew HIO2CommDeviceException("GetWritableBytes() failed.");
    }
}

//----------------------------------------------------------------------------------------------------
void
HIO2CommDevice::Write(int channel, array<Byte>^ buf, int offset, int size)
{
    if (buf == nullptr)
    {
        throw gcnew ArgumentNullException();
    }

    if (offset < 0 || size < 0 || offset + size > buf->Length)
    {
        throw gcnew ArgumentOutOfRangeException();
    }

    pin_ptr<Byte> pinBuf = &buf[offset];
    if (! pWriter_->Write(channel, pinBuf, size))
    {
        throw gcnew HIO2CommDeviceException("Write() failed.");
    }
}

//----------------------------------------------------------------------------------------------------
UInt32
HIO2CommDevice::GetWriteBufferSize()
{
    return pWriter_->GetBufferSize();
}

//----------------------------------------------------------------------------------------------------
void
HIO2CommDevice::DeviceFinalize()
{
    if (devHandle_ != HIO2_INVALID_HANDLE_VALUE)
    {
        try
        {
            CloseDevice(devHandle_);
        }
        catch (HIO2FunctionException^ e)
        {
            Debug::WriteLine(String::Format("HIO2Close() failed. - ({0})", e->ErrorCode));
        }

        devHandle_ = HIO2_INVALID_HANDLE_VALUE;
    }

    if (bInitHIO2_)
    {
        spfHIO2Exit_();
        bInitHIO2_ = false;
    }
}

//----------------------------------------------------------------------------------------------------
HIO2CommDevice::HIO2FunctionException::HIO2FunctionException()
:   Exception(String::Format("HIO2 API error. - ({0})", int(spfHIO2GetLastError_())))
{
    errorCode_ = spfHIO2GetLastError_();
}

//----------------------------------------------------------------------------------------------------
HIO2CommDevice::HIO2FunctionException::HIO2FunctionException(System::String^ description)
:   Exception(String::Format("HIO2 API error. - ({0})", int(spfHIO2GetLastError_())) + description)
{
    errorCode_ = spfHIO2GetLastError_();
}

}   // namespace McsServer

}   // namespace Nintendo
