﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Net;
using System.Net.Sockets;
using System.Diagnostics;
using System.IO;
using System.Threading;
using Nintendo.McsServer.McsUtil;

namespace Nintendo.McsServer.CafeHioCommDevice
{
    enum ChunkID : uint
    {
        Unknown = 0,
        RegistName,
        Reboot,
        InitBuffer,
        Ack,
        Disconnect,
        MessageToServer,
        MessageToAppication,
        ServerTime,
        CheckAlive,
    }

    enum ResultCode : uint
    {
        Success, // 成功。
        ProtocolError, // 通信プロトコルのエラーです。
        AlreadyRegisted, // 既に登録されています。
    }

    /// <summary>
    /// 検出されたデバイスごとに呼ばれるコールバックです。
    /// </summary>
    /// <param name="id">デバイスを識別するIDです。</param>
    /// <returns>デバイスの検出を完了（中断）したい場合は true を返します。</returns>
    public delegate bool EnumDeviceCallback(object id);

    /// <summary>
    /// Cafe HIO による通信モジュールです。
    /// </summary>
    public class CommDevice : ICommDevice, ICheckAlive
    {
        /// <summary>
        /// MCSチャンクの受信を行うスレッド。
        /// </summary>
        Thread _ReadThread = null;

        /// <summary>
        /// 受信スレッドの動作を停止するための処理。
        /// </summary>
        Action _ReadThreadAborter;

        /// <summary>
        /// MCS通信用のソケット
        /// </summary>
        McsSocket _CommSocket = null;

        /// <summary>
        /// リードスレッドからメインスレッドへの要求キュー。
        /// </summary>
        Queue<NegoRequest> _NegotiateQueue = new Queue<NegoRequest>();

        /// <summary>
        /// Negotiate() で認知されたメッセージ。
        /// </summary>
        Queue<MessageData> _ReadQueue = new Queue<MessageData>();

        /// <summary>
        /// 書き込みキュー。
        /// </summary>
        Queue<ArraySegment<byte>> _WriteQueue = new Queue<ArraySegment<byte>>();

        /// <summary>
        /// 書き込み進行中を表すフラグ。
        /// </summary>
        bool _bWriteProgress = false;

        /// <summary>
        /// MCS通信に使用するHIOチャンネル。
        /// </summary>
        byte[] _ChannelName = Encoding.ASCII.GetBytes("NW_MCS");

        /// <summary>
        /// 同期用オブジェクト。
        /// </summary>
        readonly object _syncObj = new object();

        /// <summary>
        /// (コマンド)パケットヘッダサイズ。
        /// </summary>
        const int ChunkHeaderSize = 8;

        /// <summary>
        /// 読み込みバッファサイズ
        /// </summary>
        const int ReadBuffSize = 32 * 1024;

        /// <summary>
        /// 最大チャンクボディサイズ
        /// </summary>
        const int MaxChunkBodySize = ReadBuffSize - ChunkHeaderSize;

        /// <summary>
        /// 最大送信チャンクデータサイズ
        /// TODO: 8バイト小さくしておかないと実機側がOnReadで止まってしまう。
        /// </summary>
        const int MaxWriteChunkBodySize = 0x1000 - ChunkHeaderSize - 8;

        /// <summary>
        /// データバッファ。
        /// 実機アプリからMCSサーバへの書き込みは、ソケットを通じて、
        /// 一旦このバッファに書き込まれます。
        /// </summary>
        readonly byte[] _dataReadBuf = new byte[ReadBuffSize];

        /// <summary>
        /// メッセージヘッダ
        /// UInt32 channel
        /// </summary>
        const int MessageHeaderSize = 8;

        /// <summary>
        /// 接続状況フラグ。
        /// 通信が可能な場合は true になります。
        /// </summary>
        bool _bTargetConnected = false;

        /// <summary>
        /// ターゲットプログラムが停止したと判断する上限の時間[秒]。
        /// 0 の場合はタイムアウトによる生存判定は無効。
        /// </summary>
        uint _CheckAliveTimeout = 0;

        /// <summary>
        /// 生存確認の失敗によりターゲットとの接続を切った場合に設定されます。
        /// </summary>
        bool _DisconnectByCheckAlive = false;

        /// <summary>
        /// 最後にターゲットプログラムからの受信データを処理した時間。
        /// </summary>
        DateTime _LastNegotiateTime = DateTime.Now;

        /// <summary>
        /// ターゲットプログラムに生存確認要求をしたか。
        /// （要求を繰り返し送信しないようにするため）
        /// </summary>
        bool _CheckAliveWaiting;

        static CommDevice _sInstance = null;

        public static void Start()
        {
            _sInstance = new CommDevice();
        }

        public static void Stop()
        {
            if (_sInstance != null)
            {
                _sInstance.Dispose();
                _sInstance = null;
            }
        }

        /// <summary>
        /// 利用可能な通信デバイスを列挙します。
        /// </summary>
        /// <param name="callback">通信デバイスのIDを引数としてコールバックされます。</param>
        public static void EnumDevices(EnumDeviceCallback callback)
        {
            if (_sInstance != null)
            {
                callback.Invoke(_sInstance);
            }
        }

        /// <summary>
        /// 通信デバイスを取得します。
        /// </summary>
        /// <param name="id">EnumDevices() で取得したデバイス ID のうちの１つを指定します。</param>
        /// <returns>通信デバイスを返します。引数 id がこの通信デバイスのもので無い場合は null を返します。</returns>
        public static CommDevice GetDevice(object id)
        {
            if (id is CommDevice)
            {
                return (CommDevice)id;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// 通信状態のデバイスを１つ取得します。
        /// </summary>
        /// <returns></returns>
        public static CommDevice GetDevice()
        {
            return _sInstance;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        CommDevice()
        {
        }

        /// <summary>
        /// デストラクタです。
        /// </summary>
        ~CommDevice()
        {
            _ReadThread = null;
        }

        void StopReadThread()
        {
            lock (_syncObj)
            {
                if (_ReadThread != null)
                {
                    if (_ReadThreadAborter != null)
                    {
                        _ReadThreadAborter.Invoke();
                    }

                    _ReadThread.Abort();
                    _ReadThread = null;
                }
            }
        }

        void StartReadThread()
        {
            lock (_syncObj)
            {
                _ReadThread = new Thread(ReadThreadProc);
                _ReadThread.Start();
            }
        }

        void SetKeepAlive(Socket socket, int msec)
        {
            byte[] inBuffer = new byte[12];
            BitConverter.GetBytes(1).CopyTo(inBuffer, 0);     // スイッチ
            BitConverter.GetBytes(msec).CopyTo(inBuffer, 4); // Interval
            socket.IOControl(IOControlCode.KeepAliveValues, inBuffer, null);
        }

        void ReadThreadProc()
        {
            // MCS 用のポートを取得
            int port = -1;

            try
            {
                TcpClient client = new TcpClient("localhost", 6003);
                if (client != null)
                {
                    lock (_syncObj)
                    {
                        _ReadThreadAborter = delegate()
                        {
                            try { client.GetStream().Close(); }
                            catch { }
                        };
                    }

                    port = this.GetTargetPort(client);

                    lock (_syncObj)
                    {
                        _ReadThreadAborter = null;
                    }

                    client.Close();
                }
            }
            catch (SocketException)
            {
            }

            if (port == -1)
            {
                lock (_syncObj)
                {
                    _ReadThread = null;
                    return;
                }
            }

            BinaryReader reader = new NetBinaryReader(new MemoryStream(_dataReadBuf));

            try
            {
                TcpClient client = new TcpClient("localhost", port);
                this.SetKeepAlive(client.Client, 1000);
                _CommSocket = new McsSocket(client.Client);
                _CommSocket.CancelWaitHandle = new ManualResetEvent(false);

                lock (_syncObj)
                {
                    _ReadThreadAborter = delegate()
                    {
                        try
                        {
                            ManualResetEvent ev = (ManualResetEvent)_CommSocket.CancelWaitHandle;
                            ev.Set();
                            _CommSocket.Close();
                        }
                        catch { }
                    };
                }

                while (true)
                {
                    _CommSocket.Read(_dataReadBuf, 0, ChunkHeaderSize);
                    reader.BaseStream.Position = 0;

                    ChunkID chunkID = (ChunkID)reader.ReadUInt32();
                    int chunkBodySize = (int)reader.ReadUInt32();

                    switch (chunkID)
                    {
                        case ChunkID.MessageToServer:
                            {
                                // メッセージを受信し、キューに登録します。
                                _CommSocket.Read(_dataReadBuf, 0, MessageHeaderSize);
                                reader.BaseStream.Position = 0;
                                int msgChannel = (int)reader.ReadUInt32();
                                int msgBytes = (int)reader.ReadUInt32();

                                if (MessageHeaderSize + msgBytes > chunkBodySize)
                                {
                                    throw new ChunkException(chunkID);
                                }

                                byte[] msgBody = new byte[chunkBodySize - MessageHeaderSize];
                                _CommSocket.Read(msgBody, 0, msgBody.Length);

                                MessageData msgData = new MessageData(msgChannel, msgBody, 0, msgBytes);

                                lock (_syncObj)
                                {
                                    _NegotiateQueue.Enqueue(new NegoRequest(ChunkID.MessageToServer,  msgData));
                                }
                            }
                            break;

                        case ChunkID.Reboot:
                            _CommSocket.Skip(chunkBodySize);
                            lock (_syncObj)
                            {
                                _NegotiateQueue.Enqueue(new NegoRequest(ChunkID.Reboot, _CommSocket));
                            }
                            break;

                        case ChunkID.Ack:
                            _CommSocket.Skip(chunkBodySize);
                            lock (_syncObj)
                            {
                                _NegotiateQueue.Enqueue(new NegoRequest(ChunkID.Ack, _CommSocket));
                            }

                            break;

                        case ChunkID.CheckAlive:
                            _CommSocket.Skip(chunkBodySize);
                            lock (_syncObj)
                            {
                                _NegotiateQueue.Enqueue(new NegoRequest(ChunkID.CheckAlive));
                            }
                            break;

                        default:
                            _CommSocket.Skip(chunkBodySize);
                            break;
                    }
                }
            }
            catch (System.Exception)
            {
            }
            finally
            {
                if (_CommSocket != null)
                {
                    _CommSocket.Close();
                }

                lock (_syncObj)
                {
                    _NegotiateQueue.Enqueue(new NegoRequest(ChunkID.Disconnect));
                    _ReadThreadAborter = null;
                    _CommSocket = null;
                }
            }
        }

        void Response(ChunkID chunkID)
        {
            const UInt32 MessageSize = 0;
            byte[] resBuf = new byte[ChunkHeaderSize + MessageSize];
            var bw = new NetBinaryWriter(new MemoryStream(resBuf));
            bw.Write((UInt32)chunkID);
            bw.Write(MessageSize);
            bw.Flush();

            try
            {
                this.WriteData(resBuf, 0, resBuf.Length);
            }
            catch
            {
            }
        }

        void Response(ChunkID chunkID, ResultCode result)
        {
            const UInt32 MessageSize = 4;
            byte[] resBuf = new byte[ChunkHeaderSize + MessageSize];
            var bw = new NetBinaryWriter(new MemoryStream(resBuf));
            bw.Write((UInt32)chunkID);
            bw.Write(MessageSize);
            bw.Write((UInt32)result);
            bw.Flush();

            try
            {
                this.WriteData(resBuf, 0, resBuf.Length);
            }
            catch
            {
            }
        }

        void SendServerTime()
        {
            const UInt32 MessageSize = 8;
            byte[] resBuf = new byte[ChunkHeaderSize + MessageSize];
            var bw = new NetBinaryWriter(new MemoryStream(resBuf));
            bw.Write((UInt32)ChunkID.ServerTime);
            bw.Write(MessageSize);

            // 2000年 1月 1日 午前0時からの経過時間(msec)。
            long msec = (long)(DateTime.Now - new DateTime(2000, 1, 1)).TotalMilliseconds;
            bw.Write((Int64)msec);
            bw.Flush();

            try
            {
                this.WriteData(resBuf, 0, resBuf.Length);
            }
            catch
            {
            }
        }

        /// <summary>
        /// 文字列をチャンネル情報の配列に分割します。
        /// </summary>
        /// <param name="message">全チャンネル情報のはいった文字列です。</param>
        /// <returns>チャンネル情報単位に分割して配列を返します。</returns>
        IList<string> SplitChannelInfos(string message)
        {
            List<string> stringList = new List<string>();

            for (int index = 0; index < message.Length;)
            {
                if (message[index] == '+' || message[index] == '-')
                {
                    int i = index + 1;
                    for (; i < message.Length; ++i)
                    {
                        if (message[i] == '+' || message[i] == '-')
                        {
                            break;
                        }
                    }

                    stringList.Add(message.Substring(index, i - index));
                    index = i;
                }
                else
                {
                    ++index;
                }
            }

            return stringList;
        }

        /// <summary>
        /// ネットワーク上のストリームからポート情報を読み出します。
        /// </summary>
        /// <param name="stream">ポート情報を取得する為のストリームです。</param>
        /// <returns>NW_MCS のポート番号を返します。</returns>
        int GetTargetPort(TcpClient client)
        {
            MemoryStream memory = new MemoryStream();

            NetworkStream stream = client.GetStream();

            try
            {
                int mcsPortNo = -1;

                int readBytes = 0;
                byte[] buffer = new byte[256];

                do
                {
                    readBytes = stream.Read(buffer, 0, buffer.Length);

                    if (readBytes > 0)
                    {
                        memory.Write(buffer, 0, readBytes);
                    }

                } while (client.Available > 0);

                string infoString = Encoding.ASCII.GetString(memory.ToArray());

                IList<string> channels = this.SplitChannelInfos(infoString);

                foreach (string channel in channels)
                {
                    string[] tokens = channel.Split(new char[] { ':' }, 2);

                    if (tokens.Length != 2)
                    {
                        continue;
                    }

                    string channelName = tokens[0];
                    int port = int.Parse(tokens[1]);

                    if (0 != channelName.Substring(1).CompareTo("NW_MCS"))
                    {
                        continue;
                    }

                    if (channelName[0] == '+')
                    {
                        return port;
                    }

                    mcsPortNo = port;
                }

                return mcsPortNo;
            }
            catch (IOException)
            {
                return -1;
            }
            finally
            {
                stream.Close();
            }
        }

        public void Negotiate()
        {
            if (_ReadThread != null && !_ReadThread.IsAlive)
            {
                _ReadThread = null;
            }

            if (_ReadThread == null)
            {
                this.StartReadThread();
            }

            bool bNegoProcessed = false;

            while (_NegotiateQueue.Count > 0)
            {
                bNegoProcessed = true;
                NegoRequest negoRequest;

                lock (_syncObj)
                {
                    negoRequest = _NegotiateQueue.Dequeue();
                }

                switch (negoRequest.ChunkID)
                {
                    case ChunkID.Reboot:
                        this.Response(ChunkID.InitBuffer);
                        _ReadQueue.Clear();
                        lock ((_WriteQueue as ICollection).SyncRoot)
                        {
                            _WriteQueue.Clear();
                            _bWriteProgress = false;
                        }
                        break;

                    case ChunkID.Ack:
                        _bTargetConnected = true;
                        _DisconnectByCheckAlive = false;

                        // Negotiate 中に状態は一度しか変化させない。
                        goto Pass;

                    case ChunkID.Disconnect:
                        _bTargetConnected = false;

                        // Negotiate 中に状態は一度しか変化させない。
                        goto Pass;

                    case ChunkID.MessageToServer:
                        _ReadQueue.Enqueue((MessageData)negoRequest.Data);
                        break;

                    case ChunkID.CheckAlive:
                        break;

                    default:
                        break;
                }
            }

            Pass:

            // ターゲットプログラムの生存確認
            if (_CheckAliveTimeout > 0)
            {
                if (bNegoProcessed)
                {
                    _LastNegotiateTime = DateTime.Now;
                    _CheckAliveWaiting = false;
                }

                if (_bTargetConnected)
                {
                    TimeSpan timeSpan = DateTime.Now - _LastNegotiateTime;

                    if (!_CheckAliveWaiting)
                    {
                        // しばらくターゲットからのデータの受信がなかったら生存確認を要求する。
                        if (timeSpan > TimeSpan.FromSeconds(0.5 * _CheckAliveTimeout))
                        {
                            this.Response(ChunkID.CheckAlive);
                            _CheckAliveWaiting = true;
                        }
                    }
                    else
                    {
                        // 規定時間までにターゲットから何らかのデータの受信がなかったら停止したと判断する。
                        if (timeSpan > TimeSpan.FromSeconds(_CheckAliveTimeout))
                        {
                            this.Response(ChunkID.Disconnect);
                            _bTargetConnected = false;
                            _DisconnectByCheckAlive = true;
                        }
                    }
                }
            }

            if (_bTargetConnected)
            {
#if false // FIXME! Writeがブロックされてデッドロックする
                try
                {
                    this.SendServerTime();
                }
                catch (IOException)
                {
                }
                catch (SocketException)
                {
                }
                catch (ObjectDisposedException)
                {
                }
#endif
            }
        }

        public string Name
        {
            get { return "CAT-DEV"; }
        }

        public bool IsAttach
        {
            get { return true; }
        }

        public bool IsTargetConnect
        {
            get { return _bTargetConnected; }
        }

        uint ICheckAlive.CheckAliveTimeout
        {
            get
            {
                return _CheckAliveTimeout;
            }
            set
            {
                _CheckAliveTimeout = value;
            }
        }

        bool ICheckAlive.IsDisconnectByCheckAlive
        {
            get
            {
                return _DisconnectByCheckAlive;
            }
        }

        public bool Read()
        {
            if (_ReadQueue.Count > 0)
            {
                return true;
            }

            return false;
        }

        public MessageData GetMessage()
        {
            if (_ReadQueue.Count == 0)
            {
                return null;
            }
            else
            {
                return _ReadQueue.Dequeue();
            }
        }

        public int GetWritableBytes(bool withUpdate)
        {
            try
            {
                lock (_syncObj)
                {
                    if (_CommSocket != null)
                    {
                        return Math.Min(MaxWriteChunkBodySize, _CommSocket.SendBufferSize);
                    }
                    else
                    {
                        return 0;
                    }
                }
            }
            catch (ObjectDisposedException)
            {
                return 0;
            }
        }

        public void Write(int channel, byte[] buf, int offset, int size)
        {
            try
            {
                while (0 < size)
                {
                    var writeSize = Math.Min(size, MaxWriteChunkBodySize - MessageHeaderSize);
                    var chunkBodySize = MessageHeaderSize + writeSize;
                    var message = new byte[ChunkHeaderSize + MessageHeaderSize];
                    var bw = new NetBinaryWriter(new MemoryStream(message));
                    bw.Write((UInt32)ChunkID.MessageToAppication);
                    bw.Write((UInt32)chunkBodySize);
                    bw.Write((UInt32)channel);
                    bw.Write((UInt32)writeSize);
                    bw.Flush();

                    this.WriteData(message, 0, message.Length);
                    this.WriteData(buf, offset, writeSize);

                    offset += writeSize;
                    size -= writeSize;
                }
            }
            catch
            {
            }
        }

        private void WriteData(byte[] buff, int offset, int size)
        {
            lock ((_WriteQueue as ICollection).SyncRoot)
            {
                if (_WriteQueue.Count > 0)
                {
                    _WriteQueue.Enqueue(new ArraySegment<byte>(buff, offset, size));

                    if (!_bWriteProgress && _CommSocket != null)
                    {
                        ArraySegment<byte> item = _WriteQueue.Dequeue();
                        _bWriteProgress = true;
                        _CommSocket.BeginWrite(item.Array, item.Offset, item.Count, Callback_WriteData);
                    }
                }
                else
                {
                    if (!_bWriteProgress && _CommSocket != null)
                    {
                        _bWriteProgress = true;
                        _CommSocket.BeginWrite(buff, offset, size, Callback_WriteData);
                    }
                    else
                    {
                        _WriteQueue.Enqueue(new ArraySegment<byte>(buff, offset, size));
                    }
                }
            }
        }

        private void Callback_WriteData(IAsyncResult ar)
        {
            try
            {
                _CommSocket.EndWrite(ar);
            }
            catch (OperationCanceledException)
            {
                return;
            }
            catch
            {
                return;
            }

            lock ((_WriteQueue as ICollection).SyncRoot)
            {
                if (_CommSocket != null && _WriteQueue.Count > 0)
                {
                    ArraySegment<byte> item = _WriteQueue.Dequeue();
                    _CommSocket.BeginWrite(item.Array, item.Offset, item.Count, Callback_WriteData);
                }
                else
                {
                    _bWriteProgress = false;
                }
            }
        }

        public uint GetWriteBufferSize()
        {
            return (uint)this.GetWritableBytes(false);
        }

        void Dispose(Exception e)
        {
            ReportException(e);
            Dispose();
        }

        public void Dispose()
        {
            lock (_syncObj)
            {
                // Socket.Close()を呼ぶ前に取得しておく。
                string infoStr = GetInfoStr();

                try
                {
                    this.StopReadThread();

                    Log(infoStr + " : socket closed.");
                }
                catch (SocketException e)
                {
                    InternalError(infoStr + " : " + e);
                }

                this._bTargetConnected = false;
            }
        }

        void Log(string message)
        {
            Debug.WriteLine(message);
        }

        void Log(string format, object[] args)
        {
            Debug.Print(format, args);
            Debug.WriteLine("");
        }

        void InternalError(string message)
        {
            Debug.WriteLine(message);
        }

        void InternalError(string format, object[] args)
        {
            Debug.Write("internal error : ");
            Debug.Print(format, args);
            Debug.WriteLine("");
        }

        void ReportException(Exception e)
        {
            if (e == null)
            {
                return;
            }

            if (e is SocketException)
            {
                var e2 = (SocketException) e;

                switch (e2.SocketErrorCode)
                {
                    case SocketError.Shutdown:
                    case SocketError.ConnectionReset:
                    case SocketError.ConnectionAborted:
                        Debug.WriteLine(GetInfoStr() + " : " + e2.SocketErrorCode);
                        return;
                }
            }

            Debug.WriteLine(GetInfoStr() + " : " + e);
        }

        /// <summary>
        /// 説明文字列を取得します。
        /// </summary>
        /// <returns>文字列を返します。</returns>
        string GetInfoStr()
        {
            System.Net.EndPoint endPoint = null;

            lock (_syncObj)
            {
                if (_CommSocket != null)
                {
                    endPoint = _CommSocket.RemoteEndPoint;
                }
            }

            if (endPoint != null)
            {
                return string.Format("AppSocket [{0}]", endPoint);
            }
            else
            {
                return string.Format("AppSocket [Unknown]");
            }
        }

        public override string ToString()
        {
            return Name;
        }

        /// <summary>
        /// リードスレッドからメインスレッドへの要求データ。
        /// </summary>
        class NegoRequest
        {
            public ChunkID ChunkID
            {
                get;
                set;
            }

            public object Data
            {
                get;
                set;
            }

            public NegoRequest(ChunkID chunkID)
                : this(chunkID, null)
            {
            }

            public NegoRequest(ChunkID chunkID, object data)
            {
                this.ChunkID = chunkID;
                this.Data = data;
            }
        }

        /// <summary>
        /// ソケットがShutdownされたと判断した場合にスローする例外。
        /// コネクションが切れる。
        /// </summary>
        class ShutdownException : Exception
        {
        }

        /// <summary>
        /// チャンクの内容に不正があった場合にスローする例外。
        /// チャンクの受付は継続する。
        /// </summary>
        class ChunkException : Exception
        {
            readonly ChunkID _chunkID;

            public ChunkID ChunkID { get { return _chunkID; } }

            public ChunkException(ChunkID id)
            {
                _chunkID = id;
            }
        }
    }

    class StreamClosedException : Exception
    {
    }
}
