﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "XpadInputer.h"

namespace
{
    // HOMEボタン(-2), CAPTUREボタン(-1)
    const int g_XpadButtonMaxValue = 15;
    const int g_XpadButtonMinValue = -2;
}


XpadInputer::XpadInputer(IoPortSender* sender, int controllerId)
    : m_Sender(sender)
    , m_ControllerId(controllerId)
{
}

XpadInputer::~XpadInputer()
{
}

ControllerButtonValueType XpadInputer::Button2Flag(Button button)
{
    return (static_cast<ControllerButtonValueType>(1) << button);
}

void XpadInputer::BeginAutoPilot()
{
    m_Sender->SetXpadConnection(m_ControllerId, static_cast<int32_t>(HidShellXpadPowerState::OnBattery));
}

void XpadInputer::EndAutoPilot()
{
    m_Sender->SetXpadConnection(m_ControllerId, static_cast<int32_t>(HidShellXpadPowerState::Disconnected));
}

void XpadInputer::SetDefault()
{
    m_Sender->SetXpadDefault(m_ControllerId);
}

void XpadInputer::SetState(const Button buttonList[], int buttonCount,
    const Stick stickList[], const ControllerStickValueType stickXList[], const ControllerStickValueType stickYList[], int stickCount)
{
    ControllerButtonValueType buttons = static_cast<ControllerButtonValueType>(0);
    for (int i = 0; i < buttonCount; i++)
    {
        CheckButton(buttonList[i]);
        buttons |= Button2Flag(buttonList[i]);
    }
    for (int i = 0; i < stickCount; i++)
    {
        CheckStick(stickList[i]);
    }
    m_Sender->SetXPadState(m_ControllerId, buttons, stickList, stickXList, stickYList, stickCount);
}

void XpadInputer::CheckButton(Button type)
{
    if (type < g_XpadButtonMinValue || type > g_XpadButtonMaxValue)
    {
        throw InvalidButtonException(type);
    }
}


void XpadInputer::CheckStick(Stick type)
{
    if (type < 0 || type >= Stick::Stick_MaxCount)
    {
        throw InvalidStickException(type);
    }
}

void XpadInputer::Press(Button button)
{
    CheckButton(button);
    switch (button)
    {
        case Button::Button_Home:
            m_Sender->SetHomeButton(m_ControllerId, 1);
            break;
        case Button::Button_Capture:
             m_Sender->SetCaptureButton(m_ControllerId, 1);
             break;
        default:
            m_Sender->SetXpadPressButton(m_ControllerId, Button2Flag(button));
            break;
    }
}

void XpadInputer::Release(Button button)
{
    CheckButton(button);
    switch (button)
    {
        case Button::Button_Home:
            m_Sender->SetHomeButton(m_ControllerId, 0);
            break;
        case Button::Button_Capture:
             m_Sender->SetCaptureButton(m_ControllerId, 0);
             break;
        default:
            m_Sender->SetXpadReleaseButton(m_ControllerId, Button2Flag(button));
            break;
    }
}

void XpadInputer::SetStickPos(Stick stick, ControllerStickValueType x, ControllerStickValueType y)
{
    CheckStick(stick);
    switch (stick)
    {
        case Stick::Stick_L:
            SetLStickPos(x, y);
            break;
        case Stick::Stick_R:
            SetRStickPos(x, y);
            break;
        default:
            break;
    }
}

void XpadInputer::SetLStickPos(ControllerStickValueType x, ControllerStickValueType y)
{
    m_Sender->SetXpadStickL(m_ControllerId, x, y);
}

void XpadInputer::SetRStickPos(ControllerStickValueType x, ControllerStickValueType y)
{
    m_Sender->SetXpadStickR(m_ControllerId, x, y);
}
