﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VirtualPadInputer.h"

VirtualPadInputer::VirtualPadInputer(
    IoPortSender* sender, int controllerId,
    ControllerDeviceType deviceType, ControllerInterfaceType interfaceType,
    ControllerColor mainColor, ControllerColor subColor)
    : m_Sender(sender)
    , m_ControllerId(controllerId)
    , m_DeviceType(deviceType)
    , m_InterfaceType(interfaceType)
    , m_MainColor(mainColor)
    , m_SubColor(subColor)
{
}

VirtualPadInputer::~VirtualPadInputer()
{
}

void VirtualPadInputer::BeginAutoPilot()
{
    switch (m_DeviceType)
    {
        case ControllerDeviceType::ControllerDeviceType_SwitchProController:
            if (m_InterfaceType == ControllerInterfaceType::ControllerInterfaceType_Bluetooth)
            {
                m_Sender->SetVirtualPadDeviceType(m_ControllerId, AbstractedPadDeviceType::AbstractedPadDeviceType_SwitchProController);
                m_Sender->SetVirtualPadInterfaceType(m_ControllerId, AbstractedPadInterfaceType::AbstractedPadInterfaceType_Bluetooth);
            }
            break;
        case ControllerDeviceType::ControllerDeviceType_JoyConLeft:
            if (m_InterfaceType == ControllerInterfaceType::ControllerInterfaceType_Bluetooth)
            {
                m_Sender->SetVirtualPadDeviceType(m_ControllerId, AbstractedPadDeviceType::AbstractedPadDeviceType_JoyConLeft);
                m_Sender->SetVirtualPadInterfaceType(m_ControllerId, AbstractedPadInterfaceType::AbstractedPadInterfaceType_Bluetooth);
            }
            else if (m_InterfaceType == ControllerInterfaceType::ControllerInterfaceType_Rail)
            {
                m_Sender->SetVirtualPadDeviceType(m_ControllerId, AbstractedPadDeviceType::AbstractedPadDeviceType_HandheldJoyLeft);
                m_Sender->SetVirtualPadInterfaceType(m_ControllerId, AbstractedPadInterfaceType::AbstractedPadInterfaceType_Rail);
            }
            break;
        case ControllerDeviceType::ControllerDeviceType_JoyConRight:
            if (m_InterfaceType == ControllerInterfaceType::ControllerInterfaceType_Bluetooth)
            {
                m_Sender->SetVirtualPadDeviceType(m_ControllerId, AbstractedPadDeviceType::AbstractedPadDeviceType_JoyConRight);
                m_Sender->SetVirtualPadInterfaceType(m_ControllerId, AbstractedPadInterfaceType::AbstractedPadInterfaceType_Bluetooth);
            }
            else if (m_InterfaceType == ControllerInterfaceType::ControllerInterfaceType_Rail)
            {
                m_Sender->SetVirtualPadDeviceType(m_ControllerId, AbstractedPadDeviceType::AbstractedPadDeviceType_HandheldJoyRight);
                m_Sender->SetVirtualPadInterfaceType(m_ControllerId, AbstractedPadInterfaceType::AbstractedPadInterfaceType_Rail);
            }
            break;
        default:
            throw InvalidControllerDeviceInofException(m_DeviceType, m_InterfaceType);
    }


    m_Sender->SetVirtualPadMainColor(m_ControllerId, Color2IntValue(m_MainColor));
    m_Sender->SetVirtualPadSubColor(m_ControllerId, Color2IntValue(m_SubColor));
    m_Sender->SetVirtualPadConnection(m_ControllerId, 1);
}

void VirtualPadInputer::EndAutoPilot()
{
    m_Sender->SetVirtualPadConnection(m_ControllerId, 0);
}

void VirtualPadInputer::SetDefault()
{
    m_Sender->SetVirtualPadDefault(m_ControllerId);
}

void VirtualPadInputer::SetState(const Button buttonList[], int buttonCount,
    const Stick stickList[], const ControllerStickValueType stickXList[], const ControllerStickValueType stickYList[], int stickCount)
{
    ControllerButtonValueType buttons = static_cast<ControllerButtonValueType>(0);
    for (int i = 0; i < buttonCount; i++)
    {
        CheckButton(buttonList[i]);
        buttons |= Button2Flag(buttonList[i]);
    }
    for (int i = 0; i < stickCount; i++)
    {
        CheckStick(stickList[i]);
    }

    m_Sender->SetVirtualPadState(m_ControllerId, buttons, stickList, stickXList, stickYList, stickCount);
}

void VirtualPadInputer::Press(Button button)
{
    CheckButton(button);
    m_Sender->SetVirtualPadPressButton(m_ControllerId, Button2Flag(button));
}
void VirtualPadInputer::Release(Button button)
{
    CheckButton(button);
    m_Sender->SetVirtualPadReleaseButton(m_ControllerId, Button2Flag(button));
}

void VirtualPadInputer::SetStickPos(Stick stick, ControllerStickValueType x, ControllerStickValueType y)
{
    CheckStick(stick);
    switch (stick)
    {
        case Stick::Stick_L:
            SetLStickPos(x, y);
            break;
        case Stick::Stick_R:
            SetRStickPos(x, y);
            break;
        default:
            break;
    }
}

void VirtualPadInputer::SetLStickPos(ControllerStickValueType x, ControllerStickValueType y)
{
    CheckStick(Stick::Stick_L);
    m_Sender->SetVirtualPadStickL(m_ControllerId, x, y);
}
void VirtualPadInputer::SetRStickPos(ControllerStickValueType x, ControllerStickValueType y)
{
    CheckStick(Stick::Stick_R);
    m_Sender->SetVirtualPadStickR(m_ControllerId, x, y);
}
ControllerButtonValueType VirtualPadInputer::Button2Flag(Button button)
{
    switch (button)
    {
        case Button::Button_Capture:
            return HidShellAbstractedPadButton().Shot;
        case Button::Button_Home:
            return HidShellAbstractedPadButton().Home;
        default:
            return (static_cast<ControllerButtonValueType>(1) << button);
    }
}
int VirtualPadInputer::Color2IntValue(ControllerColor color)
{
    int colorValue = 0;
    colorValue |= (color.red   << 16);
    colorValue |= (color.green << 8);
    colorValue |= (color.blue);
    return colorValue;
}

void VirtualPadInputer::CheckStick(Stick type)
{
    if (type < 0 || type >= Stick::Stick_MaxCount)
    {
        throw InvalidStickException(type);
    }
    switch (m_DeviceType)
    {
        case ControllerDeviceType::ControllerDeviceType_SwitchProController:
            break;

        case ControllerDeviceType::ControllerDeviceType_JoyConLeft:
            if (type != Stick::Stick_L)
            {
                throw InvalidStickException(type);
            }
            break;

        case ControllerDeviceType::ControllerDeviceType_JoyConRight:
            if (type != Stick::Stick_R)
            {
                throw InvalidStickException(type);
            }
            break;
        default:
            break;
    }
}

void VirtualPadInputer::CheckButton(Button button)
{
    // HOMEボタン(-2), CAPTUREボタン(-1)
    if (button < -2 || button >= Button::Button_MaxCount)
    {
        throw InvalidButtonException(button);
    }
    switch (m_DeviceType)
    {
        case ControllerDeviceType::ControllerDeviceType_SwitchProController:
            if (button == Button::Button_SL
                || button == Button::Button_SR)
            {
                throw InvalidButtonException(button);
            }
            break;
        case ControllerDeviceType::ControllerDeviceType_JoyConLeft:
            if (button == Button::Button_StickR
                || button == Button::Button_Plus
                || button == Button::Button_Home
                || button == Button::Button_R
                || button == Button::Button_ZR
                || button == Button::Button_A
                || button == Button::Button_B
                || button == Button::Button_Y
                || button == Button::Button_X)
            {
                throw InvalidButtonException(button);
            }

            if (m_InterfaceType == ControllerInterfaceType::ControllerInterfaceType_Rail &&
                (button == Button::Button_SL
                || button == Button::Button_SR))
            {
                throw InvalidButtonException(button);
            }
            break;
        case ControllerDeviceType::ControllerDeviceType_JoyConRight:
            if (button == Button::Button_StickL
                || button == Button::Button_Minus
                || button == Button::Button_Capture
                || button == Button::Button_L
                || button == Button::Button_ZL
                || button == Button::Button_Left
                || button == Button::Button_Up
                || button == Button::Button_Right
                || button == Button::Button_Down)
            {
                throw InvalidButtonException(button);
            }
            if (m_InterfaceType == ControllerInterfaceType::ControllerInterfaceType_Rail &&
                (button == Button::Button_SL
                || button == Button::Button_SR))
            {
                throw InvalidButtonException(button);
            }
            break;
        default:
            break;
    }
}
