﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <chrono>
#include "Common\autoTestAssistTool_StringConverter.h"
#include "PythonCodeExporter.h"

#define RAD_TO_DEG(rad) (((rad)/2.0f/3.141592f)*360.0f)
#define CODE_HEADER    "# -*- coding: utf-8 -*-\n\
from __future__ import division, print_function, absolute_import, unicode_literals\n\
\n\
import sys\n\
import time\n\
\n\
from HACDllDriver import factory, Button, CaptureMode, ControllerDeviceType, ControllerInterfaceType\n\
\n\
if __name__ == \"__main__\":\n\
"
namespace
{
    const int       BaseIndentLevel = 2;
    const int       IndentSpaceCount = 4;
    const char*     DebugPadIdVariableName = "debug_pad_id";
    const char*     SerialNumberVariableName = "serial_number";

    bool GetDeviceType(ControllerDeviceType *pOutValue, uint8_t deviceType)
    {
        switch (deviceType)
        {
        case AbstractedPadDeviceType_SwitchProController:
            (*pOutValue) = ControllerDeviceType_SwitchProController;
            return true;

        case AbstractedPadDeviceType_HandheldJoyLeft:
        case AbstractedPadDeviceType_JoyConLeft:
            (*pOutValue) = ControllerDeviceType_JoyConLeft;
            return true;

        case AbstractedPadDeviceType_HandheldJoyRight:
        case AbstractedPadDeviceType_JoyConRight:
            (*pOutValue) = ControllerDeviceType_JoyConRight;
            return true;

        default:
            ERRMSG("GetDeviceType: Unknown DeviceType " + std::to_string(deviceType));
            return false;
        }
    }

    bool GetInterfaceType(ControllerInterfaceType *pOutValue, uint8_t interfaceType)
    {
        switch (interfaceType)
        {
        case AbstractedPadInterfaceType_Bluetooth:
        case AbstractedPadInterfaceType_Usb: // Usb も Bluetooth として扱う
            (*pOutValue) = ControllerInterfaceType_Bluetooth;
            return true;
        case AbstractedPadInterfaceType_Rail:
            (*pOutValue) = ControllerInterfaceType_Rail;
            return true;
        default:
            ERRMSG("GetIntefaceType: Unknown InterfaceType " + std::to_string(interfaceType));
            return false;
        }
    }
}

PythonCodeExporter::PythonCodeExporter()
    : m_IsUpdateXpadConnection(), m_IsUpdateXpadState(), m_XpadState(),
    m_IsUpdateDebugPadConnection(false), m_IsUpdateDebugPadState(), m_DebugPadState(),
    m_IsUpdateTouchScreenState(false), m_TouchState(),
    m_IsUpdateHomeButtonState(false), m_HomeButtonState(false),
    m_IsUpdateCaptureButtonState(false), m_CaptureButtonState(false),
    m_IsUpdateVirtualPadConnection(), m_IsUpdateVirtualPadState(), m_VirtualPadState()
{
    // ボタン値マップ生成
    m_XpadButtonMap.clear();
    HidShellBasicXpadButton xPadButton;
    m_XpadButtonMap[ xPadButton.A ]             = "A";
    m_XpadButtonMap[ xPadButton.B ]             = "B";
    m_XpadButtonMap[ xPadButton.X ]             = "X";
    m_XpadButtonMap[ xPadButton.Y ]             = "Y";
    m_XpadButtonMap[ xPadButton.StickL]         = "STICK_L";
    m_XpadButtonMap[ xPadButton.StickR]         = "STICK_R";
    m_XpadButtonMap[ xPadButton.L]              = "L";
    m_XpadButtonMap[ xPadButton.R]              = "R";
    m_XpadButtonMap[ xPadButton.ZL]             = "ZL";
    m_XpadButtonMap[ xPadButton.ZR]             = "ZR";
    m_XpadButtonMap[ xPadButton.Start]          = "START";
    m_XpadButtonMap[ xPadButton.Select]         = "SELECT";
    m_XpadButtonMap[ xPadButton.Left]           = "LEFT";
    m_XpadButtonMap[ xPadButton.Up]             = "UP";
    m_XpadButtonMap[ xPadButton.Right]          = "RIGHT";
    m_XpadButtonMap[ xPadButton.Down]           = "DOWN";

    m_DebugPadButtonMap.clear();
    HidShellDebugPadButton debugPadButton;
    m_DebugPadButtonMap[ debugPadButton.A ]     = "A";
    m_DebugPadButtonMap[ debugPadButton.B ]     = "B";
    m_DebugPadButtonMap[ debugPadButton.X ]     = "X";
    m_DebugPadButtonMap[ debugPadButton.Y ]     = "Y";
    m_DebugPadButtonMap[ debugPadButton.L ]     = "L";
    m_DebugPadButtonMap[ debugPadButton.R ]     = "R";
    m_DebugPadButtonMap[ debugPadButton.ZL ]    = "ZL";
    m_DebugPadButtonMap[ debugPadButton.ZR ]    = "ZR";
    m_DebugPadButtonMap[ debugPadButton.Start ] = "START";
    m_DebugPadButtonMap[ debugPadButton.Select ]= "SELECT";
    m_DebugPadButtonMap[ debugPadButton.Left ]  = "LEFT";
    m_DebugPadButtonMap[ debugPadButton.Up ]    = "UP";
    m_DebugPadButtonMap[ debugPadButton.Right ] = "RIGHT";
    m_DebugPadButtonMap[ debugPadButton.Down ]  = "DOWN";

    m_VirtualPadButtonMap.clear();
    HidShellAbstractedPadButton virtualPadButton;
    m_VirtualPadButtonMap[virtualPadButton.A] = "A";
    m_VirtualPadButtonMap[virtualPadButton.B] = "B";
    m_VirtualPadButtonMap[virtualPadButton.X] = "X";
    m_VirtualPadButtonMap[virtualPadButton.Y] = "Y";
    m_VirtualPadButtonMap[virtualPadButton.StickL] = "STICK_L";
    m_VirtualPadButtonMap[virtualPadButton.StickR] = "STICK_R";
    m_VirtualPadButtonMap[virtualPadButton.L] = "L";
    m_VirtualPadButtonMap[virtualPadButton.R] = "R";
    m_VirtualPadButtonMap[virtualPadButton.ZL] = "ZL";
    m_VirtualPadButtonMap[virtualPadButton.ZR] = "ZR";
    m_VirtualPadButtonMap[virtualPadButton.Start] = "START";
    m_VirtualPadButtonMap[virtualPadButton.Select] = "SELECT";
    m_VirtualPadButtonMap[virtualPadButton.Left] = "LEFT";
    m_VirtualPadButtonMap[virtualPadButton.Up] = "UP";
    m_VirtualPadButtonMap[virtualPadButton.Right] = "RIGHT";
    m_VirtualPadButtonMap[virtualPadButton.Down] = "DOWN";
    m_VirtualPadButtonMap[virtualPadButton.SL] = "SL";
    m_VirtualPadButtonMap[virtualPadButton.SR] = "SR";
    m_VirtualPadButtonMap[virtualPadButton.Home] = "HOME";
    m_VirtualPadButtonMap[virtualPadButton.Shot] = "CAPTURE";
}

PythonCodeExporter::~PythonCodeExporter()
{
}

HidInputerResult PythonCodeExporter::ExportHidInputRecords(const char* exportFileName, const std::vector<HidTaskRecord>& taskList)
{
    if (exportFileName == NULL)
    {
        return HidInputerResult::HidInputerResult_Unexpected;
    }

    std::string filename(exportFileName);

    FILE *fp;
    if (_wfopen_s(&fp, autoTestAssistTool::util::cp_to_wide(filename, CP_UTF8).c_str(), L"w") != 0)
    {
        // ファイルオープン失敗
        return HidInputerResult::HidInputerResult_Unexpected;
    }

    ExportScriptHeader(fp);

    if (taskList.empty())
    {
        // 空の場合は pass を出力
        ExportIndent(fp, BaseIndentLevel);
        fprintf(fp, "pass\n");
    }
    else
    {
        ExportScriptBody(fp, BaseIndentLevel, taskList);
    }

    fclose(fp);
    return HidInputerResult::HidInputerResult_Success;
}

void PythonCodeExporter::ExportIndent(FILE* fp, int indentLevel)
{
    for (int i = 0; i < indentLevel * IndentSpaceCount; i++)
    {
        fputc(' ', fp);
    }
}

void PythonCodeExporter::ExportSleep(FILE* fp, int indentLevel, int sleepSpan)
{
    ExportIndent(fp, indentLevel);
    fprintf( fp, "time.sleep( %f )\n", sleepSpan / 1000.0f);
}


void PythonCodeExporter::ExportScriptHeader(FILE*fp)
{
    // 初期化情報出力
    fprintf(fp, CODE_HEADER);

    ExportIndent(fp, 1);
    fprintf(fp, "# sys.argv[1] = Target serial number.\n");
    ExportIndent(fp, 1);
    fprintf(fp, "%s = None\n", SerialNumberVariableName);
    ExportIndent(fp, 1);
    fprintf(fp, "if len(sys.argv) != 1:\n");
    ExportIndent(fp, 2);
    fprintf(fp, "%s = sys.argv[1]\n", SerialNumberVariableName);

    ExportIndent(fp, 1);

    fprintf(fp, "with factory(target_serial=%s, capture_device_id=None, capture_mode=CaptureMode.CaptureMode_720p) as driver:\n\n", SerialNumberVariableName);
    ExportIndent(fp, 2);
    fprintf(fp, "for i in driver.controllers.get_active_cont_id_list() :\n");
    ExportIndent(fp, 3);
    fprintf(fp, "driver.controllers.delete(i)\n\n");
}

void PythonCodeExporter::ExportScriptBody(FILE*fp, int indentLevel, const std::vector<HidTaskRecord>& taskList)
{
    for (auto it = taskList.begin(); it != taskList.end(); it++)
    {
        switch (it->taskType)
        {
            case HidTaskType_SetXpadConnection:
                m_IsUpdateXpadConnection[it->id] = true;
                m_XpadState[it->id].powerState = static_cast<HidShellXpadPowerState>(it->value);
                break;
            case HidTaskType_SetDebugPadConnection:
                m_IsUpdateDebugPadConnection = true;
                m_DebugPadState.attributes = it->value;
                break;
            case HidTaskType_SetVirtualPadConnection:
                m_VirtualPadState[it->id].attribute = it->value;
                m_IsUpdateVirtualPadConnection[it->id] = true;
                break;
            case HidTaskType_SetXpadPressButton:
            case HidTaskType_SetXpadReleaseButton:
            case HidTaskType_SetXpadStickLX:
            case HidTaskType_SetXpadStickRX:
                UpdateXpadState(m_XpadState[it->id], it);
                m_IsUpdateXpadState[it->id] = true;
                break;
            case HidTaskType_SetDebugPadPressButton:
            case HidTaskType_SetDebugPadReleaseButton:
            case HidTaskType_SetDebugPadStickLX:
            case HidTaskType_SetDebugPadStickRX:
                UpdateDebugPadState(m_DebugPadState, it);
                m_IsUpdateDebugPadState = true;
                break;
            case HidTaskType_SetFingerId:
            case HidTaskType_SetTouchState:
            case HidTaskType_MoveTouchPoint_X:
                UpdateTouchState(m_TouchState, it);
                m_IsUpdateTouchScreenState = true;
                break;
            case HidTaskType_SetHomeButton:
                m_HomeButtonState = (it->value != 0);
                break;
            case HidTaskType_SetCaptureButton:
                m_CaptureButtonState = (it->value != 0);
                break;
            case HidTaskType_SetVirtualPadPressButton:
            case HidTaskType_SetVirtualPadReleaseButton:
            case HidTaskType_SetVirtualPadStickLX:
            case HidTaskType_SetVirtualPadStickRX:
            case HidTaskType_SetVirtualPadDeviceType:
            case HidTaskType_SetVirtualPadInterfaceType:
            case HidTaskType_SetVirtualPadMainColor:
            case HidTaskType_SetVirtualPadSubColor:
                UpdateVirtualPadState(m_VirtualPadState[it->id], it);
                m_IsUpdateVirtualPadState[it->id] = true;
                break;
            default:
                // TIPS: システムの不具合のため、ユーザー側で対処できないので、
                // 特に処理せずデバッグメッセージのみを出力します。
                ERRMSG("PythonCodeExporter::ExportScriptBody: " + UNEXPECTED_HID_TASK_EXPORT_MESSAGE + std::to_string(it->taskType));
                break;
        }

        // スリープ時間チェック
        bool isTaskListEnd = ((it + 1) == taskList.end());
        unsigned int sleepSpan = 0;
        if (!isTaskListEnd)
        {
            sleepSpan = static_cast<unsigned int>(
                std::chrono::duration_cast<std::chrono::milliseconds>(
                    (it + 1)->timestamp - it->timestamp).count());
        }

        // 同時刻の操作がある場合は次のタスクを取得する
        if (sleepSpan == 0 && !isTaskListEnd)
        {
            continue;
        }

        // スクリプト出力
        ExportHidControl(fp, indentLevel);

        // スリープ出力
        if (sleepSpan > 0)
        {
            ExportSleep(fp, indentLevel, sleepSpan);
        }
    }
}

void PythonCodeExporter::ExportHidControl(FILE*fp, int indentLevel)
{
    // xpad スクリプト出力
    for (int i = 0; i < MaxControllerCount; i++)
    {
        // 接続状態出力
        // TIPS: 接続タスクは先にスクリプト出力する
        if (m_IsUpdateXpadConnection[i] && m_XpadState[i].powerState != HidShellXpadPowerState::Disconnected)
        {
            ExportXpadConnectionRecord(fp, indentLevel, std::to_string(i).c_str(), true);
        }

        // ステータス出力
        if (m_IsUpdateXpadState[i])
        {
            ExportXpadState(fp, indentLevel, i, m_XpadState[i]);

            m_IsUpdateXpadState[i] = false;
        }

        // 接続状態出力
        // TIPS: 切断タスクは先にスクリプト出力する
        if (m_IsUpdateXpadConnection[i] && m_XpadState[i].powerState == HidShellXpadPowerState::Disconnected)
        {
            ExportXpadConnectionRecord(fp, indentLevel, std::to_string(i).c_str(), false);
        }
        m_IsUpdateXpadConnection[i] = false;
    }
    // Debug Pad スクリプト出力
    // 接続状態出力
    // TIPS: 接続タスクは先にスクリプト出力する
    if (m_IsUpdateDebugPadConnection && m_DebugPadState.attributes != 0)
    {
        ExportDebugPadConnectionRecord(fp, indentLevel, true);
    }

    if (m_IsUpdateDebugPadState)
    {
        ExportDebugPadState(fp, indentLevel, m_DebugPadState);
        m_IsUpdateDebugPadState = false;
    }

    // 接続状態出力
    // TIPS: 切断タスクは先にスクリプト出力する
    if (m_IsUpdateDebugPadConnection && m_DebugPadState.attributes == 0)
    {
        ExportDebugPadConnectionRecord(fp, indentLevel, false);
    }
    m_IsUpdateDebugPadConnection = false;


    // touch screen スクリプト出力
    if (m_IsUpdateTouchScreenState)
    {
        ExportTouchState(fp, indentLevel, m_TouchState);
        m_IsUpdateTouchScreenState = false;
    }

    if (m_IsUpdateHomeButtonState)
    {
        ExportHomeButtonRecord(fp, indentLevel, m_HomeButtonState);
        m_IsUpdateHomeButtonState = false;
    }

    if (m_IsUpdateCaptureButtonState)
    {
        ExportCaptureButtonRecord(fp, indentLevel, m_CaptureButtonState);
        m_IsUpdateCaptureButtonState = false;
    }

    // VirutalPad スクリプト出力
    const uint32_t connectedAtrribute = 1u;
    for (int i = 0; i < MaxControllerCount; i++)
    {
        // 接続状態出力
        // TIPS: 接続タスクは先にスクリプト出力する
        if (m_IsUpdateVirtualPadConnection[i] && (m_VirtualPadState[i].attribute & connectedAtrribute))
        {
            ExportVirtualPadConnectionRecord(fp, indentLevel, i, true, m_VirtualPadState[i]);
        }

        // ステータス出力
        if (m_IsUpdateVirtualPadState[i] && (m_VirtualPadState[i].attribute & connectedAtrribute))
        {
            ExportVirtualPadState(fp, indentLevel, i, m_VirtualPadState[i]);

            m_IsUpdateVirtualPadState[i] = false;
        }

        // 接続状態出力
        // TIPS: 切断タスクは最後にスクリプト出力する
        if (m_IsUpdateVirtualPadConnection[i] && !(m_VirtualPadState[i].attribute & connectedAtrribute))
        {
            ExportVirtualPadConnectionRecord(fp, indentLevel, i, false, m_VirtualPadState[i]);
        }
        m_IsUpdateVirtualPadConnection[i] = false;
    }
}

void PythonCodeExporter::UpdateXpadState(HidShellBasicXpadState &state, std::vector<HidTaskRecord>::const_iterator& task)
{
    switch (task->taskType)
    {
        case HidTaskType_SetXpadPressButton:
            state.buttons |= task->value;
            break;
        case HidTaskType_SetXpadReleaseButton:
            state.buttons &= ~(task->value);
            break;
        case HidTaskType_SetXpadStickLX:
            assert((task + 1)->taskType == HidTaskType_SetXpadStickLY);
            state.stickL.x = (task + 0)->value;
            state.stickL.y = (task + 1)->value;
            task++;
            break;
        case HidTaskType_SetXpadStickRX:
            assert((task + 1)->taskType == HidTaskType_SetXpadStickRY);
            state.stickR.x = (task + 0)->value;
            state.stickR.y = (task + 1)->value;
            task++;
            break;
        default:
            ERRMSG("PythonCodeExporter::UpdateXpadState: " + UNEXPECTED_HID_TASK_EXCUTED_MESSAGE + " taskType=" + std::to_string(task->taskType));
            break;
    }
}

void PythonCodeExporter::ExportXpadState(FILE* fp, int indentLevel, int id, const HidShellBasicXpadState &state)
{
    // ボタン情報
    int buttonCount = 0;
    std::string buttonString;

    for (auto it = m_XpadButtonMap.begin(); it != m_XpadButtonMap.end(); it++)
    {
        if (state.buttons & it->first)
        {
            if (buttonCount > 0)
            {
                buttonString = buttonString + ",";
            }
            buttonString = buttonString + "\"" + it->second.c_str() + "\"";
            buttonCount++;
        }
    }

    // スティック情報
    float lStickRadian  = static_cast<float>(atan2(state.stickL.y, state.stickL.x));
    float lStickPower   = static_cast<float>(std::sqrt((state.stickL.y * state.stickL.y) + (state.stickL.x * state.stickL.x)) / sqrt(std::pow(cos(lStickRadian) * DefaultStickRadius, 2.0f) + std::pow(sin(lStickRadian) * DefaultStickRadius, 2.0f)));
    if (lStickPower > 1.0f)
    {
        lStickPower = 1.0f;
    }

    float rStickRadian  = static_cast<float>(atan2(state.stickR.y, state.stickR.x));
    float rStickPower   = static_cast<float>(std::sqrt((state.stickR.y * state.stickR.y) + (state.stickR.x * state.stickR.x)) / sqrt(std::pow(cos(rStickRadian) * DefaultStickRadius, 2.0f) + std::pow(sin(rStickRadian) * DefaultStickRadius, 2.0f)));
    if (rStickPower > 1.0f)
    {
        rStickPower = 1.0f;
    }

    ExportIndent(fp, indentLevel);
    fprintf(fp, "driver.controllers[%d].set_controller_state([%s], %f, %f, %f, %f, ms_before=0, ms_after=0)\n",
                                            id,
                                            buttonString.c_str(),
                                             -static_cast<float>(RAD_TO_DEG(lStickRadian)),
                                            lStickPower,
                                             -static_cast<float>(RAD_TO_DEG(rStickRadian)),
                                            rStickPower );
}



void PythonCodeExporter::UpdateDebugPadState(HidShellDebugPadState &state, std::vector<HidTaskRecord>::const_iterator& task)
{
    switch (task->taskType)
    {
        case HidTaskType_SetDebugPadPressButton:
            state.buttons |= task->value;
            break;
        case HidTaskType_SetDebugPadReleaseButton:
            state.buttons &= ~(task->value);
            break;
        case HidTaskType_SetDebugPadStickLX:
            assert((task + 1)->taskType == HidTaskType_SetDebugPadStickLY);
            state.stickL.x = (task + 0)->value;
            state.stickL.y = (task + 1)->value;
            task++;
            break;
        case HidTaskType_SetDebugPadStickRX:
            assert((task + 1)->taskType == HidTaskType_SetDebugPadStickRY);
            state.stickR.x = (task + 0)->value;
            state.stickR.y = (task + 1)->value;
            task++;
            break;
        default:
            ERRMSG("PythonCodeExporter::UpdateDebugPadState: "+ UNEXPECTED_HID_TASK_EXCUTED_MESSAGE + "taskType=" + std::to_string(task->taskType));
            break;
    }
}

void PythonCodeExporter::ExportDebugPadState(FILE* fp, int indentLevel, const HidShellDebugPadState& state)
{
    int buttonCount = 0;
    std::string buttonString;

    for (auto it = m_DebugPadButtonMap.begin(); it != m_DebugPadButtonMap.end(); it++)
    {
        if (state.buttons & it->first)
        {
            if (buttonCount > 0)
            {
                buttonString = buttonString + ",";
            }
            buttonString = buttonString + "\"" + it->second.c_str() + "\"";
            buttonCount++;
        }
    }

    // スティック情報
    float lStickRadian  = static_cast<float>(atan2(state.stickL.y, state.stickL.x));
    float lStickPower   = static_cast<float>(std::sqrt((state.stickL.y * state.stickL.y) + (state.stickL.x * state.stickL.x)) / sqrt(std::pow(cos(lStickRadian) * DefaultStickRadius, 2.0f) + std::pow(sin(lStickRadian) * DefaultStickRadius, 2.0f)));
    if (lStickPower > 1.0f)
    {
        lStickPower = 1.0f;
    }

    float rStickRadian  = static_cast<float>(atan2(state.stickR.y, state.stickR.x));
    float rStickPower   = static_cast<float>(std::sqrt((state.stickR.y * state.stickR.y) + (state.stickR.x * state.stickR.x)) / sqrt(std::pow(cos(rStickRadian) * DefaultStickRadius, 2.0f) + std::pow(sin(rStickRadian) * DefaultStickRadius, 2.0f)));
    if (rStickPower > 1.0f)
    {
        rStickPower = 1.0f;
    }

    ExportIndent(fp, indentLevel);
    fprintf(fp, "driver.controllers[%s].set_controller_state([%s], %f, %f, %f, %f, ms_before=0, ms_after=0)\n",
                                            DebugPadIdVariableName,
                                            buttonString.c_str(),
                                             -static_cast<float>(RAD_TO_DEG(lStickRadian)),
                                            lStickPower,
                                             -static_cast<float>(RAD_TO_DEG(rStickRadian)),
                                            rStickPower );

}

void PythonCodeExporter::UpdateTouchState(HidShellTouchScreenState &state, std::vector<HidTaskRecord>::const_iterator& task)
{
    int32_t x;
    int32_t y;
    switch (task->taskType)
    {
        case HidTaskType_SetFingerId:
            state.touches[task->id].fingerId = task->value;
            break;
        case HidTaskType_SetTouchState:
            if (task->value)
            {
                assert((task + 1)->taskType == HidTaskType_MoveTouchPoint_X);
                assert((task + 2)->taskType == HidTaskType_MoveTouchPoint_Y);
                x = (task + 1)->value;
                y = (task + 2)->value;

                state.touches[task->id].attributes = 1;
                state.touches[task->id].x = x;
                state.touches[task->id].y = y;
                task = task + 2;
            }
            else
            {
                state.touches[task->id].attributes = 0;
            }
            break;
        case HidTaskType_MoveTouchPoint_X:
            assert((task + 1)->taskType == HidTaskType_MoveTouchPoint_Y);
            x = (task + 0)->value;
            y = (task + 1)->value;
            state.touches[task->id].x = x;
            state.touches[task->id].y = y;
            task++;
            break;
        default:
            ERRMSG("PythonCodeExporter::UpdateTouchState: " + UNEXPECTED_HID_TASK_EXCUTED_MESSAGE + " taskType=" + std::to_string(task->taskType));
            break;

    }
}

void PythonCodeExporter::ExportTouchState(FILE* fp, int indentLevel, const HidShellTouchScreenState& state)
{
    HidShellTouchAttribute touchAttribute;

    int touchCount = 0;
    std::string touchIdList = "";
    std::string fingerIdList = "";
    std::string positionList = "";
    for (int i = 0; i < MaxTouchCount; i++)
    {
        if (state.touches[i].attributes == touchAttribute.Start)
        {
            if (touchCount > 0)
            {
                touchIdList = touchIdList + ",";
                fingerIdList = fingerIdList + ",";
                positionList = positionList + ",";
            }
            touchIdList = touchIdList + std::to_string(i);
            fingerIdList = fingerIdList + std::to_string(state.touches[i].fingerId);
            positionList = positionList + "[" + std::to_string(state.touches[i].x) + "," + std::to_string(state.touches[i].y) + "]";

            touchCount = touchCount + 1;
        }
    }

    ExportIndent(fp, indentLevel);
    fprintf(fp, "driver.screen.set_touch_state( [%s], [%s], [%s], ms_before=0, ms_after=0 )\n", touchIdList.c_str(), fingerIdList.c_str(), positionList.c_str());
}

void PythonCodeExporter::ExportXpadConnectionRecord(FILE* fp, int indentLevel, const char* id, bool isConnect)
{
    ExportIndent(fp, indentLevel);
    if (isConnect)
    {
        fprintf( fp,"driver.controllers.add(%s)\n", id);
    }
    else
    {
        fprintf( fp,"driver.controllers.delete(%s)\n", id);
    }
}


void PythonCodeExporter::ExportDebugPadConnectionRecord(FILE* fp, int indentLevel, bool isConnect)
{
    ExportIndent(fp, indentLevel);

    if (isConnect)
    {
        fprintf(fp, "%s = driver.controllers.add_debug_pad()\n", DebugPadIdVariableName);
    }
    else
    {
        fprintf(fp, "driver.controllers.delete(%s)\n", DebugPadIdVariableName);
    }
}



void PythonCodeExporter::ExportHomeButtonRecord(FILE* fp, int indentLevel, bool value)
{
    ExportIndent(fp, indentLevel);
    if (value)
    {
        fprintf(fp, "driver.controllers[0].press(\"HOME\")");
    }
    else
    {
        fprintf(fp, "driver.controllers[0].release(\"HOME\")");
    }
}
void PythonCodeExporter::ExportCaptureButtonRecord(FILE* fp, int indentLevel, bool value)
{
    ExportIndent(fp, indentLevel);
    if (value)
    {
        fprintf(fp, "driver.controllers[0].press(\"CAPTURE\")");
    }
    else
    {
        fprintf(fp, "driver.controllers[0].release(\"CAPTURE\")");
    }
}

void PythonCodeExporter::ExportVirtualPadConnectionRecord(FILE* fp, int indentLevel, int id, bool isConnect, HidShellAbstractedPadState &state)
{
    if (isConnect)
    {
        ExportIndent(fp, indentLevel);
        ControllerDeviceType deviceType;
        ControllerInterfaceType interfaceType;
        if (GetDeviceType(&deviceType, state.deviceType)
            && GetInterfaceType(&interfaceType, state.interfaceType))
        {
            fprintf(fp, "controller_id%d = driver.controllers.add_controller(%d, %d, (%d, %d, %d), (%d, %d, %d))\n",
                id,
                deviceType, interfaceType,
                state.color.main[0], state.color.main[1], state.color.main[2],
                state.color.sub[0], state.color.sub[1], state.color.sub[2]);
        }
        else
        {
            fprintf(fp, "controller_id%d = driver.controllers.add_controller(None, None, (%d, %d, %d), (%d, %d, %d))\n",
                id,
                state.color.main[0], state.color.main[1], state.color.main[2],
                state.color.sub[0], state.color.sub[1], state.color.sub[2]);
        }
    }
    else
    {
        ExportIndent(fp, indentLevel);
        fprintf(fp, "driver.controllers.delete(controller_id%d)\n", id);
    }
}

void PythonCodeExporter::UpdateVirtualPadState(HidShellAbstractedPadState &state, std::vector<HidTaskRecord>::const_iterator& task)
{
    switch (task->taskType)
    {
    case HidTaskType_SetVirtualPadPressButton:
        state.buttons |= task->value;
        break;
    case HidTaskType_SetVirtualPadReleaseButton:
        state.buttons &= ~(task->value);
        break;
    case HidTaskType_SetVirtualPadStickLX:
        assert((task + 1)->taskType == HidTaskType_SetVirtualPadStickLY);
        state.stickL.x = (task + 0)->value;
        state.stickL.y = (task + 1)->value;
        task++;
        break;
    case HidTaskType_SetVirtualPadStickRX:
        assert((task + 1)->taskType == HidTaskType_SetVirtualPadStickRY);
        state.stickR.x = (task + 0)->value;
        state.stickR.y = (task + 1)->value;
        task++;
        break;
    case HidTaskType_SetVirtualPadDeviceType:
        state.deviceType = task->value;
        break;
    case HidTaskType_SetVirtualPadInterfaceType:
        state.interfaceType = task->value;
        break;
    case HidTaskType_SetVirtualPadMainColor:
        state.color.main[0] = task->value & 0xFF;
        state.color.main[1] = (task->value >> 8) & 0xFF;
        state.color.main[2] = (task->value >> 16) & 0xFF;
        state.color.main[3] = (task->value >> 24) & 0xFF;
        break;
    case HidTaskType_SetVirtualPadSubColor:
        state.color.sub[0] = task->value & 0xFF;
        state.color.sub[1] = (task->value >> 8) & 0xFF;
        state.color.sub[2] = (task->value >> 16) & 0xFF;
        state.color.sub[3] = (task->value >> 24) & 0xFF;
        break;
    default:
        ERRMSG("PythonCodeExporter::UpdateVirtualPadState: " + UNEXPECTED_HID_TASK_EXCUTED_MESSAGE + " taskType=" + std::to_string(task->taskType));
        break;
    }
}

void PythonCodeExporter::ExportVirtualPadState(FILE* fp, int indentLevel, int id, const HidShellAbstractedPadState &state)
{
    // ボタン情報
    int buttonCount = 0;
    std::string buttonString;

    for (auto it = m_VirtualPadButtonMap.begin(); it != m_VirtualPadButtonMap.end(); it++)
    {
        if (state.buttons & it->first)
        {
            if (buttonCount > 0)
            {
                buttonString = buttonString + ",";
            }
            buttonString = buttonString + "\"" + it->second.c_str() + "\"";
            buttonCount++;
        }
    }

    // スティック情報
    float lStickRadian = static_cast<float>(atan2(state.stickL.y, state.stickL.x));
    float lStickPower = static_cast<float>(std::sqrt((state.stickL.y * state.stickL.y) + (state.stickL.x * state.stickL.x)) / sqrt(std::pow(cos(lStickRadian) * DefaultStickRadius, 2.0f) + std::pow(sin(lStickRadian) * DefaultStickRadius, 2.0f)));
    if (lStickPower > 1.0f)
    {
        lStickPower = 1.0f;
    }

    float rStickRadian = static_cast<float>(atan2(state.stickR.y, state.stickR.x));
    float rStickPower = static_cast<float>(std::sqrt((state.stickR.y * state.stickR.y) + (state.stickR.x * state.stickR.x)) / sqrt(std::pow(cos(rStickRadian) * DefaultStickRadius, 2.0f) + std::pow(sin(rStickRadian) * DefaultStickRadius, 2.0f)));
    if (rStickPower > 1.0f)
    {
        rStickPower = 1.0f;
    }

    if ((state.deviceType == AbstractedPadDeviceType_HandheldJoyLeft)
        || (state.deviceType == AbstractedPadDeviceType_JoyConLeft))
    {
        ExportIndent(fp, indentLevel);
        fprintf(fp, "driver.controllers[controller_id%d].set_controller_state([%s], %f, %f, None, None, ms_before=0, ms_after=0)\n",
            id,
            buttonString.c_str(),
            -static_cast<float>(RAD_TO_DEG(lStickRadian)),
            lStickPower);
    }
    if ((state.deviceType == AbstractedPadDeviceType_HandheldJoyRight)
        || (state.deviceType == AbstractedPadDeviceType_JoyConRight))
    {
        ExportIndent(fp, indentLevel);
        fprintf(fp, "driver.controllers[controller_id%d].set_controller_state([%s], None, None, %f, %f, ms_before=0, ms_after=0)\n",
            id,
            buttonString.c_str(),
            -static_cast<float>(RAD_TO_DEG(rStickRadian)),
            rStickPower);
    }
    else if(state.deviceType == AbstractedPadDeviceType_SwitchProController)
    {
        ExportIndent(fp, indentLevel);
        fprintf(fp, "driver.controllers[controller_id%d].set_controller_state([%s], %f, %f, %f, %f, ms_before=0, ms_after=0)\n",
            id,
            buttonString.c_str(),
            -static_cast<float>(RAD_TO_DEG(lStickRadian)),
            lStickPower,
            -static_cast<float>(RAD_TO_DEG(rStickRadian)),
            rStickPower);
    }
}
