﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <set>
#include <unordered_map>

#include "ControllerException.h"
#include "ControllerInterface.h"
#include "IoPortSender.h"

const int ControllerMaxCount = 8;
const int DebugControllerMaxCount = 1;
const int VirtualControllerMaxCount = 8;
const int ControllerTotalCount = ControllerMaxCount + DebugControllerMaxCount + VirtualControllerMaxCount;

class MultiControllerManager
{
private:
    IoPortSender           *m_Sender;

    ControllerInterface* m_ControllerInterfaces[ControllerTotalCount];

    explicit MultiControllerManager(IoPortSender *sender);
    ~MultiControllerManager();
    MultiControllerManager& operator=(const MultiControllerManager &copy);
    MultiControllerManager(const MultiControllerManager &copy);

    void CheckActiveControllerIdRange(int controllerId);
    void CheckActiveControllerId(int controllerId);
    void CheckControllerDeviceInfo(const ControllerDeviceInfo& deviceInfo);

    // コントローラーの初期化
    void InitializeController();
    void InitializeDebugController();
    void InitializeVirtualController();

    ControllerDeviceInfo m_VirtualControllerDeviceInfo[VirtualControllerMaxCount];

public:
    static MultiControllerManager &GetInstance(IoPortSender *sender);

    // コントローラー接続の初期化
    void InitializeControllerConnection();

    ControllerResult AddController(int *pOutControllerId);
    ControllerResult AddController(int controllerId);
    ControllerResult AddDebugController(int *pOutControllerId);
    ControllerResult AddControllerWithDeviceInfo(int *pOutControllerId, const ControllerDeviceInfo& deviceInfo);
    ControllerResult AddControllerWithDeviceInfo(int controllerId, const ControllerDeviceInfo& deviceInfo);
    ControllerResult ReleaseController(int controllerId);

    ControllerResult GetConnectedControllerCount(int *pOutCount);
    ControllerResult GetConnectedControllerIds(int *pOutCount,
                    int pOutIdsArray[], int arrayLength );
    ControllerResult GetConnectedControllerDeviceInfo(ControllerDeviceInfo* pOutDeviceInfo,
                    int controllerId);
    ControllerResult GetConnectedDebugControllerCount(int *pOutCount);
    ControllerResult GetConnectedDebugControllerIds(int *pOutCount,
                    int pOutIdsArray[], int arrayLength );

    ControllerResult ClearAllControllerState();

    ControllerResult SetControllerState(int controllerId, Button buttonList[], int buttonCount, Stick stickList[], float stickDegreeList[], float stickPowerList[], int stickCount);
    ControllerResult ResetControllerState(int controllerId);

    // ボタン操作
    ControllerResult PressButton(int controllerId, Button button);
    ControllerResult ReleaseButton(int controllerId, Button button);
    ControllerResult PushButton(int controllerId, Button button, int waitPress);
    ControllerResult PushMultiSeq(int controllerId, Button buttonList[], int buttonCount, int waitPress, int waitSpan);

    // 複数ボタン操作
    ControllerResult PressMultiButton(int controllerId, Button buttonList[], int buttonCount);
    ControllerResult ReleaseMultiButton(int controllerId, Button buttonList[], int buttonCount);
    ControllerResult PushMultiButton(int controllerId, Button buttonList[], int buttonCount, int waitPress);

    // アナログスティック
    ControllerResult HoldAnalogStick(int controllerId, Stick stick, float degree, float power);
    ControllerResult ReleaseAnalogStick(int controllerId, Stick stick);
    ControllerResult TiltAnalogStick(int controllerId, Stick stick, float degree, float power, int waitPress);
};
