﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include "IoPortRecorder.h"
#include "IoPortSender.h"

IoPortRecorder::IoPortRecorder()
{
    // 初期状態クリア
    // Xpad
    for (int i = 0; i < HidShellBasicXpadCountMax; i++)
    {
        memset(&m_OldXpadState[i], 0, sizeof(HidShellBasicXpadState));
        memset(&m_NewXpadState[i], 0, sizeof(HidShellBasicXpadState));
    }
    // Debug pad
    memset(&m_OldDebugPadState, 0, sizeof(HidShellDebugPadState));
    memset(&m_NewDebugPadState, 0, sizeof(HidShellDebugPadState));
    // Touch screen
    memset(&m_OldTouchScreenState, 0, sizeof(HidShellTouchScreenState));
    memset(&m_NewTouchScreenState, 0, sizeof(HidShellTouchScreenState));
    // Home button
    memset(&m_OldHomeButtonState, 0, sizeof(HidShellHomeButtonState));
    memset(&m_NewHomeButtonState, 0, sizeof(HidShellHomeButtonState));
    // Capture Button
    memset(&m_OldCaptureButtonState, 0, sizeof(HidShellCaptureButtonState));
    memset(&m_NewCpatureButtonState, 0, sizeof(HidShellCaptureButtonState));
    // Abstracted Pad
    for (int i = 0; i < HidShellAbstractedPadCountMax; i++)
    {
        memset(&m_OldAbstractedPadState[i], 0, sizeof(HidShellAbstractedPadState));
        memset(&m_NewAbstractedPadState[i], 0, sizeof(HidShellAbstractedPadState));
    }
}

IoPortRecorder::~IoPortRecorder()
{
    ClearHidRecord();
}

void IoPortRecorder::StoreHidInput(HidShellHandle handle, uint32_t port, int32_t direction)
{
    std::chrono::system_clock::time_point timestamp = std::chrono::system_clock::now();

    StoreAbstractedPadInput(timestamp, handle, port, direction);
    StoreXPadInput( timestamp, handle, port, direction);
    StoreDebugPadInput(timestamp, handle, port, direction);
    StoreTouchScreenInput(timestamp, handle, port, direction);
}

void IoPortRecorder::ClearHidRecord()
{
    m_TaskList.clear();
}

const std::vector<HidTaskRecord>& IoPortRecorder::GetRecords()
{
    return m_TaskList;
}

void IoPortRecorder::SetTask( const std::chrono::system_clock::time_point &time, int32_t id, int32_t value, HidTaskType taskType )
{
    HidTaskRecord task = { time, id, value, taskType };
    m_TaskList.push_back(task);
}

void IoPortRecorder::StoreAbstractedPadInput(
    const std::chrono::system_clock::time_point &time, HidShellHandle handle,
    uint32_t port, int32_t direction)
{
    HidShellLibraryWrapper &wrapper = HidShellLibraryWrapper::GetInstance();
    for (int i = 0; i < HidShellAbstractedPadCountMax; ++i)
    {
        wrapper.m_GetHidShellAbstractedPadState(reinterpret_cast<void*>(handle),
            &m_NewAbstractedPadState[i], i, port, direction);

        // 接続状態チェック
        // TIPS: 接続タスクは、最初にスタックへ追加
        uint32_t ConnectedAttribute = 0x1u;
        if (m_OldAbstractedPadState[i].attribute != m_NewAbstractedPadState[i].attribute)
        {
            if ((m_NewAbstractedPadState[i].attribute & ConnectedAttribute) != 0)
            {
                auto color = m_NewAbstractedPadState[i].color;
                uint32_t mainColor = color.main[0] + (color.main[1] << 8)
                    + (color.main[2] << 16) + (color.main[3] << 24);
                uint32_t subColor = color.sub[0] + (color.sub[1] << 8)
                    + (color.sub[2] << 16) + (color.sub[3] << 24);

                SetTask(time, i, m_NewAbstractedPadState[i].deviceType,
                    HidTaskType::HidTaskType_SetVirtualPadDeviceType);
                SetTask(time, i, m_NewAbstractedPadState[i].interfaceType,
                    HidTaskType::HidTaskType_SetVirtualPadInterfaceType);
                SetTask(time, i, mainColor,
                    HidTaskType::HidTaskType_SetVirtualPadMainColor);
                SetTask(time, i, subColor,
                    HidTaskType::HidTaskType_SetVirtualPadSubColor);
                SetTask(time, i, m_NewAbstractedPadState[i].attribute,
                    HidTaskType::HidTaskType_SetVirtualPadConnection);
            }
        }

        // スティック動作チェック
        if (m_OldAbstractedPadState[i].stickL.x != m_NewAbstractedPadState[i].stickL.x
            || m_OldAbstractedPadState[i].stickL.y != m_NewAbstractedPadState[i].stickL.y)
        {
            // TIPS: X, Y の順番でタスクを登録する必要があります。
            SetTask(time, i, m_NewAbstractedPadState[i].stickL.x,
                HidTaskType::HidTaskType_SetVirtualPadStickLX);
            SetTask(time, i, m_NewAbstractedPadState[i].stickL.y,
                HidTaskType::HidTaskType_SetVirtualPadStickLY);
        }
        if (m_OldAbstractedPadState[i].stickR.x != m_NewAbstractedPadState[i].stickR.x
            || m_OldAbstractedPadState[i].stickR.y != m_NewAbstractedPadState[i].stickR.y)
        {
            // TIPS: X, Y の順番でタスクを登録する必要があります。
            SetTask(time, i, m_NewAbstractedPadState[i].stickR.x,
                HidTaskType::HidTaskType_SetVirtualPadStickRX);
            SetTask(time, i, m_NewAbstractedPadState[i].stickR.y,
                HidTaskType::HidTaskType_SetVirtualPadStickRY);
        }

        // ボタンチェック
        uint32_t pushButtonList =
            m_NewAbstractedPadState[i].buttons & ~(m_OldAbstractedPadState[i].buttons);
        uint32_t releaseButtonList =
            ~(m_NewAbstractedPadState[i].buttons) & m_OldAbstractedPadState[i].buttons;
        if (pushButtonList)
        {
            SetTask(time, i, pushButtonList,
                HidTaskType::HidTaskType_SetVirtualPadPressButton);
        }
        if (releaseButtonList)
        {
            SetTask(time, i, releaseButtonList,
                HidTaskType::HidTaskType_SetVirtualPadReleaseButton);
        }

        // 接続状態チェック
        // TIPS: 切断タスクは、最後にスタックへ追加
        if (m_OldAbstractedPadState[i].attribute != m_NewAbstractedPadState[i].attribute)
        {
            if ((m_NewAbstractedPadState[i].attribute & ConnectedAttribute) == 0)
            {
                SetTask(time, i, m_NewAbstractedPadState[i].attribute,
                    HidTaskType::HidTaskType_SetVirtualPadConnection);
            }
        }
        m_OldAbstractedPadState[i] = m_NewAbstractedPadState[i];
    }
}

void IoPortRecorder::StoreXPadInput( const std::chrono::system_clock::time_point &time, HidShellHandle handle, uint32_t port, int32_t direction)
{
    HidShellLibraryWrapper &wrapper = HidShellLibraryWrapper::GetInstance();
    for (int i = 0; i < HidShellBasicXpadCountMax; i++)
    {
        wrapper.m_GetHidShellBasicXpadState(reinterpret_cast<void*>(handle), &m_NewXpadState[i], i, port, direction);

        // 接続状態チェック
        // TIPS: 接続タスクは、最初にスタックへ追加
        if (m_OldXpadState[i].powerState != m_NewXpadState[i].powerState && m_NewXpadState[i].powerState != HidShellXpadPowerState::Disconnected)
        {
            SetTask(time, i, static_cast<int32_t>(m_NewXpadState[i].powerState), HidTaskType::HidTaskType_SetXpadConnection);
        }

        // スティック動作チェック
        if (m_OldXpadState[i].stickL.x != m_NewXpadState[i].stickL.x || m_OldXpadState[i].stickL.y != m_NewXpadState[i].stickL.y)
        {
            // TIPS: X, Y の順番でタスクを登録する必要があります。
            SetTask(time, i, m_NewXpadState[i].stickL.x, HidTaskType::HidTaskType_SetXpadStickLX);
            SetTask(time, i, m_NewXpadState[i].stickL.y, HidTaskType::HidTaskType_SetXpadStickLY);
        }
        if (m_OldXpadState[i].stickR.x != m_NewXpadState[i].stickR.x || m_OldXpadState[i].stickR.y != m_NewXpadState[i].stickR.y)
        {
            // TIPS: X, Y の順番でタスクを登録する必要があります。
            SetTask(time, i, m_NewXpadState[i].stickR.x, HidTaskType::HidTaskType_SetXpadStickRX);
            SetTask(time, i, m_NewXpadState[i].stickR.y, HidTaskType::HidTaskType_SetXpadStickRY);
        }
        // ボタンチェック
        uint32_t pushButtonList     = m_NewXpadState[i].buttons & ~(m_OldXpadState[i].buttons);
        uint32_t releaseButtonList  = ~(m_NewXpadState[i].buttons) & m_OldXpadState[i].buttons;
        if (pushButtonList)
        {
            SetTask(time, i, pushButtonList, HidTaskType::HidTaskType_SetXpadPressButton);
        }
        if (releaseButtonList)
        {
            SetTask(time, i, releaseButtonList, HidTaskType::HidTaskType_SetXpadReleaseButton);
        }

        // 接続状態チェック
        // TIPS: 切断タスクは、最後にスタックへ追加
        if (m_OldXpadState[i].powerState != m_NewXpadState[i].powerState && m_NewXpadState[i].powerState == HidShellXpadPowerState::Disconnected)
        {
            SetTask(time, i, static_cast<int32_t>(m_NewXpadState[i].powerState), HidTaskType::HidTaskType_SetXpadConnection);
        }

        m_OldXpadState[i] = m_NewXpadState[i];
    }

    // Home ボタン
    wrapper.m_GetHidShellHomeButtonState(reinterpret_cast<void*>(handle), &m_NewHomeButtonState, port, direction);
    if (m_OldHomeButtonState.buttons != m_NewHomeButtonState.buttons)
    {
        HidShellHomeButton homeButtonState;
        SetTask(time, 0, (m_NewHomeButtonState.buttons == homeButtonState.Active), HidTaskType::HidTaskType_SetHomeButton);
    }
    m_OldHomeButtonState = m_NewHomeButtonState;

    // Capture ボタン
    wrapper.m_GetHidShellCaptureButtonState(reinterpret_cast<void*>(handle), &m_OldCaptureButtonState, port, direction);
    if (m_OldCaptureButtonState.buttons != m_NewCpatureButtonState.buttons)
    {
        HidShellCaptureButton captureButtonState;
        SetTask(time, 0, (m_NewCpatureButtonState.buttons == captureButtonState.Active), HidTaskType::HidTaskType_SetCaptureButton);
    }
    m_OldCaptureButtonState = m_NewCpatureButtonState;
}

void IoPortRecorder::StoreDebugPadInput( const std::chrono::system_clock::time_point &time, HidShellHandle handle, uint32_t port, int32_t direction )
{
    HidShellLibraryWrapper &wrapper = HidShellLibraryWrapper::GetInstance();

    // デバッグパッドの最新状態取得
    wrapper.m_GetHidShellDebugPadState(reinterpret_cast<void*>(handle), &m_NewDebugPadState, port, direction);

    // 接続状態チェック
    // TIPS: 接続タスクは、最初にスタックへ追加
    if (m_OldDebugPadState.attributes != m_NewDebugPadState.attributes && m_NewDebugPadState.attributes )
    {
        SetTask(time, 0, m_NewDebugPadState.attributes, HidTaskType::HidTaskType_SetDebugPadConnection);
    }

    // スティック動作チェック
    if (m_OldDebugPadState.stickL.x != m_NewDebugPadState.stickL.x || m_OldDebugPadState.stickL.y != m_NewDebugPadState.stickL.y)
    {
        // TIPS: X, Y の順番でタスクを登録する必要があります。
        SetTask(time, 0, m_NewDebugPadState.stickL.x, HidTaskType::HidTaskType_SetDebugPadStickLX);
        SetTask(time, 0, m_NewDebugPadState.stickL.y, HidTaskType::HidTaskType_SetDebugPadStickLY);
    }
    if (m_OldDebugPadState.stickR.x != m_NewDebugPadState.stickR.x || m_OldDebugPadState.stickR.y != m_NewDebugPadState.stickR.y)
    {
        // TIPS: X, Y の順番でタスクを登録する必要があります。
        SetTask(time, 0, m_NewDebugPadState.stickR.x, HidTaskType::HidTaskType_SetDebugPadStickRX);
        SetTask(time, 0, m_NewDebugPadState.stickR.y, HidTaskType::HidTaskType_SetDebugPadStickRY);
    }

    // ボタンチェック
    uint32_t pushButtonList     = m_NewDebugPadState.buttons & ~(m_OldDebugPadState.buttons);
    uint32_t releaseButtonList  = ~(m_NewDebugPadState.buttons) & m_OldDebugPadState.buttons;
    if (pushButtonList)
    {
        SetTask(time, 0, pushButtonList, HidTaskType::HidTaskType_SetDebugPadPressButton);
    }
    if (releaseButtonList)
    {
        SetTask(time, 0, releaseButtonList, HidTaskType::HidTaskType_SetDebugPadReleaseButton);
    }

    // 接続状態チェック
    // TIPS: 切断タスクは、最後にスタックへ追加
    if (m_OldDebugPadState.attributes != m_NewDebugPadState.attributes && !m_NewDebugPadState.attributes )
    {
        SetTask(time, 0, m_NewDebugPadState.attributes, HidTaskType::HidTaskType_SetDebugPadConnection);
    }

    // 過去入力情報の更新
    m_OldDebugPadState = m_NewDebugPadState;
}

void IoPortRecorder::StoreTouchScreenInput(const std::chrono::system_clock::time_point &time, HidShellHandle handle, uint32_t port, int32_t direction)
{
    HidShellLibraryWrapper &wrapper = HidShellLibraryWrapper::GetInstance();

    // タッチスクリーン最新状態取得
    wrapper.m_GetHidShellTouchScreenState(reinterpret_cast<void*>(handle), &m_NewTouchScreenState, port, direction);

    for (int newTouchId = 0; newTouchId < min( m_NewTouchScreenState.count, MaxTouchCount); newTouchId++)
    {
        int newFingerId = m_NewTouchScreenState.touches[newTouchId].fingerId;

        // 既存タッチの変更
        if (newTouchId < m_OldTouchScreenState.count)
        {
            // fingerId 変更の確認
            if (m_OldTouchScreenState.touches[newTouchId].fingerId != newFingerId)
            {
                SetTask(time, newTouchId, newFingerId, HidTaskType::HidTaskType_SetFingerId);
            }

            // タッチ移動の確認
            if (m_OldTouchScreenState.touches[newTouchId].x != m_NewTouchScreenState.touches[newTouchId].x ||
                m_OldTouchScreenState.touches[newTouchId].y != m_NewTouchScreenState.touches[newTouchId].y)
            {
                // TIPS: X, Y の順番でタスクを登録する必要があります。
                SetTask(time, newTouchId, m_NewTouchScreenState.touches[newTouchId].x, HidTaskType::HidTaskType_MoveTouchPoint_X);
                SetTask(time, newTouchId, m_NewTouchScreenState.touches[newTouchId].y, HidTaskType::HidTaskType_MoveTouchPoint_Y);
            }

            continue;
        }

        // 新規タッチの追加
        // TIPS: HidTaskType_SetTouchState タスクの後は、座標指定タスクをセットしておく必要があります。
        // 座標の順番は、X, Y の順番としなければなりません。
        SetTask(time, newTouchId, newFingerId, HidTaskType::HidTaskType_SetFingerId);
        SetTask(time, newTouchId, 1, HidTaskType::HidTaskType_SetTouchState);
        SetTask(time, newTouchId, m_NewTouchScreenState.touches[newTouchId].x, HidTaskType::HidTaskType_MoveTouchPoint_X);
        SetTask(time, newTouchId, m_NewTouchScreenState.touches[newTouchId].y, HidTaskType::HidTaskType_MoveTouchPoint_Y);
    }

    // タッチ解除コマンド
    for (int i = m_NewTouchScreenState.count; i < m_OldTouchScreenState.count; i++)
    {
        SetTask(time, i, 0, HidTaskType::HidTaskType_SetTouchState);
    }

    m_OldTouchScreenState = m_NewTouchScreenState;
}
