﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "ButtonController.h"
#include "DebugPadInputer.h"
#include "IoPortSender.h"
#include "PadInputer.h"
#include "VirtualPadInputer.h"
#include "XpadInputer.h"

class ControllerInterface
{
private:
    ButtonController            *m_ButtonController;
    PadInputer                  *m_Pad;
    IoPortSender                *m_Sender;

    int                         m_ControllerId;
    bool                        m_IsManaged;

public:
    explicit ControllerInterface(IoPortSender *sender, int controllerId, PadInputer* pad);
    virtual ~ControllerInterface();

    void Begin();
    void Close();

    void SetDefault();
    void SetState(Button buttonList[], int buttonCount, Stick stickList[], float stickDegreeList[], float stickPowerList[], int stickCount);

    // 単一ボタン操作
    void Press(Button button);
    void Release(Button button);
    void Push(Button button, int waitPress);

    // 複数同時ボタン操作
    void PressMulti(Button buttonList[], int count);
    void ReleaseMulti(Button buttonList[], int count);
    void PushMulti(Button buttonList[], int count, int waitPress);

    // スティック操作
    void HoldAnalogStick(Stick stick, float degree, float power);
    void ReleaseAnalogStick(Stick stick);
    void TiltAnalogStick(Stick stick, float degree, float power, int waitPress);

    void SetManagedStatus(bool isManaged);
};

/*-------------------------------------------------------
/ 通常コントローラーの factory メソッドクラス
--------------------------------------------------------*/
class HacController : public ControllerInterface
{
private:
    XpadInputer     *m_Pad;
    PadInputer* createPad(IoPortSender* sender, int controllerId)
    {
        m_Pad = new XpadInputer(sender, controllerId);
        return m_Pad;
    }

public:
    HacController(IoPortSender *sender, int controllerId)
        : m_Pad(NULL)
        , ControllerInterface(sender, controllerId, createPad(sender, controllerId))
    {
    }

    ~HacController()
    {
        if (m_Pad != NULL)
        {
            delete m_Pad;
        }
    }
};

class HacDebugController : public ControllerInterface
{
private:
    DebugPadInputer *m_Pad;
    PadInputer* createPad(IoPortSender* sender, int controllerId)
    {
        m_Pad = new DebugPadInputer(sender, 0);
        return m_Pad;
    }

public:
    explicit HacDebugController(IoPortSender *sender)
        : m_Pad(NULL)
        , ControllerInterface(sender, 0, createPad(sender, 0 ))
    {
    }

    ~HacDebugController()
    {
        if (m_Pad != NULL)
        {
            delete m_Pad;
        }
    }
};

class HacVirtualController : public ControllerInterface
{
private:
    VirtualPadInputer     *m_Pad;

    PadInputer* createPad(IoPortSender* sender,
            int controllerId,
            ControllerDeviceType deviceType,
            ControllerInterfaceType interfaceType,
            ControllerColor mainColor,
            ControllerColor subColor)
    {
        m_Pad = new VirtualPadInputer(sender, controllerId, deviceType, interfaceType, mainColor, subColor);
        return m_Pad;
    }

public:
    HacVirtualController(IoPortSender* sender, int controllerId,
        ControllerDeviceType deviceType,
        ControllerInterfaceType interfaceType,
        ControllerColor mainColor, ControllerColor subColor)
        : m_Pad(NULL)
        , ControllerInterface(sender, controllerId, createPad(sender, controllerId, deviceType, interfaceType, mainColor, subColor))
    {
    }
    ~HacVirtualController()
    {
        if (m_Pad != NULL)
        {
            delete m_Pad;
        }
    }
};
