﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ButtonController.h"

inline float Degree2Radian(float degree)
{
    return static_cast<float>(((degree / 180.0f) * (M_PI)));
}

ButtonController::ButtonController(PadInputer *pad)
{
    m_Pad = pad;
}

ButtonController::~ButtonController()
{
}

void ButtonController::SetState(const Button buttonList[], int buttonCount,
    const Stick stickList[], const float stickDegreeList[], const float stickPowerList[], int stickCount)
{
    int selectedStickCount = 0;
    bool isStickSelected[Stick::Stick_MaxCount] = {};
    Stick selectedStickList[Stick::Stick_MaxCount] = {};
    static ControllerStickValueType stickXList[Stick::Stick_MaxCount] = {};
    static ControllerStickValueType stickYList[Stick::Stick_MaxCount] = {};

    // TIPS:　stickList 内で、同一スティック操作が複数あった場合は一つに集約させる。
    // また、stickList に指定が無いスティックに関しては、状態を更新しない。
    for (int i = 0; i < stickCount; i++)
    {
        Stick type = stickList[i];
        switch (type)
        {
            case Stick::Stick_L:
            case Stick::Stick_R:
                if (!isStickSelected[type])
                {
                    isStickSelected[type] = true;
                    selectedStickList[selectedStickCount] = type;
                    // 同一スティック操作が複数指定された場合は、最初の値を利用
                    Degree2XyValue(&stickXList[selectedStickCount], &stickYList[selectedStickCount],
                        stickDegreeList[i], stickPowerList[i]);
                    selectedStickCount++;
                }
                break;
            default:
                throw InvalidStickException(stickList[i]);
        }
    }

    m_Pad->SetState(buttonList, buttonCount, selectedStickList, stickXList, stickYList, selectedStickCount);
}

void ButtonController::Press(Button button)
{
    m_Pad->Press(button);
}

void ButtonController::PressMulti(const Button buttonList[], int count)
{
    for (int i = 0; i < count; i++)
    {
        m_Pad->Press(buttonList[i]);
    }
}

void ButtonController::Release(Button button)
{
    m_Pad->Release(button);
}

void ButtonController::ReleaseMulti(const Button buttonList[], int count)
{
    for (int i = 0; i < count; i++)
    {
        m_Pad->Release(buttonList[i]);
    }
}

void ButtonController::Push(Button button, int waitPress)
{
    m_Pad->Press(button);
    Sleep(waitPress);
    m_Pad->Release(button);
}

void ButtonController::PushMulti(const Button buttonList[], int count, int waitPress)
{

    for (int i = 0; i < count; i++)
    {
        m_Pad->Press(buttonList[i]);
    }

    Sleep(waitPress);
    for (int i = 0; i < count; i++)
    {
        m_Pad->Release(buttonList[i]);
    }
}

void ButtonController::Degree2XyValue(ControllerStickValueType *pOutX, ControllerStickValueType *pOutY, float degree, float power)
{
    // 時計回り方向
    float rad = Degree2Radian(degree);
    (*pOutX) =  (ControllerStickValueType)(DefaultStickRadius * power * cos(rad));
    (*pOutY) = -(ControllerStickValueType)(DefaultStickRadius * power * sin(rad));
}

void ButtonController::HoldAnalogStick(Stick stick, float degree, float power)
{
    ControllerStickValueType x = 0;
    ControllerStickValueType y = 0;
    Degree2XyValue(&x, &y, degree, power);

    m_Pad->SetStickPos(stick, x, y);

}

void ButtonController::ReleaseAnalogStick(Stick stick)
{
    m_Pad->SetStickPos(stick, 0, 0);
}

void ButtonController::TiltAnalogStick(Stick stick, float degree, float power, int waitHold)
{
    ControllerStickValueType x = 0;
    ControllerStickValueType y = 0;
    Degree2XyValue(&x, &y, degree, power);

    m_Pad->SetStickPos(stick, x, y);
    Sleep(waitHold);
    m_Pad->SetStickPos(stick, 0, 0);
}
