﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "Common\autoTestAssistTool_FpsController.h"
#include "DirectshowCapture.h"
#include <chrono>

namespace
{
    const int   DEFAULT_FPS = 60;
    const int   WINDOW_720P_WIDTH       = 1280;
    const int   WINDOW_720P_HEIGHT      = 720;
    const int   WINDOW_1080P_WIDTH      = 1920;
    const int   WINDOW_1080P_HEIGHT     = 1080;
}

DirectshowCapture::DirectshowCapture(int deviceId, bool isPreview, CaptureResolution resolution)
{
    m_CaptureThread = NULL;
    ResetCapture(deviceId, isPreview, resolution);
}

DirectshowCapture::~DirectshowCapture()
{
    // キャプチャ終了
    EndCapture();

    // イメージ解放
    ReleaseAllImages();

    // キャプチャボード解放
    if (m_CaptureGrabber.IsOpened())
    {
        m_CaptureGrabber.Release();
    }
}

void DirectshowCapture::ResetCapture(int deviceId, bool isPreview, CaptureResolution resolution)
{
    // キャプチャ処理終了
    EndCapture();

    // イメージ解放
    ReleaseAllImages();

    // キャプチャボード解放
    if (m_CaptureGrabber.IsOpened())
    {
        m_CaptureGrabber.Release();
    }

    switch (resolution)
    {
        case CaptureResolution::CaptureResolution_720p:
            m_CaptureResolutionWidth = WINDOW_720P_WIDTH;
            m_CaptureResolutionHeight = WINDOW_720P_HEIGHT;
            break;
        case CaptureResolution::CaptureResolution_1080p:
            m_CaptureResolutionWidth = WINDOW_1080P_WIDTH;
            m_CaptureResolutionHeight = WINDOW_1080P_HEIGHT;
            break;
        default:
            // 不正な値指定時は、720p で動作させる
            m_CaptureResolutionWidth = WINDOW_720P_WIDTH;
            m_CaptureResolutionHeight = WINDOW_720P_HEIGHT;
            break;
    }

    // m_CaptureImage 初期化
    {
        std::lock_guard<std::mutex> lock(m_CaptureMutex);
        m_CaptureImage.create(cv::Size(m_CaptureResolutionWidth, m_CaptureResolutionHeight), CV_8UC3);
    }

    // キャプチャ開始
    m_IsPreview = isPreview;

    m_CaptureDeviceId = deviceId;
    assert(m_CaptureDeviceId >= 0);

    // キャプチャボード接続
    bool openResult = m_CaptureGrabber.Open(m_CaptureDeviceId, m_CaptureResolutionWidth, m_CaptureResolutionHeight);
    assert(openResult == true);

    // キャプチャ開始
    StartCapture();
}

std::string DirectshowCapture::GetPreviewName()
{
    std::ostringstream previewName;
    previewName << "[" << m_CaptureDeviceId << "] " << m_CaptureGrabber.GetDeviceName() << " - Auto Test Assist Tool";
    return previewName.str();
}

int DirectshowCapture::GetOriginalPreviewWidth()
{
    return m_CaptureResolutionWidth;
}

int DirectshowCapture::GetOriginalPreviewHeight()
{
    return m_CaptureResolutionHeight;
}

/*----------------------------------------------------
/ キャプチャ処理
/---------------------------------------------------*/
void DirectshowCapture::StartCapture()
{
    m_IsCapture = true;
    if (m_CaptureThread != NULL)
    {
        return;
    }

    // 開始前に一度キャプチャを取得しておく
    Capture();

    m_CaptureThread.reset(new std::thread(&DirectshowCapture::RunCapture, this));

    // プレビュー開始
    if (m_IsPreview)
    {
        StartPreview();
    }
}

void DirectshowCapture::EndCapture()
{
    // プレビュー終了
    EndPreview();

    m_IsCapture = false;
    if (m_CaptureThread != NULL)
    {
        m_CaptureThread->join();
        m_CaptureThread = NULL;
    }
}

void DirectshowCapture::RunCapture()
{
    autoTestAssistTool::util::FpsController fpsController(60);

    fpsController.BeginLoop();
    while (m_IsCapture)
    {
        Capture();

        // FPS 調整
        fpsController.SleepFrameInterval();
    }
    fpsController.EndLoop();
}

void DirectshowCapture::Capture()
{
    std::lock_guard<std::mutex> lock(m_CaptureMutex);
    // CaptureGrabber 内部で、データを RGB 24 へ変換しているため、常にこのサイズで取得することが出来る
    m_CaptureGrabber.Read(m_CaptureImage.ptr(), m_CaptureResolutionWidth * m_CaptureResolutionHeight * 3);
}
