﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <iostream>
#include <autoTestAssistTool/CaptureProcessorLibrary.h>

#include "CaptureDeviceUtil.h"
#include "DirectshowCapture.h"
#include "DevkitCapture.h"
#include "Common\autoTestAssistTool_TargetManagerUtil.h"
#include "Common\autoTestAssistTool_PathUtil.h"
#include "..\..\..\..\..\..\..\Externals\Oasis\include\tm.h"

#include <delayimp.h>

#pragma comment( lib, "delayimp.lib" )
#pragma comment( lib, "glew32s.lib")
#pragma comment( lib, "opengl32.lib")

#ifdef _DEBUG
#pragma comment( lib,"opencv_world300d.lib")
#pragma comment( lib, "tesseract305d.lib")
#else
#pragma comment( lib,"opencv_world300.lib")
#pragma comment( lib, "tesseract305.lib")
#endif

namespace
{
    ICapture *capture = NULL;

    // テンプレートマッチ用デフォルトパラメータ
    const int DefaultTemplateMatchingMethod = CV_TM_CCOEFF_NORMED;

    bool IsInitialized()
    {
        if (capture == NULL)
        {
            ERRMSG("Initialization is not completed.");
            return false;
        }
        return true;
    }
}

void InitializeForCaptureDevice(int deviceId, bool isPreview, CaptureResolution captureResolution)
{
    Finalize();

    capture = new DirectshowCapture(deviceId, isPreview, captureResolution);
}

CaptureResult InitializeForDevkitCapture(const SerialNumberString& serialNumber, bool isPreview)
{
    Finalize();

    std::string strIp;
    nn::tm::Error tmError = autoTestAssistTool::util::GetTargetNameFromSerial(&strIp, serialNumber);
    if (tmError != nn::tm::Error_Ok)
    {
        ERRMSG("GetTargetnameFromSerial failed\n");
        if (tmError == nn::tm::Error_InvalidTarget)
        {
            return CaptureResult::CaptureResult_NotFound;
        }
        return CaptureResult::CaptureResult_TargetManagerError;
    }

    SerialNumberString targetName = serialNumber;
    targetName.value[sizeof(SerialNumberString::value) - 1] = '\0';

    capture = new DevkitCapture(strIp, targetName.value, isPreview);

    return CaptureResult::CaptureResult_Success;
}

void Finalize()
{
    if (capture != NULL)
    {
        cv::destroyAllWindows();
        delete capture;
        capture = NULL;
    }
}

CaptureResult GetCaptureDeviceCount(int* pOutDeviceCount)
{
    CaptureDeviceUtil &captureDeviceUtil = CaptureDeviceUtil::GetInstance();
    return captureDeviceUtil.GetCaptureDeviceCount(pOutDeviceCount);
}

CaptureResult GetCaptureDeviceName(char* pOutNameBuffer, int nameBufferSize,int deviceId)
{
    CaptureDeviceUtil &captureDeviceUtil = CaptureDeviceUtil::GetInstance();
    return captureDeviceUtil.GetCaptureDeviceName(pOutNameBuffer, nameBufferSize, deviceId);
}

CaptureResult SaveImageFile(const char* filepath, ImageIdType id)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->SaveImageFile(filepath, id);
}

CaptureResult GetImageFromCaptureDevice(ImageIdType *pOutImageId)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->GetImageFromCaptureDevice(pOutImageId);
}

CaptureResult LoadImageFile( ImageIdType *pOutImageId, const char *filepath )
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->LoadImageFile(pOutImageId, filepath);
}

CaptureResult ReleaseImage(ImageIdType id)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->ReleaseImage(id);
}

CaptureResult ReleaseAllImages()
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }

    return capture->ReleaseAllImages();
}

CaptureResult GetImageSize(int* pOutWidth, int* pOutHeight, ImageIdType id)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->GetImageSize(pOutWidth, pOutHeight, id);

}
CaptureResult GetImageData( void* pOutColorBuffer, size_t colorBufferSize, ImageIdType id)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->GetImageData(pOutColorBuffer, colorBufferSize, id);
}


/*----------------------------------------------------------------------
/ 標準テンプレートマッチング
/---------------------------------------------------------------------*/
MatchingResult DetectObject(int* pOutX, int* pOutY, ImageIdType id, double threshold)
{
    if (!IsInitialized())
    {
        return MatchingResult::MatchingResult_NotInitialized;
    }

    return capture->DetectObject(pOutX, pOutY, id, threshold, DefaultTemplateMatchingMethod);
}


MatchingResult WaitDetectObject(int* pOutX, int* pOutY, ImageIdType id, double threshold, int timeout)
{
    if (!IsInitialized())
    {
        return MatchingResult::MatchingResult_NotInitialized;
    }

    return capture->WaitDetectObject(pOutX, pOutY, id, threshold, timeout, DefaultTemplateMatchingMethod);
}

MatchingResult DetectObjectWithFilter(int* pOutX, int* pOutY, ImageIdType id, double threshold, FilterIdType filterId)
{
    if (!IsInitialized())
    {
        return MatchingResult::MatchingResult_NotInitialized;
    }
    return capture->DetectObjectWithFilter(pOutX, pOutY, id, threshold, filterId, DefaultTemplateMatchingMethod);
}

MatchingResult WaitDetectObjectWithFilter(int* pOutX, int* pOutY, ImageIdType id, double threshold, int timeout, FilterIdType filterId)
{
    if (!IsInitialized())
    {
        return MatchingResult::MatchingResult_NotInitialized;
    }
    return capture->WaitDetectObjectWithFilter(pOutX, pOutY, id, threshold, timeout, filterId, DefaultTemplateMatchingMethod);
}

/*----------------------------------------------------------------------
/ フィルタ
/---------------------------------------------------------------------*/
CaptureResult CreateFilter(FilterIdType *pOutFilterId)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->CreateFilter(pOutFilterId);
}

CaptureResult DeleteFilter(FilterIdType filterId)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->DeleteFilter(filterId);
}

CaptureResult ApplyFilter(ImageIdType *pOutId, ImageIdType id, FilterIdType filterId)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->ApplyFilter(pOutId, id, filterId);
}

CaptureResult SetBinarizationFilter(FilterIdType filterId, int binarizeThreshold)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->SetBinarizationFilter(filterId, binarizeThreshold);
}

CaptureResult SetColorRangeFilter(FilterIdType filterId, Rgb24Color lowerColor, Rgb24Color upperColor)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->SetColorRangeFilter(filterId, lowerColor, upperColor);
}

CaptureResult SetMorphologyFilter(FilterIdType filterId, MorphologyConversionType type, int iterations)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->SetMorphologyFilter(filterId, type, iterations);
}

/*----------------------------------------------------------------------
/ 色検出
/---------------------------------------------------------------------*/
CaptureResult GetDotColor( Rgb24Color* pOutColor, ImageIdType id, int x, int y )
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->GetDotColor(pOutColor, id, x, y);
}

MatchingResult DetectDotColor(int x, int y, Rgb24Color lowerColor, Rgb24Color upperColor)
{
    if (!IsInitialized())
    {
        return MatchingResult::MatchingResult_NotInitialized;
    }
    return capture->DetectDotColor(x, y, lowerColor, upperColor);
}

MatchingResult WaitDetectDotColor(int x, int y, Rgb24Color lowerColor, Rgb24Color upperColor, int timeout)
{
    if (!IsInitialized())
    {
        return MatchingResult::MatchingResult_NotInitialized;
    }
    return capture->WaitDetectDotColor(x, y, lowerColor, upperColor, timeout);
}

/*----------------------------------------------------------------------
/ ラベリング
/---------------------------------------------------------------------*/
CaptureResult ExecuteLabeling(int *pOutResultAreaCount, LabelingResultHandleType *pOutLabelingResultHandle, ImageIdType imageId)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->ExecuteLabeling(pOutResultAreaCount, pOutLabelingResultHandle, imageId);
}

CaptureResult GetLabelingResults(int *pOutResultsCount, LabelingResult pOutResultArray[], LabelingResultHandleType labelingResultHandle, int arrayCount)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->GetLabelingResults(pOutResultsCount, pOutResultArray, labelingResultHandle, arrayCount);
}

CaptureResult ReleaseLabelingResult(LabelingResultHandleType labelingResultHandle)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->ReleaseLabelingResult(labelingResultHandle);
}

/*----------------------------------------------------------------------
/ OCR
/---------------------------------------------------------------------*/
MatchingResult DetectTextForImage(ImageRect *pOutRect,
    ImageIdType id,
    const char regexString[],
    OcrSeparateLevel separate,
    OcrLanguage language)
{
    if (!IsInitialized())
    {
        return MatchingResult::MatchingResult_NotInitialized;
    }
    return capture->DetectTextForImage(pOutRect, id, regexString, separate, language);
}

CaptureResult ExecuteOcr(int *pOutResultCount, OcrResultHandleType *pOutOcrResultHandle, ImageIdType id, OcrSeparateLevel separate, OcrLanguage language)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->ExecuteOcr(pOutResultCount, pOutOcrResultHandle, id, separate, language);
}

CaptureResult GetOcrResultStringSize(int *pOutResultSize, OcrResultHandleType ocrResultHandle, int number)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->GetOcrResultStringSize(pOutResultSize, ocrResultHandle, number);
}

CaptureResult GetOcrResultString(ImageRect *pOutRect, int *pOutResultSize, char pOutResultString[], int bufferSize, OcrResultHandleType ocrResultHandle, int number)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->GetOcrResultString( pOutRect, pOutResultSize, pOutResultString, bufferSize, ocrResultHandle, number);
}

CaptureResult ReleaseOcrResult( OcrResultHandleType ocrResultHandle )
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->ReleaseOcrResult(ocrResultHandle);
}

/*----------------------------------------------------------------------
/ その他
/---------------------------------------------------------------------*/
CaptureResult OpenImageWindow(const char* windowName, ImageIdType id)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->OpenImageWindow(windowName, id);
}

CaptureResult CloseImageWindow(const char* windowName)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->CloseImageWindow(windowName);
}

CaptureResult ExtractImage(ImageIdType *pOutImageId, ImageIdType id, ImageRect rect)
{
    if (!IsInitialized())
    {
        return CaptureResult::CaptureResult_NotInitialized;
    }
    return capture->ExtractImage(pOutImageId, id, rect);
}

namespace
{
    FARPROC WINAPI DelayHook(unsigned dliNotify, PDelayLoadInfo pdli)
    {
        switch (dliNotify)
        {
            case dliNotePreLoadLibrary:
            {
                std::string dllName = pdli->szDll;
                if (dllName.find("opencv") != std::string::npos)
                {
                    std::string libraryPath = autoTestAssistTool::util::GetSdkRootPath() + "\\Tools\\AutoTestAssistTools\\Libraries\\opencv_world300.dll";
                    HMODULE hModule = LoadLibrary(libraryPath.c_str());
                    return (FARPROC)hModule;
                }
                else if (dllName.find("tesseract") != std::string::npos)
                {
                    // Leptonica
                    std::string libraryPath = autoTestAssistTool::util::GetSdkRootPath() + "\\Tools\\AutoTestAssistTools\\Libraries\\leptonica-1.75.3.dll";
                    HMODULE hModule = LoadLibrary(libraryPath.c_str());
                    if (hModule == NULL)
                    {
                        return (FARPROC)hModule;
                    }

                    // Tesseract-Ocr
                    libraryPath = autoTestAssistTool::util::GetSdkRootPath() + "\\Tools\\AutoTestAssistTools\\Libraries\\tesseract305.dll";
                    hModule = LoadLibrary(libraryPath.c_str());
                    return (FARPROC)hModule;
                }
            }
            default:
                break;
        }
        return NULL;
    }
}

const PfnDliHook __pfnDliNotifyHook2 = DelayHook;

BOOL APIENTRY DllMain(HMODULE hModule,
    DWORD  ul_reason_for_call,
    LPVOID lpReserved
)
{
    switch (ul_reason_for_call)
    {
        case DLL_PROCESS_ATTACH:
            break;
        case DLL_THREAD_ATTACH:
            break;
        case DLL_THREAD_DETACH:
            break;
        case DLL_PROCESS_DETACH:
            Finalize();
            break;
        default:
            break;
    }
    return TRUE;
}
