﻿using NAudio.Wave;
using System;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;

namespace AudioRedirector
{
    public class Program
    {
        private const string TargetServerPortName = "AudioRedirector";
        private const int OldTargetManagerHtcsPort = 8003;
        private TcpClient _client;
        private BinaryWriter _writer;
        private IWaveIn _capture;
        private bool _exit = false;

        public bool Run(string[] args)
        {
            try
            {
                var endpoint = GetAudioRedirectionEndpoint();
                if (endpoint == null)
                {
                    Console.Error.WriteLine("cannot find AudioRedirectorServer");
                    return false;
                }
                _client = new TcpClient();
                _client.Connect(endpoint);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine(e.Message);
                return false;
            }

            _writer = new BinaryWriter(_client.GetStream(), Encoding.ASCII, true);

            _capture = new WasapiLoopbackCapture();
            _capture.DataAvailable += OnDataAvailable;

            // TODO: need to introduce WaveFormat converter
            if (_capture.WaveFormat.SampleRate != 48000 ||
                (_capture.WaveFormat.Encoding != WaveFormatEncoding.IeeeFloat && _capture.WaveFormat.Encoding != WaveFormatEncoding.Extensible) ||
                _capture.WaveFormat.BitsPerSample != 32)
            {
                Console.Error.WriteLine("default audio render endpoint should be 48kHz");
                _writer.Dispose();
                _client.Close();
                return false;
            }

            Console.WriteLine("start redirecting, press any key to exit");
            _capture.StartRecording();

            while (!Console.KeyAvailable && !_exit)
            {
                if (Console.KeyAvailable)
                {
                    Console.WriteLine("exit");
                    break;
                }
                if (_exit)
                {
                    Console.WriteLine("connection lost");
                }
            }

            _capture.StopRecording();

            _writer.Dispose();
            _client.Close();

            return true;
        }

        private void OnDataAvailable(object sender, WaveInEventArgs e)
        {
            var format = _capture.WaveFormat;
            if (format.Encoding == WaveFormatEncoding.IeeeFloat || format.Encoding == WaveFormatEncoding.Extensible)
            {
                var buffer = new byte[e.BytesRecorded / (format.BitsPerSample / 8) * sizeof(short)];
                if (format.BitsPerSample == 32)
                {
                    for (var i = 0; i < buffer.Length / sizeof(short); ++i)
                    {
                        var sample = (int)(BitConverter.ToSingle(e.Buffer, i * 4) * 32768);
                        var converted = (short)(sample < Int16.MaxValue ? sample > Int16.MinValue ? sample : Int16.MinValue : Int16.MaxValue);
                        BitConverter.GetBytes(converted).CopyTo(buffer, 2 * i);
                    }
                    try
                    {
                        _writer.Write((byte[])buffer.Cast<byte>(), 0, buffer.Length);
                    }
                    catch (Exception)
                    {
                        _exit = true;
                    }
                }
            }
        }

        private Socket CreateConnectedSocket(IPEndPoint endPoint)
        {
            var socket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
            socket.Connect(endPoint);
            return socket;
        }

        private bool IsHtcsControlPort(int port)
        {
            try
            {
                using (var socket = CreateConnectedSocket(new IPEndPoint(IPAddress.Loopback, port)))
                {
                    using (var socketStream = new NetworkStream(socket))
                    {
                        using (var socketReader = new StreamReader(socketStream))
                        {
                            Task<string> str = socketReader.ReadLineAsync();
                            if (str.Wait(3000))
                            {
                                return (str.Result.IndexOf("<HtcsInfo><TargetList>") == 0);
                            }
                        }
                    }
                }
            }
            catch
            {
                return false;
            }
            return false;
        }

        private IPEndPoint GetAudioRedirectionEndpoint()
        {
            Nintendo.Tm.Target defaultTarget;
            Nintendo.Tm.TargetManager.TryGetDefaultTarget(out defaultTarget);
            var defaultTargetName = defaultTarget.GetName();
            var defaultTargetSerialNumber = defaultTarget.GetSerialNumber();
            Console.WriteLine("connecting to the default target " + defaultTargetName);

            // find TargetManager htcs port
            int htcsPort = 0;
            try
            {
                htcsPort = Nintendo.Tm.TargetManager.GetHtcsControlPort();
            }
            catch (Nintendo.Tm.TmException e)
            {
                if (e.GetErrorCode() == Nintendo.Tm.Error.IncompatibleTargetManager && IsHtcsControlPort(OldTargetManagerHtcsPort))
                {
                    htcsPort = OldTargetManagerHtcsPort;
                }
                else
                {
                    return null;
                }
            }

            // connect to TargetManager
            var client = new TcpClient();
            try
            {
                client.Connect("localhost", htcsPort);
            }
            catch (Exception)
            {
                return null;
            }

            // find audio redirection endpoint
            var start = DateTime.Now;
            var timeout = TimeSpan.FromSeconds(10);
            using (var sr = new StreamReader(client.GetStream(), new UTF8Encoding(false), false, 1024, true))
            {
                while (true)
                {
                    try
                    {
                        if (sr.Peek() <= 0)
                        {
                            System.Threading.Thread.Sleep(TimeSpan.FromMilliseconds(100));
                            if (DateTime.Now - start >= timeout)
                            {
                                return null;
                            }
                            continue;
                        }
                        string line = sr.ReadLine();
                        var element = XElement.Parse(line);
                        if (element.Name == "HtcsInfo")
                        {
                            foreach (var portMapItem in element.Element("PortMap").Elements("PortMapItem"))
                            {
                                if (portMapItem.Element("HtcsPortName").Value == TargetServerPortName && portMapItem.Element("HtcsPeerName").Value == defaultTargetSerialNumber)
                                {
                                    var ipAddress = IPAddress.Parse(portMapItem.Element("IPAddress").Value);
                                    var portNumber = int.Parse(portMapItem.Element("TcpPortNumber").Value, CultureInfo.InvariantCulture);
                                    return new IPEndPoint(ipAddress, portNumber);
                                }
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        Console.Error.WriteLine(e.Message);
                    }
                }
            }
        }

        public static int Main(string[] args)
        {
            return (new Program().Run(args) ? 0 : 1);
        }
    }
}
