﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <cstring>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/gc/detail/gc_Util.h>
#include <nn/gc/gc.h>
#include <nn/util/util_StringUtil.h>


const size_t FilePathSize         = 512;
const size_t CardHeaderHashSize   = 32;
const size_t LogLineMaxNum        = 15;
const size_t LogMaxLength         = 128;

enum CardState
{
    CardState_ValidCard,
    CardState_InvalidCardData,
    CardState_InvalidCardType,
    CardState_Accessing,
    CardState_NotInserted,
    CardState_BrokenCard,
    CardState_Repairing,
    CardState_UnKnownError
};

class DisplayData
{
private:
    DisplayData()
    {
        memset(m_FilePath, 0x00, FilePathSize);
        memset(m_ParamPath, 0x00, FilePathSize);
        memset(m_CardHeaderHash, 0x00, CardHeaderHashSize);
        memset(m_DeviceId, 0x00, nn::gc::GcCardDeviceIdSize);
        cardState = CardState_NotInserted;
        isMountError = false;
        isWriteCancelConfirm = false;
    }
    char m_FilePath[FilePathSize];
    char m_ParamPath[FilePathSize];
    char m_CardHeaderHash[32];
    char m_DeviceId[nn::gc::GcCardDeviceIdSize];

public:
    static DisplayData* GetInstance()
    {
        static DisplayData displayData;
        return &displayData;
    }

    void Reset()
    {
        memset(m_CardHeaderHash, 0x0, CardHeaderHashSize);
    }

    void SetFileName(const char* filePath, size_t filePathSize)
    {
        if(filePathSize - 1 > FilePathSize)
        {
            NN_ABORT("FilePath is %s\nfile path length must be lower than %d\n", FilePathSize);
        }
        nn::util::Strlcpy(m_FilePath, filePath, FilePathSize);
    }
    char* GetFileName()
    {
        return m_FilePath;
    }

    bool IsMatchExtension(const char* fileEx, const char* expEx)
    {
        if(fileEx == nullptr || nn::util::Strnlen(fileEx, FilePathSize) != nn::util::Strnlen(expEx, FilePathSize))
        {
            return false;
        }
        return nn::util::Strncmp(fileEx, expEx, nn::util::Strnlen(expEx, FilePathSize)) == 0;
    }

    bool IsWritableFile()
    {
#if defined(NN_XCIE_WRITER_XCIE_MODE)
        return IsMatchExtension(strrchr(m_FilePath, static_cast<int>('.')), ".xcie");
#elif defined(NN_XCIE_WRITER_XCIR_MODE)
        return IsMatchExtension(strrchr(m_FilePath, static_cast<int>('.')), ".xcir");
#else
        return false;
#endif
    }

    bool IsXciFile()
    {
        return IsMatchExtension(strrchr(m_FilePath, static_cast<int>('.')), ".xci");
    }

    char* GetParamPath()
    {
        return m_ParamPath;
    }

    void SetCardHeaderHash(char* cardHeaderHash, size_t cardHeaderHashSize)
    {
        if(cardHeaderHashSize > CardHeaderHashSize)
        {
            NN_LOG("File is Too Long\n");
            return;
        }
        memcpy(m_CardHeaderHash, cardHeaderHash, cardHeaderHashSize);
    }
    char* GetCardHeaderHash()
    {
        return m_CardHeaderHash;
    }
    void SetDeviceId(char* deviceIdBuf, size_t deviceIdBufSize)
    {
        if(deviceIdBufSize != nn::gc::GcCardDeviceIdSize)
        {
            NN_LOG("Device Buffer Over\n");
            return;
        }
        memcpy(m_DeviceId, deviceIdBuf, nn::gc::GcCardDeviceIdSize);
    }
    void ResetDeviceId()
    {
        memset(m_DeviceId, 0x00, nn::gc::GcCardDeviceIdSize);
    }
    char* GetDeviceId()
    {
        return m_DeviceId;
    }


    bool IsCardWritable()
    {
        return (cardState == CardState_ValidCard);
    }

    nn::os::Tick errorTick;
    int64_t fileSize;
    int64_t validDataSize;
    int64_t currentWrittenSize;

    bool isMountError;
    bool isFileExist;
    bool isParamExist;
    bool isWriteCancelConfirm;
    CardState cardState;
};
