﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShopMonitoringTool_UiMenuScene.h"

MenuScene::MenuScene(std::string titleName)
{
    m_SelectedItemIndex = 0;
    m_MenuTitleName = titleName;
}

MenuScene::~MenuScene()
{

}

MenuScene* MenuScene::GetPtr()
{
    return this;
}

void MenuScene::PushItem(MenuItem* pItem)
{
    return m_ItemList.push_back(pItem);
}

void MenuScene::SetupScene()
{
    for (int i = 0; i < m_ItemList.size(); i++)
    {
        MenuItem* pItem = GetItemList().at(i);
        pItem->SetState(MenuItem::SelectiveState::Default);
    }

    //Selectableな先頭の要素が初期値
    m_SelectedItemIndex = 0;
    for (int i = 0; i < m_ItemList.size(); i++)
    {
        MenuItem* pItem = GetItemList().at(i);
        if (pItem->IsSelectable())
        {
            m_SelectedItemIndex = i;
            pItem->SetState(MenuItem::SelectiveState::Selected);
            break;
        }
    }
}

std::vector<MenuItem*>& MenuScene::GetItemList()
{
    return m_ItemList;
}

MenuItem* MenuScene::GetSelectedItem()
{
    NN_ASSERT(this->m_ItemList.size() > 0);
    return m_ItemList.at(m_SelectedItemIndex);
}

void MenuScene::NextItem()
{
    MenuItem* pCurrentItem = GetSelectedItem();
    int currentSelectedItemIndex = m_SelectedItemIndex;
    while (true)
    {
        // Selectableな要素になるまでインクリメント
        (++m_SelectedItemIndex) %= m_ItemList.size();
        if (GetSelectedItem()->IsSelectable())
        {
            break;
        }
        NN_ASSERT(currentSelectedItemIndex != m_SelectedItemIndex);
    }

    pCurrentItem->SetState(MenuItem::SelectiveState::Default);
    GetSelectedItem()->SetState(MenuItem::SelectiveState::Selected);
}
void MenuScene::PrevItem()
{
    MenuItem* pCurrentItem = GetSelectedItem();
    int currentSelectedItemIndex = m_SelectedItemIndex;
    while (true)
    {
        // Selectableな要素になるまでデクリメント
        m_SelectedItemIndex = (m_SelectedItemIndex + m_ItemList.size() - 1) % m_ItemList.size();
        if (GetSelectedItem()->IsSelectable())
        {
            break;
        }
        NN_ASSERT(currentSelectedItemIndex != m_SelectedItemIndex);
    }

    pCurrentItem->SetState(MenuItem::SelectiveState::Default);
    GetSelectedItem()->SetState(MenuItem::SelectiveState::Selected);
}
void MenuScene::EnterItem()
{
    Scene* pNextScene = GetSelectedItem()->CallEnter();
    GetSelectedItem()->AtatchedGeneralFunction();
    if (pNextScene == nullptr)
    {
        return;
    }

    // 次のシーンへの繋ぎ
    GetSelectedItem()->SetState(MenuItem::SelectiveState::Decided);
    this->SetState(Scene::State::EnabledInactive);
    this->SetNextScene(pNextScene);
    pNextScene->SetState(Scene::State::EnabledActive);
    pNextScene->SetPrevScene(GetPtr());

}
void MenuScene::Draw() {

    static const nn::util::Color4u8Type titleBarColor       = { {  35,  35,  35, 255 } };
    static const nn::util::Color4u8Type titleNameColor      = { {  80,  80,  80, 255 } };
    static const nn::util::Color4u8Type windowActiveColor   = { { 120, 120, 120, 255 } };
    static const nn::util::Color4u8Type windowInactiveColor = { {  80,  80,  80, 255 } };
    static const nn::util::Color4u8Type windowDisableColor  = { {  60,  60,  60, 255 } };
    static const nn::util::Color4u8Type itemDefaultColor    = { {   0,   0,   0, 255 } };
    static const nn::util::Color4u8Type itemSelectedColor   = { {   0,  50,  30, 255 } };
    static const nn::util::Color4u8Type itemDicidedColor    = { {  50,  10,  10, 255 } };
    static const nn::util::Color4u8Type itemHeaderColor     = { {  30,  30,  60, 255 } };
    static const nn::util::Color4u8Type frameColor          = { {  15,  15,  15, 255 } };

    const int lineHeihgt = 24;
    const int itemMarginW = 16;
    const int headerMarginLeft = 8;
    const int windowWidth = 500;
    const int windowHeight = lineHeihgt * m_ItemList.size();
    const int itemStartPosY = m_DrawPosY + lineHeihgt;

    // タイトルバー描画
    gDrawer.DrawRect(m_DrawPosX, m_DrawPosY, windowWidth, lineHeihgt, titleBarColor);
    gWriter.SetTextColor(titleNameColor);
    gWriter.SetCursor(m_DrawPosX + headerMarginLeft, m_DrawPosY);
    gWriter.Print(m_MenuTitleName.c_str());

    // メインウィンドウ描画
    nn::util::Color4u8Type windowColor;
    switch (GetState())
    {
    case Scene::State::EnabledActive:
        windowColor = windowActiveColor;
        break;
    case Scene::State::EnabledInactive:
        windowColor = windowInactiveColor;
        break;
    case Scene::State::Disabled:
    default:
        windowColor = windowDisableColor;
        break;
    }
    gDrawer.DrawRect(m_DrawPosX, itemStartPosY, windowWidth, windowHeight, windowColor);

    // 外枠
    gDrawer.DrawFrame(m_DrawPosX, m_DrawPosY, windowWidth, lineHeihgt + windowHeight, frameColor);

    // アイテム描画
    nn::util::Color4u8Type itemColor;
    for (int i = 0; i < m_ItemList.size(); i++)
    {
        MenuItem* pItem = GetItemList().at(i);
        if (pItem->IsSelectable()) {
            switch (pItem->GetState())
            {
            case MenuItem::SelectiveState::Selected:
                itemColor = itemSelectedColor;
                break;
            case MenuItem::SelectiveState::Decided:
                itemColor = itemDicidedColor;
                break;
            case MenuItem::SelectiveState::Default:
            default:
                itemColor = itemDefaultColor;
                break;
            }
            if (!pItem->IsEditable())
            {
                itemColor.v[3] = 100;
            }

            gWriter.SetTextColor(itemColor);
            gWriter.SetCursor(m_DrawPosX + itemMarginW, itemStartPosY + i*lineHeihgt);
            gWriter.Print(pItem->GetName().c_str());

            int stringWidth = gWriter.CalculateStringWidth(pItem->GetValueName().c_str());
            gWriter.SetCursor(m_DrawPosX + windowWidth - stringWidth - itemMarginW, itemStartPosY + i*lineHeihgt);
            gWriter.Print(pItem->GetValueName().c_str());
        }
        else
        {
            gWriter.SetTextColor(itemHeaderColor);
            gWriter.SetCursor(m_DrawPosX + headerMarginLeft, itemStartPosY + i*lineHeihgt);
            gWriter.Print(pItem->GetName().c_str());
        }
    }

    // NextMenu
    if (GetState() == Scene::State::EnabledInactive)
    {
        m_pNextScene->Draw();
    }
}
void MenuScene::Update()
{
    if (this->m_ItemList.size() < 1)
    {
        return;
    }

    if (GetState() == Scene::State::EnabledInactive)
    {
        m_pNextScene->Update();

        // こちらのシーンに戻ってきた場合の処理
        if (m_pNextScene->GetState() == Scene::State::Disabled)
        {
            //APPLOG_DEBUG("--Back--\n");
            GetSelectedItem()->SetState(MenuItem::SelectiveState::Selected);
            SetState(Scene::State::EnabledActive);
        }
    }
}
void MenuScene::Control()
{
    // 非アクティブな場合入力を受けない
    if (GetState() == Scene::State::EnabledInactive)
    {
        m_pNextScene->Control();
        return;
    }

    // 上位メニューへ戻る
    if (gInput.isTrg(nn::hid::NpadButton::B::Mask))
    {
        EscapeScene();
    }

    // アイテムが無い
    if (this->m_ItemList.size() < 1)
    {
        return;
    }

    // 選択不可なアイテム
    if (!GetSelectedItem()->IsSelectable())
    {
        return;
    }

    NN_ASSERT(GetSelectedItem()->GetState() == MenuItem::SelectiveState::Selected)

    // カーソル移動
    if (gInput.isRep(nn::hid::NpadButton::Up::Mask))
    {
        PrevItem();
    }
    else
    if (gInput.isRep(nn::hid::NpadButton::Down::Mask))
    {
        NextItem();
    }

    // 編集不可な（無効な）アイテム
    if (!GetSelectedItem()->IsEditable())
    {
        return;
    }

    // 編集・実行・Scene遷移
    if (gInput.isTrg(nn::hid::NpadButton::A::Mask))
    {
        EnterItem();
    }
    else
    if (gInput.isRep(nn::hid::NpadButton::Right::Mask))
    {
        GetSelectedItem()->CallIncrement();
    }
    else
    if (gInput.isRep(nn::hid::NpadButton::Left::Mask))
    {
        GetSelectedItem()->CallDecrement();
    }
}

std::string MenuScene::GetDescription()
{
    // アイテムが無い
    if (this->m_ItemList.size() < 1)
    {
        return "Empty";
    }

    // 選択不可なアイテムが選択されている
    if (!GetSelectedItem()->IsSelectable())
    {
        return "Unselectable Item";
    }

    // 非アクティブ
    if (GetState() == Scene::State::EnabledInactive)
    {
        return m_pNextScene->GetDescription();
    }

    return GetSelectedItem()->GetDescription();
}
