﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShopMonitoringTool_UiExecutorScene.h"

NN_OS_ALIGNAS_THREAD_STACK char ExecutorScene::m_ThreadStack[ThreadStackSize];

ExecutorScene::ExecutorScene()
{
    SetupScene();

    // イベントを初期化する
    nn::os::InitializeEvent(&m_ProcessCancelEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_ProcessCompleteEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_ProcessTerminateEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_ProcessStartAsContinuousEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_ProcessStartAsSelectiveEvent, false, nn::os::EventClearMode_AutoClear);

    //スレッド開始
    nn::Result result = nn::os::CreateThread(
        &m_ProcessThread,
        ProcessThreadMain,
        this,
        m_ThreadStack,
        ThreadStackSize,
        nn::os::DefaultThreadPriority
    );
    NN_ASSERT(result.IsSuccess(), "Cannot create thread.");
    nn::os::StartThread(&m_ProcessThread);
}

ExecutorScene::~ExecutorScene()
{
    // プロセススレッドを終了させる
    nn::os::SignalEvent(&m_ProcessTerminateEvent);
    nn::os::WaitThread(&m_ProcessThread);
    nn::os::DestroyThread(&m_ProcessThread);

    // イベントをファイナライズする
    nn::os::FinalizeEvent(&m_ProcessStartAsContinuousEvent);
    nn::os::FinalizeEvent(&m_ProcessCancelEvent);
    nn::os::FinalizeEvent(&m_ProcessCompleteEvent);
    nn::os::FinalizeEvent(&m_ProcessTerminateEvent);
    nn::os::FinalizeEvent(&m_ProcessStartAsSelectiveEvent);
}

void ExecutorScene::SetupScene()
{
    m_ProcessState = ProcessState::Standby;
    m_ExecuteMode = ExecuteMode::Selective;
    m_CurrentRemainingTime = 0.f;
    m_CurrentErrorCount = 0;
    m_LapCount = 0;
    m_LastTaskState = ExecuteItem::ExecuteState::Unexecuted;
    m_LastTaskName = "";
    m_LastTaskErrorDescription = "";
    m_IsEnableAutoRestart = true;
    m_IsEnablShowErrorApplet = false;
    m_SelectTaskNum = 0;
    m_AutoRestartWaitTime = 5;
    m_AutoRetryWaitTime = 3;
    m_AutoContinueWaitTime = 1;
    m_CurrentRemainingTime = 0.f;
    m_CurrentErrorCount = 0;
    m_ErrorCountLimit = 3;
    ResetAllTaskState();
    ClearTaskList();
}

void ExecutorScene::SetExecutionMode(ExecuteMode mode)
{
    m_ExecuteMode = mode;
}

void ExecutorScene::SetEnableAutoRestart(bool enableAutoRestart)
{
    m_IsEnableAutoRestart = enableAutoRestart;
}

void ExecutorScene::SetErrorCountLimit(int errorCountLimit)
{
    m_ErrorCountLimit = errorCountLimit;
}

void ExecutorScene::SetAutoRestartWaitTime(int autoRestartWaitTime)
{
    m_AutoRestartWaitTime = autoRestartWaitTime;
}

void ExecutorScene::SetAutoRetryWaitTime(int autoRetryWaitTime)
{
    m_AutoRetryWaitTime = autoRetryWaitTime;
}

void ExecutorScene::SetEnablShowErrorApplet(bool enablShowErrorApplet)
{
    m_IsEnablShowErrorApplet = enablShowErrorApplet;
}

void ExecutorScene::PushTask(ExecuteItem* _pTask)
{
    m_TaskList.push_back(_pTask);
}

void ExecutorScene::ResetAllTaskState()
{
    for (int i = 0; i < m_TaskList.size(); i++)
    {
        m_TaskList.at(i)->ResetState();
    }
}

void ExecutorScene::ClearTaskList()
{
    m_TaskList.clear();
    m_TaskList.shrink_to_fit();
}

void ExecutorScene::ClearLastTaskState()
{
    m_LastTaskState = ExecuteItem::ExecuteState::Unexecuted;
    m_LastTaskName = "";
    m_LastTaskErrorDescription = "";
}

void ExecutorScene::ThrowError()
{
    m_CurrentErrorCount++;
    if (m_CurrentErrorCount >= m_ErrorCountLimit)
    {
        nn::os::SignalEvent(&m_ProcessCancelEvent);
    }
}

void ExecutorScene::ResetErrorCount()
{
    m_CurrentErrorCount = 0;
}

void ExecutorScene::NextTask()
{
    if (m_TaskList.size() < 1)
    {
        return;
    }
    (++m_SelectTaskNum) %= m_TaskList.size();
}

void ExecutorScene::PrevTask()
{
    if (m_TaskList.size() < 1)
    {
        return;
    }
    m_SelectTaskNum = (m_SelectTaskNum + m_TaskList.size() - 1) % m_TaskList.size();
}

void ExecutorScene::ProcessThreadMain(void *arg)
{
    ExecutorScene* caller = (ExecutorScene*)arg;
    while (true)
    {//多重イベント待ち
        int signaled = nn::os::WaitAny(
            &caller->m_ProcessStartAsContinuousEvent,
            &caller->m_ProcessStartAsSelectiveEvent,
            &caller->m_ProcessTerminateEvent
        );

        if (signaled == 0)//m_ProcessStartMultiEvent
        {
            nn::os::ClearEvent(&caller->m_ProcessStartAsContinuousEvent);
            caller->ExecuteContinuousTask(arg);
        }
        else
        if (signaled == 1)//m_ProcessStartSingleTaskEvent
        {
            nn::os::ClearEvent(&caller->m_ProcessStartAsSelectiveEvent);
            caller->ExecuteSelectedTask(arg);
        }
        else
        if (signaled == 2)//m_ProcessTerminateEvent
        {
            nn::os::ClearEvent(&caller->m_ProcessTerminateEvent);
            break;
        }

        //タスク終了イベント発行
        nn::os::SignalEvent(&caller->m_ProcessCompleteEvent);
    }
}

void ExecutorScene::ExecuteContinuousTask(void *arg)
{
    ExecutorScene* caller = (ExecutorScene*)arg;
    //タスク実行
    int taskNum = 0;
    while (taskNum < caller->m_TaskList.size())
    {
        caller->m_LastTaskName = caller->m_TaskList.at(taskNum)->GetName();

        //実行
        bool boolResult = false;
        nn::Result result = caller->m_TaskList.at(taskNum)->Execute(&boolResult);

        //nn::Result ハンドリング
        if (result.IsFailure())
        {
            char errorCodeBuf[16];
            sprintf(errorCodeBuf, "2%03d-%04d", result.GetModule(), result.GetDescription());
            APPLOG_ERROR("ErrorCode : %s\n", errorCodeBuf);
            caller->m_LastTaskErrorDescription = errorCodeBuf;
            if (caller->m_IsEnablShowErrorApplet)
            {
                nn::err::ShowError(result);
            }
        }
        else
        {
            caller->m_LastTaskErrorDescription = "";
        }
        caller->m_LastTaskState = caller->m_TaskList.at(m_SelectTaskNum)->GetState();

        if (boolResult)
        {//成功なら次へ
            taskNum++;
            caller->ResetErrorCount();
        }
        else
        {//失敗ならエラーを発行
            caller->ThrowError();
        }

        //キャンセル
        if (nn::os::TryWaitEvent(&caller->m_ProcessCancelEvent))
        {
            //APPLOG_WARNING("Canceled\n");
            break;
        }

        // インターバル兼キャンセル待ち
        if (taskNum < caller->m_TaskList.size())
        {
            int waitTime = (boolResult) ? m_AutoContinueWaitTime : m_AutoRetryWaitTime;
            if (nn::os::TimedWaitEvent(&caller->m_ProcessCancelEvent, nn::TimeSpan::FromSeconds(waitTime)))
            {
                break;
            }
        }
    }
}

void ExecutorScene::ExecuteSelectedTask(void *arg)
{
    ExecutorScene* caller = (ExecutorScene*)arg;

    caller->m_LastTaskName = caller->m_TaskList.at(m_SelectTaskNum)->GetName();

    //実行
    bool boolResult = false;
    nn::Result result = caller->m_TaskList.at(m_SelectTaskNum)->Execute(&boolResult);

    //nn::Result ハンドリング
    if (result.IsFailure())
    {
        char errorCodeBuf[16];
        sprintf(errorCodeBuf, "2%03d-%04d", result.GetModule(), result.GetDescription());
        APPLOG_ERROR("ErrorCode : %s\n", errorCodeBuf);
        caller->m_LastTaskErrorDescription = errorCodeBuf;
        if (caller->m_IsEnablShowErrorApplet)
        {
            nn::err::ShowError(result);
        }
    }
    else
    {
        caller->m_LastTaskErrorDescription = "";
    }
    caller->m_LastTaskState = caller->m_TaskList.at(m_SelectTaskNum)->GetState();
}

void ExecutorScene::Draw()
{
    static const nn::util::Color4u8Type textDefaultColor = { { 0,   0,   0, 255 } };
    static const nn::util::Color4u8Type windowSelectiveColor  = { { 130, 160, 130, 255 } };
    static const nn::util::Color4u8Type windowContinuousColor = { { 130, 130, 160, 255 } };
    static const nn::util::Color4u8Type stateBarColor = { { 120, 120, 120, 255 } };
    static const nn::util::Color4u8Type defaultErrorCounterColor = { { 60,60,60, 255 } };
    static const nn::util::Color4u8Type failErrorCounterColor = { { 120,60,60, 255 } };
    static const nn::util::Color4u8Type selectedColor = { { 40,100,40,255 } };
    static const nn::util::Color4u8Type processingColor = { { 40,40,100,255 } };

    const int lineHeight = 30;
    const int marginW = 30;
    const int marginH = 10;
    const int paddingTopInline = 4;
    const int windowWidth = 600;
    const int windowHeight = m_TaskList.size() * lineHeight + marginH * 2;
    const int cursorWidth = 5;
    const int cursorMarginW = marginW / 2;

    //メインウィンドウ
    nn::util::Color4u8Type windowColor = (m_ExecuteMode == ExecuteMode::Selective) ? windowSelectiveColor : windowContinuousColor;
    gDrawer.DrawRect(m_DrawPosX, m_DrawPosY, windowWidth, windowHeight, windowColor);

    //ステート
    gDrawer.DrawRect(m_DrawPosX, m_DrawPosY + m_TaskList.size() * lineHeight + marginH * 2 , windowWidth, lineHeight, stateBarColor);
    gWriter.SetTextColor(textDefaultColor);
    gWriter.SetCursor(m_DrawPosX + marginW, m_DrawPosY + windowHeight + paddingTopInline);
    gWriter.Print("State : ");
    switch (m_ProcessState)
    {
    case ProcessState::Standby:
        gWriter.Print("Ready");
        break;
    case ProcessState::Processing:
        gWriter.Print("Processing");
        break;
    case ProcessState::Canceling:
        gWriter.Print("Canceling");
        break;
    case ProcessState::Result:
        gWriter.Print("Result");
        break;
    case ProcessState::Waiting:
        char s[20];
        sprintf(s, "Wait : %05.2f", m_CurrentRemainingTime);
        gWriter.Print(s);
        break;
    case ProcessState::Alert:
        gWriter.Print("Alert");
        break;
    default:
        gWriter.Print("Undefined");
        break;
    }

    //タスク
    nn::util::Color4u8Type sColor;
    for (int i = 0; i < m_TaskList.size(); i++)
    {
        // 選択中カーソル
        if (m_SelectTaskNum == i && m_ExecuteMode == ExecuteMode::Selective)
        {
            sColor = selectedColor;
            gDrawer.DrawRect(m_DrawPosX + cursorMarginW, m_DrawPosY + marginH + i * lineHeight, cursorWidth, lineHeight, sColor);
        }

        // 処理中カーソル
        if (m_TaskList.at(i)->GetState() == ExecuteItem::ExecuteState::Processing)
        {
            sColor = processingColor;
            gWriter.SetTextColor(sColor);
            gDrawer.DrawRect(m_DrawPosX + cursorMarginW, m_DrawPosY + marginH + i * lineHeight, cursorWidth, lineHeight, sColor);
        }
        else
        {
            gWriter.SetTextColor(textDefaultColor);
        }

        //
        gWriter.SetCursor(m_DrawPosX + marginW, m_DrawPosY + marginH + i * lineHeight + paddingTopInline);
        gWriter.Print(m_TaskList.at(i)->GetName());

        // ExecuteState 表示
        std::string execStateString;
        switch (m_TaskList.at(i)->GetState())
        {
        case ExecuteItem::ExecuteState::Unexecuted:
            execStateString = "[Unexecuted]";
            break;
        case ExecuteItem::ExecuteState::Processing:
            execStateString = "[Processing]";
            break;
        case ExecuteItem::ExecuteState::Succeeded:
            execStateString = "[Successed]";
            break;
        case ExecuteItem::ExecuteState::Failed:
            execStateString = "[Failed]";
            break;
        default:
            break;
        }
        int stringWidth = gWriter.CalculateStringWidth(execStateString.c_str());
        gWriter.SetCursor(m_DrawPosX + windowWidth - stringWidth - marginW, m_DrawPosY + marginH + i * lineHeight + paddingTopInline);
        gWriter.Print(execStateString.c_str());
    }

    //エラーカウンタ
    const int errorCounterSize = 20;
    const int errorCounterMarginW = 10;
    const int paddingTopErrorCounter = 5;
    if (m_ExecuteMode == ExecuteMode::Continuous)
    {
        for (int i = 0; i < m_ErrorCountLimit; i++)
        {
            gDrawer.DrawRect(m_DrawPosX + windowWidth - marginW - (m_ErrorCountLimit - i) * (errorCounterSize + errorCounterMarginW) + errorCounterMarginW,
                m_DrawPosY + windowHeight + paddingTopErrorCounter, errorCounterSize, errorCounterSize, defaultErrorCounterColor);
        }
        for (int i = 0; i < m_CurrentErrorCount; i++)
        {
            gDrawer.DrawRect(m_DrawPosX + windowWidth - marginW - (m_ErrorCountLimit - i) * (errorCounterSize + errorCounterMarginW) + errorCounterMarginW,
                m_DrawPosY + windowHeight + paddingTopErrorCounter, errorCounterSize, errorCounterSize, failErrorCounterColor);
        }
    }
} // NOLINT (readability/fn_size)

void ExecutorScene::Update()
{

    if (this->m_TaskList.size() < 1)
    {
        return ;
    }

    switch (m_ProcessState)
    {
    case ProcessState::Standby:
        break;
    case ProcessState::Processing:
    case ProcessState::Canceling:
        //処理終了
        if (nn::os::TryWaitEvent(&m_ProcessCompleteEvent))
        {
            APPLOG_INFO("Process finished");

            if (m_ExecuteMode == ExecuteMode::Selective)
            {//Selectiveモードなら無条件で待機
                m_ProcessState = ProcessState::Standby;
            }
            else
            if (m_CurrentErrorCount >= m_ErrorCountLimit)
            {//処理失敗で発砲
                m_ProcessState = ProcessState::Alert;
                gAudio.PlayAlarm();
                APPLOG_ERROR("Problem has been detected");
            }
            else
            if (m_ProcessState == ProcessState::Canceling)
            {//キャンセル完了で待機
                m_ProcessState = ProcessState::Result;
            }
            else
            if (m_IsEnableAutoRestart)
            {//処理成功で自動ループ
                m_LapCount++;
                m_Timer.StartWaitTimer(m_AutoRestartWaitTime);
                m_ProcessState = ProcessState::Waiting;
                APPLOG_INFO("Wait %d seconds...\n", m_AutoRestartWaitTime);
            }
            else
            {//処理成功で待機
                m_ProcessState = ProcessState::Result;
            }
        }
        break;
    case ProcessState::Waiting:
        m_CurrentRemainingTime = m_Timer.GetRemainingTime();
        if (!m_Timer.IsRemining())
        {//自動再実行
            APPLOG_INFO("Restart to execute");
            ResetAllTaskState();
            nn::os::SignalEvent(&m_ProcessStartAsContinuousEvent);
            m_ProcessState = ProcessState::Processing;
        }
        break;
    case ProcessState::Alert:
        break;
    default:
        break;
    }
}

void ExecutorScene::Control()
{
    if (this->m_TaskList.size() < 1)
    {
        if (gInput.isTrg(nn::hid::NpadButton::B::Mask))
        {
            EscapeScene();
        }
        return;
    }

    if (gInput.isTrg(nn::hid::NpadButton::A::Mask))
    {
        if (m_ProcessState == ProcessState::Standby)
        {//処理開始
            m_ProcessState = ProcessState::Processing;
            if (m_ExecuteMode == ExecuteMode::Selective)
            {
                nn::os::SignalEvent(&m_ProcessStartAsSelectiveEvent);
            }
            else
            if (m_ExecuteMode == ExecuteMode::Continuous)
            {
                APPLOG_INFO("Start to execute");
                nn::os::SignalEvent(&m_ProcessStartAsContinuousEvent);
            }
        }
        else
        if (m_ProcessState == ProcessState::Alert || m_ProcessState == ProcessState::Result)
        {//リザルト・警報からの待機へ
            m_LapCount = 0;
            ResetErrorCount();
            ResetAllTaskState();
            ClearLastTaskState();
            m_ProcessState = ProcessState::Standby;
            gAudio.StopAlarm();
        }
    }
    else
        if (gInput.isTrg(nn::hid::NpadButton::B::Mask))
        {
            if (m_ProcessState == ProcessState::Standby)
            {//シーン終了
                EscapeScene();
                ResetAllTaskState();
                ClearTaskList();
            }
            else
            if (m_ProcessState == ProcessState::Processing)
            {//処理キャンセル
                m_ProcessState = ProcessState::Canceling;
                nn::os::SignalEvent(&m_ProcessCancelEvent);
                APPLOG_WARNING("Cancel\n");
            }
            else
            if (m_ProcessState == ProcessState::Waiting)
            {//ループ待ちから待機へ
                m_LapCount = 0;
                ResetErrorCount();
                ResetAllTaskState();
                m_ProcessState = ProcessState::Standby;
            }
            else
            if (m_ProcessState == ProcessState::Result)
            {//リザルトからの待機へ
                m_LapCount = 0;
                ResetAllTaskState();
                ClearLastTaskState();
                m_ProcessState = ProcessState::Standby;
            }
        }

    if (m_ProcessState == ProcessState::Standby && m_ExecuteMode == ExecuteMode::Selective)
    {
        if (gInput.isRep(nn::hid::NpadButton::Up::Mask))
        {
            PrevTask();
        }
        else
        if (gInput.isRep(nn::hid::NpadButton::Down::Mask))
        {
            NextTask();
        }
    }
}

std::string ExecutorScene::GetDescription()
{

    // タスクが無い
    if (this->m_TaskList.size() < 1)
    {
        return "Empty";
    }


    if (m_ExecuteMode == ExecuteMode::Continuous)
    {
        //return "Desc : Continuous Mode";
        switch (m_ProcessState)
        {
        case ProcessState::Standby:
            return "自動監視を開始します";
            break;
        case ProcessState::Processing:
            return "自動監視を実行中です...";
            break;
        case ProcessState::Canceling:
            return "自動監視を停止しています...";
            break;
        case ProcessState::Result:
            return "自動監視を停止しました";
            break;
        case ProcessState::Waiting:
            return "待機中です...";
            break;
        case ProcessState::Alert:
            return "エラーが検出されました";
            break;
        default:
            return "Undefined";
            break;
        }
    }
    else
    {
        return m_TaskList.at(m_SelectTaskNum)->GetDescription();
    }
}
