﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShopMonitoringTool_SysInput.h"

Input& Input::GetInstance() NN_NOEXCEPT
{
    static Input Instance;
    return Instance;
}

void Input::Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();

    // 使用する Npad の ID を定義
    for (unsigned int i = 0; i < MaxNpadDeviceCount - 1; ++i)
    {
        m_NpadIds[i] = i;
    }
    m_NpadIds[MaxNpadDeviceCount - 1] = nn::hid::NpadId::Handheld;

    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleFullKey::Mask  |
        nn::hid::NpadStyleJoyDual::Mask  |
        nn::hid::NpadStyleJoyLeft::Mask  |
        nn::hid::NpadStyleJoyRight::Mask |
        nn::hid::NpadStyleHandheld::Mask
    );

    // 使用する Npad を設定
    nn::hid::SetSupportedNpadIdType(m_NpadIds, MaxNpadDeviceCount);
}

void Input::Update() NN_NOEXCEPT
{
    nn::hid::NpadButtonSet button;
    button.Reset();
    for (int i = 0; i < MaxNpadDeviceCount; ++i)
    {
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadIds[i]);
        if (style.Test<nn::hid::NpadStyleFullKey>())
        {
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, m_NpadIds[i]);
            button |= state.buttons;
        }
        else if (style.Test<nn::hid::NpadStyleJoyDual>())
        {
            nn::hid::NpadJoyDualState state;
           nn::hid::GetNpadState(&state, m_NpadIds[i]);
            button |= state.buttons;
        }
        else if (style.Test<nn::hid::NpadStyleJoyLeft>())
        {
            nn::hid::NpadJoyLeftState state;
            nn::hid::GetNpadState(&state, m_NpadIds[i]);
            button |= state.buttons;
        }
        else if (style.Test<nn::hid::NpadStyleJoyRight>())
        {
            nn::hid::NpadJoyRightState state;
            nn::hid::GetNpadState(&state, m_NpadIds[i]);
            button |= state.buttons;
        }
        else if (style.Test<nn::hid::NpadStyleHandheld>())
        {
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, m_NpadIds[i]);
            button |= state.buttons;
        }
        if (button.IsAnyOn() && m_NpadInputCount[i] < UINT64_MAX)
        {
            ++m_NpadInputCount[i];
        }
        else
        {
            m_NpadInputCount[i] = 0;
        }
    }
    button |= GetDebugPadButtons();
    m_ButtonPressCount.Update(button);
    m_InputButtonSet = button;
}

nn::hid::NpadButtonSet Input::GetNpadButtonSet() const NN_NOEXCEPT
{
    return m_InputButtonSet;
}

nn::hid::NpadButtonSet Input::GetDebugPadButtons() NN_NOEXCEPT
{
    nn::hid::NpadButtonSet result;
    result.Reset();
    nn::hid::DebugPadState state;
    nn::hid::GetDebugPadState(&state);

    result.Set<nn::hid::NpadButton::A>(state.buttons.Test< nn::hid::DebugPadButton::A >());
    result.Set<nn::hid::NpadButton::B>(state.buttons.Test< nn::hid::DebugPadButton::B >());
    result.Set<nn::hid::NpadButton::X>(state.buttons.Test< nn::hid::DebugPadButton::X >());
    result.Set<nn::hid::NpadButton::Y>(state.buttons.Test< nn::hid::DebugPadButton::Y >());
    result.Set<nn::hid::NpadButton::L>(state.buttons.Test< nn::hid::DebugPadButton::L >());
    result.Set<nn::hid::NpadButton::R>(state.buttons.Test< nn::hid::DebugPadButton::R >());
    result.Set<nn::hid::NpadButton::ZL>(state.buttons.Test< nn::hid::DebugPadButton::ZL >());
    result.Set<nn::hid::NpadButton::ZR>(state.buttons.Test< nn::hid::DebugPadButton::ZR >());
    result.Set<nn::hid::NpadButton::Plus>(state.buttons.Test< nn::hid::DebugPadButton::Start >());
    result.Set<nn::hid::NpadButton::Minus>(state.buttons.Test< nn::hid::DebugPadButton::Select >());
    result.Set<nn::hid::NpadButton::Up>(state.buttons.Test< nn::hid::DebugPadButton::Up >());
    result.Set<nn::hid::NpadButton::Down>(state.buttons.Test< nn::hid::DebugPadButton::Down >());
    result.Set<nn::hid::NpadButton::Left>(state.buttons.Test< nn::hid::DebugPadButton::Left >());
    result.Set<nn::hid::NpadButton::Right>(state.buttons.Test< nn::hid::DebugPadButton::Right >());

    return result;
}

bool Input::isTrg(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT
{
    return m_ButtonPressCount.GetCount(button, NpadButtonCounter::CountMode::Min) == 1;
}

bool Input::isHld(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT
{
    return m_ButtonPressCount.GetCount(button, NpadButtonCounter::CountMode::MaxOmitRel) > 0;
}

bool Input::isRep(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT
{
    return
        (isTrg(button)) ||
        (m_ButtonPressCount.GetCount(button, NpadButtonCounter::CountMode::MaxOmitRel) == RepeatStartTime) ||
        (m_ButtonPressCount.GetCount(button, NpadButtonCounter::CountMode::MaxOmitRel) > RepeatStartTime &&
        (m_ButtonPressCount.GetCount(button, NpadButtonCounter::CountMode::MaxOmitRel) - RepeatStartTime) % RepeatSpan == 0);
}

bool Input::isRel(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT
{
    return m_ButtonPressCount.GetCount(button, NpadButtonCounter::CountMode::Max) == UINT64_MAX;
}

//==========================================================================================

const uint64_t NpadButtonCounter::GetCount(const nn::hid::NpadButtonSet& button, CountMode mode) const NN_NOEXCEPT
{
    int count = 0;
    uint64_t result = 0;
    for (int i = 0; i < MaxNpadButtons && count < button.CountPopulation(); ++i)
    {
        if (button[i])
        {
            switch (mode)
            {
            case CountMode::Max:
                result = std::max(m_Count[i], result);
                break;
            case CountMode::Min:
                if (m_Count[i] != 0)
                {
                    if (result == 0) { result = UINT64_MAX; }
                    result = std::min(m_Count[i], result);
                }
                break;
            case CountMode::MaxOmitRel:
                if (m_Count[i] != UINT64_MAX)
                {
                    result = std::max(m_Count[i], result);
                }
                break;
            default:
                result = m_Count[i];
                break;
            }
            ++count;
        }
    }
    return result;
}

const uint64_t  NpadButtonCounter::GetCount(uint32_t index) const NN_NOEXCEPT
{
    if (index >= MaxNpadButtons)
    {
        return 0;
    }
    return m_Count[index];
}

void NpadButtonCounter::Update(const nn::hid::NpadButtonSet& button) NN_NOEXCEPT
{
    for (int i = 0; i < MaxNpadButtons; ++i)
    {
        if (m_Count[i] == UINT64_MAX)
        {
            m_Count[i] = 0;
        }
        else if (button[i])
        {
            ++m_Count[i];
        }
        else if (m_Count[i] > 0)
        {
            m_Count[i] = UINT64_MAX;
        }
    }
}
