﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShopMonitoringTool_SysAudio.h"
#include "ShopMonitoringTool_SysFileSystem.h"

NN_ALIGNAS(4096) char g_AudioRendererWorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];

nn::mem::StandardAllocator g_AudioRendererAllocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

Audio& Audio::GetInstance() NN_NOEXCEPT
{
    static Audio instance;
    return instance;
}

void Audio::Initialize() NN_NOEXCEPT
{
    // アロケータの初期化
    g_AudioRendererAllocator.Initialize(g_AudioRendererWorkBuffer, sizeof(g_AudioRendererWorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 6 + 2; // FinalMix(6) + SubMix(2)
    parameter.voiceCount = 24;
    parameter.subMixCount = 2;
    parameter.sinkCount = 1;
    parameter.effectCount = 3;
    parameter.performanceFrameCount = 0;

    // ミックスバッファとオーディオバスの関係を定義します。
    m_MainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
    m_MainBus[nn::audio::ChannelMapping_FrontRight] = 5;

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    // レンダラのワークバッファを準備します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    m_AudioRendererWorkBuffer = g_AudioRendererAllocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    //m_AudioRendererWorkBuffer = new(std::nothrow) char[workBufferSize];
    NN_ABORT_UNLESS_NOT_NULL(m_AudioRendererWorkBuffer);


    // レンダラを初期化します。
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&m_AudioRendererHandle, &m_SystemEvent, parameter, m_AudioRendererWorkBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    m_AudioRendererConfigBuffer = g_AudioRendererAllocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    //m_AudioRendererConfigBuffer = new(std::nothrow) char[workBufferSize];
    NN_ABORT_UNLESS_NOT_NULL(m_AudioRendererConfigBuffer);
    nn::audio::InitializeAudioRendererConfig(&m_AudioRendererConfig, parameter, m_AudioRendererConfigBuffer, configBufferSize);


    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    nn::audio::AcquireFinalMix(&m_AudioRendererConfig, &m_FinalMix, 6);
    nn::Result result = nn::audio::AddDeviceSink(&m_AudioRendererConfig, &deviceSink, &m_FinalMix, m_MainBus, ChannelCount, "MainAudioOut");
    NN_ABORT_UNLESS(result.IsSuccess());

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(m_AudioRendererHandle, &m_AudioRendererConfig);
    NN_ABORT_UNLESS(result.IsSuccess());

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(m_AudioRendererHandle);
    NN_ABORT_UNLESS(result.IsSuccess());

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    bool rt = AcquireMemoryPool(&m_AudioRendererConfig, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    {
        m_pAlarmAdpcmHeader = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(m_pAlarmAdpcmHeader, &m_AlarmDataBuffer, AlarmAdpcmFileName);
        nn::audio::AcquireVoiceSlot(&m_AudioRendererConfig, &m_AlarmVoice, m_pAlarmAdpcmHeader->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &m_pAlarmAdpcmHeader->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&m_AudioRendererConfig, &m_AlarmVoice, &m_FinalMix);

        m_AlarmWaveBuffer.buffer = m_AlarmDataBuffer;
        m_AlarmWaveBuffer.size = dataSeSize;
        m_AlarmWaveBuffer.startSampleOffset = 0;
        m_AlarmWaveBuffer.endSampleOffset = m_pAlarmAdpcmHeader->sampleCount;
        m_AlarmWaveBuffer.loop = true;
        m_AlarmWaveBuffer.isEndOfStream = false;
        m_AlarmWaveBuffer.pContext = &m_pAlarmAdpcmHeader->loopContext;
        m_AlarmWaveBuffer.contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&m_AlarmVoice, &m_AlarmWaveBuffer);
        nn::audio::SetVoiceMixVolume(&m_AlarmVoice, &m_FinalMix, 0.707f / 2, 0, m_MainBus[0]);
        nn::audio::SetVoiceMixVolume(&m_AlarmVoice, &m_FinalMix, 0.707f / 2, 0, m_MainBus[1]);
    }
}

void Audio::Finalize() NN_NOEXCEPT
{
    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(m_AudioRendererHandle);
    nn::audio::CloseAudioRenderer(m_AudioRendererHandle);
    nn::os::DestroySystemEvent(m_SystemEvent.GetBase());

    // メモリを解放します。
    if (m_AlarmDataBuffer)
    {
        g_WaveBufferAllocator.Free(m_AlarmDataBuffer);
        m_AlarmDataBuffer = nullptr;
        g_WaveBufferAllocator.Free(m_pAlarmAdpcmHeader);
        m_pAlarmAdpcmHeader = nullptr;
    }
    if (m_AudioRendererConfigBuffer)
    {
        g_AudioRendererAllocator.Free(m_AudioRendererConfigBuffer);
        m_AudioRendererConfigBuffer = nullptr;
    }
    if (m_AudioRendererWorkBuffer)
    {
        g_AudioRendererAllocator.Free(m_AudioRendererWorkBuffer);
        m_AudioRendererWorkBuffer = nullptr;
    }

    // ファイルシステム終了
    //FinalizeFileSystem();
}

void Audio::PlayAlarm() NN_NOEXCEPT
{
    nn::audio::AppendWaveBuffer(&m_AlarmVoice, &m_AlarmWaveBuffer);
    nn::audio::SetVoicePlayState(&m_AlarmVoice, nn::audio::VoiceType::PlayState_Play);
}

void Audio::StopAlarm() NN_NOEXCEPT
{
    nn::audio::SetVoicePlayState(&m_AlarmVoice, nn::audio::VoiceType::PlayState_Stop);
    nn::audio::GetReleasedWaveBuffer(&m_AlarmVoice);
}

void Audio::Update() NN_NOEXCEPT
{
    nn::Result result = nn::audio::RequestUpdateAudioRenderer(m_AudioRendererHandle, &m_AudioRendererConfig);
    NN_ABORT_UNLESS(result.IsSuccess());
}
