﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShopMonitoringTool_AppExecutor.h"
#include "ShopMonitoringTool_AppColorPalette.h"

CommandExecutor::CommandExecutor()
    :ExecutorScene()
{
    m_InterpolatedCancelingBoardOffsetX = 500.f;
}

void CommandExecutor::ResetCursor()
{
    m_InterpolatedScrollOffsetY = 0.f;
}

void CommandExecutor::DrawContinuousMode()
{
    const int lineHeight = 40;
    const int stringHeightOffset = (lineHeight - 20) / 2;
    const int marginW = 30;
    const int marginH = 20;
    const int stateBlockMarginW = 300;
    const int windowWidth = 400;
    const int margin = 5;

    // タスク表示
    for (int i = 0; i < m_TaskList.size(); i++)
    {
        nn::util::Color4u8Type taskNameColor = AppColor::DefaultTextColor;
        nn::util::Color4u8Type taskBlockColor = AppColor::DefaultStateLabelColor;
        int stringWidth = gWriter.CalculateStringWidth(m_TaskList.at(i)->GetName());
        switch (m_TaskList.at(i)->GetState())
        {
        case ExecuteItem::ExecuteState::Unexecuted:
            break;
        case ExecuteItem::ExecuteState::Processing:
            taskNameColor = AppColor::ProcessingColor;
            taskBlockColor = AppColor::ProcessingColor;
            m_Indicator.Draw(m_DrawPosX + marginW + stringWidth + 20, m_DrawPosY + marginH + i*lineHeight + 5, 30, AppColor::ProcessingColor);
            break;
        case ExecuteItem::ExecuteState::Succeeded:
            taskNameColor = AppColor::ProcessedColor;
            taskBlockColor = AppColor::SuccessColor;
            break;
        case ExecuteItem::ExecuteState::Failed:
            taskNameColor = AppColor::ProcessedColor;
            taskBlockColor = AppColor::ErrorColor;
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
        // タスクステートラベル
        gDrawer.DrawRect(m_DrawPosX + marginW + stateBlockMarginW, m_DrawPosY + marginH + i*lineHeight + stringHeightOffset + 1, 20, 20, taskBlockColor);
        // タスク名
        gWriter.SetTextColor(taskNameColor);
        gWriter.SetCursor(m_DrawPosX + marginW, m_DrawPosY + marginH + i*lineHeight + stringHeightOffset);
        gWriter.Print(m_TaskList.at(i)->GetName());
    }

    // タスクフレーム描画
    nn::util::Color4u8Type taskFrameColor = AppColor::DefaultTextColor;
    switch (m_ProcessState)
    {
        case ProcessState::Standby:
            taskFrameColor = AppColor::SelectedColor;
            gDrawer.DrawFrame(m_DrawPosX, m_DrawPosY, windowWidth, lineHeight * m_TaskList.size() + marginH * 2, taskFrameColor);
            gDrawer.DrawFrame(m_DrawPosX + margin, m_DrawPosY + margin,
                windowWidth - margin * 2, lineHeight * m_TaskList.size() + marginH * 2 - margin * 2, taskFrameColor);
            gWriter.SetTextColor(AppColor::SelectedColor);
            gWriter.SetCursor(m_DrawPosX + margin + 10, m_DrawPosY + margin + lineHeight * m_TaskList.size() + marginH * 2 - margin * 2 + 20);
            gWriter.Print("Press A");
            break;
        case ProcessState::Processing:
            taskFrameColor = AppColor::ProcessingColor;
            gDrawer.DrawFrame(m_DrawPosX, m_DrawPosY, windowWidth, lineHeight * m_TaskList.size() + marginH * 2, taskFrameColor);
            gDrawer.DrawFrame(m_DrawPosX + margin, m_DrawPosY + margin,
                windowWidth - margin * 2, lineHeight * m_TaskList.size() + marginH * 2 - margin * 2, taskFrameColor);
            break;
        default:
            break;
    }


    //ステート枠
    //ステートウィンドウ
    nn::util::Color4u8Type frameColor = AppColor::ProcessedColor;
    const char* stateHeader[] =
    {
        "State",
        "Lap",
        "Error",
    };
    const int stateHeaderNum = sizeof stateHeader / sizeof stateHeader[0];
    //
    const int statePosX = m_DrawPosX + windowWidth + 50;
    const int statePosY = m_DrawPosY;
    const int windowHeght = lineHeight * stateHeaderNum + marginH;
    const int valueMarginW = 90;

    // Alert
    const int aleatMargin = 20;
    m_AlertFrameCounter++;
    if (m_AlertFrameCounter >= 1) {
        m_AlertFrameCounter = 0;
        ++m_AlertDrawCounter %= aleatMargin * 2;
    }
    if (m_ProcessState == ProcessState::Alert)
    {
        for (int i = 50; i >= -1; i--)
        {
            if (i % 2 == 0) {
                gDrawer.DrawRect(
                    statePosX - aleatMargin*i - m_AlertDrawCounter,
                    statePosY - aleatMargin*i - m_AlertDrawCounter,
                    windowWidth + (aleatMargin * i + m_AlertDrawCounter) * 2,
                    windowHeght + (aleatMargin * i + m_AlertDrawCounter) * 2,
                    AppColor::ErrorColor);
            }
            else
            {
                gDrawer.DrawRect(
                    statePosX - aleatMargin*i - m_AlertDrawCounter,
                    statePosY - aleatMargin*i - m_AlertDrawCounter,
                    windowWidth + (aleatMargin * i + m_AlertDrawCounter) * 2,
                    windowHeght + (aleatMargin * i + m_AlertDrawCounter) * 2,
                    AppColor::BackgroundColor);
            }
        }
    }

    //ステートウィンドウ
    std::string stateString;
    gWriter.SetTextColor(AppColor::DefaultTextColor);
    switch (m_ProcessState)
    {
    case ProcessState::Standby:
        stateString = "Standby";
        break;
    case ProcessState::Canceling:
        stateString = "Canceling";
        break;
    case ProcessState::Processing:
        stateString = "Processing";
        break;
    case ProcessState::Alert:
        frameColor = AppColor::ErrorColor;
        stateString = "Alert";
        break;
    case ProcessState::Result:
        frameColor = AppColor::SelectedColor;
        stateString = "Result";
        break;
    case ProcessState::Waiting:
        frameColor = AppColor::ProcessingColor;
        char s[20];
        sprintf(s, "Waiting %5.1f sec", m_CurrentRemainingTime);
        stateString = s;
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    //ステートウィンドウ項目描画
    for (int i = 0; i < stateHeaderNum; i++)
    {
        gWriter.SetTextColor(AppColor::DefaultTextColor);
        gWriter.SetCursor(statePosX + marginW, statePosY + marginH + lineHeight * i);
        gWriter.Print(stateHeader[i]);
        gWriter.SetTextColor(AppColor::ValueColor);
        gWriter.SetCursor(statePosX + marginW + valueMarginW, statePosY + marginH + lineHeight * i);
        switch (i)
        {
            case 0:
                gWriter.Print(stateString.c_str());
                break;
            case 1:
                gWriter.Print(std::to_string(m_LapCount).c_str());
                break;
            case 2:
                for (int n = 0; n < m_ErrorCountLimit; n++)
                {
                    gDrawer.DrawRect(statePosX + marginW + valueMarginW + n * 30, statePosY + marginH + lineHeight * i, 20, 20, AppColor::DefaultStateLabelColor);
                }
                for (int n = 0; n < m_CurrentErrorCount; n++)
                {
                    gDrawer.DrawRect(statePosX + marginW + valueMarginW + n * 30, statePosY + marginH + lineHeight * i, 20, 20, AppColor::ErrorColor);
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // ステートウィンドウフレーム描画
    gDrawer.DrawFrame(statePosX, statePosY, windowWidth, windowHeght, frameColor);
    gDrawer.DrawFrame(statePosX + margin, statePosY + margin, windowWidth - margin * 2, windowHeght - margin * 2, frameColor);
    if (m_ProcessState == ProcessState::Result)
    {
        gWriter.SetTextColor(AppColor::SelectedColor);
        gWriter.SetCursor(statePosX + 10, statePosY + windowHeght + 10);
        gWriter.Print("Press A or B");
    }

    // キャンセルボード
    const int defaultCancelingBoardOffsetX = 500;
    int cancelingBoardOffsetX = (m_ProcessState == ProcessState::Canceling) ? 0 : defaultCancelingBoardOffsetX;
    m_InterpolatedCancelingBoardOffsetX = m_InterpolatedCancelingBoardOffsetX * 0.7f +  cancelingBoardOffsetX * 0.3f;
    nn::util::Color4u8Type cancelColor = { { 100, 90, 20, 255 } };
    gDrawer.DrawFrame((int)m_InterpolatedCancelingBoardOffsetX + statePosX, statePosY + windowHeght + 50, windowWidth, 50, cancelColor);
    gDrawer.DrawRect((int)m_InterpolatedCancelingBoardOffsetX + statePosX + margin, statePosY + windowHeght + 50 + margin, windowWidth - margin * 2, 50 - margin * 2, cancelColor);
    gWriter.SetCursor((int)m_InterpolatedCancelingBoardOffsetX + statePosX + 20 , statePosY + windowHeght + 50 + 14);
    gWriter.SetTextColor(AppColor::BackgroundColor);
    gWriter.Print("Cancel");
} // NOLINT (readability/fn_size)

void CommandExecutor::DrawSelectiveMode()
{
    const int lineHeight = 40;
    const int stringHeightOffset = (lineHeight - 20) / 2;
    const int marginW = 30;
    const int marginH = 20;
    const int windowWidth = 400;

    //
    const int visibleCount = 10;
    const int stableLimit = visibleCount - 1;

    //ウィンドウオフセットと補間値の計算
    if (m_SelectTaskNum + m_ScrollOffsetNum < 0)
    {
        m_ScrollOffsetNum += -1 * (m_SelectTaskNum + m_ScrollOffsetNum);
    }
    if (m_SelectTaskNum + m_ScrollOffsetNum > stableLimit)
    {
        m_ScrollOffsetNum += -1 * (m_SelectTaskNum + m_ScrollOffsetNum - stableLimit);
    }
    m_InterpolatedScrollOffsetY = m_InterpolatedScrollOffsetY * 0.5f + (float)m_ScrollOffsetNum*lineHeight * 0.5f;

    //
    for (int i = 0; i < m_TaskList.size(); i++)
    {
        nn::util::Color4u8Type color = AppColor::DefaultTextColor;
        gWriter.SetCursor(m_DrawPosX + marginW, m_DrawPosY + marginH + i*lineHeight + stringHeightOffset + (int)m_InterpolatedScrollOffsetY);
        //カーソル描画
        if (m_SelectTaskNum == i && m_ProcessState == ProcessState::Standby)
        {
            color = AppColor::SelectedColor;
            gDrawer.DrawRect(m_DrawPosX, m_DrawPosY + marginH + i*lineHeight + (int)m_InterpolatedScrollOffsetY, 5, lineHeight, color);
        }
        //インジケータ描画
        if (m_TaskList.at(i)->GetState() == ExecuteItem::ExecuteState::Processing)
        {
            color = AppColor::ProcessingColor;
            int stringWidth = gWriter.CalculateStringWidth(m_TaskList.at(i)->GetName());
            m_Indicator.Draw(m_DrawPosX + marginW + stringWidth + 20, m_DrawPosY + marginH + i*lineHeight + (int)m_InterpolatedScrollOffsetY + 5, 30, AppColor::ProcessingColor);

        }
        //欄外か
        if (visibleCount >= i + m_ScrollOffsetNum && i + m_ScrollOffsetNum >= -1) {
            gWriter.SetTextColor(color);
            gWriter.Print(m_TaskList.at(i)->GetName());

        }
    }

    //ステートウィンドウ
    nn::util::Color4u8Type frameColor = AppColor::ProcessedColor;
    const char* stateHeader[] =
    {
        "State",
        "Process",
        "Result",
        "Error",
    };
    const int stateHeaderNum = sizeof stateHeader / sizeof stateHeader[0];
    //
    const int statePosX = m_DrawPosX + windowWidth + 50;
    const int statePosY = m_DrawPosY;
    const int windowHeght = lineHeight * stateHeaderNum + marginH;
    const int valueMarginW = 90;
    const int margin = 5;

    //ステート表示名
    std::string stateString;
    switch (m_ProcessState)
    {
    case ProcessState::Standby:
        stateString = "Standby";
        break;
    case ProcessState::Canceling:
        stateString = "Canceling";
        break;
    case ProcessState::Processing:
        stateString = "Processing";
        break;
    case ProcessState::Result:
        stateString = "Result";
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    //ステートウィンドウ項目描画
    for (int i = 0; i < stateHeaderNum; i++)
    {
        gWriter.SetTextColor(AppColor::DefaultTextColor);
        gWriter.SetCursor(statePosX + marginW, statePosY + marginH + lineHeight * i);
        gWriter.Print(stateHeader[i]);
        gWriter.SetTextColor(AppColor::ValueColor);
        gWriter.SetCursor(statePosX + marginW + valueMarginW, statePosY + marginH + lineHeight * i);
        switch (i)
        {
            case 0:
                gWriter.Print(stateString.c_str());
                break;
            case 1:
                gWriter.Print(m_LastTaskName.c_str());
                break;
            case 2:
                if (m_LastTaskState == ExecuteItem::ExecuteState::Succeeded)
                {
                    gWriter.SetTextColor(AppColor::BackgroundColor);
                    gWriter.Print(" Success ");
                    int labelWidth = gWriter.CalculateStringWidth(" Success ");
                    gDrawer.DrawRect(statePosX + marginW + valueMarginW, statePosY + marginH + lineHeight * i - 5 + 2, labelWidth, 30, AppColor::SuccessColor);
                }
                else
                if (m_LastTaskState == ExecuteItem::ExecuteState::Failed)
                {
                    gWriter.SetTextColor(AppColor::BackgroundColor);
                    gWriter.Print(" Failure ");
                    int labelWidth = gWriter.CalculateStringWidth(" Failure ");
                    gDrawer.DrawRect(statePosX + marginW + valueMarginW, statePosY + marginH + lineHeight * i - 5 + 2, labelWidth, 30, AppColor::ErrorColor);
                }
                break;
            case 3:
                gWriter.SetTextColor(AppColor::ErrorColor);
                gWriter.Print(m_LastTaskErrorDescription.c_str());
                break;
            default: NN_UNEXPECTED_DEFAULT;
        }
    }

    //ステートウィンドウ描画
    if (m_ProcessState == ProcessState::Result)
    {
        frameColor = AppColor::SelectedColor;
        gWriter.SetTextColor(frameColor);
        gWriter.SetCursor(statePosX + 10, statePosY + windowHeght + 10);
        gWriter.Print("Press A or B");
    }
    gDrawer.DrawFrame(statePosX, statePosY, windowWidth, windowHeght, frameColor);
    gDrawer.DrawFrame(statePosX + margin, statePosY + margin, windowWidth - margin * 2, windowHeght - margin * 2, frameColor);

} // NOLINT (readability/fn_size)

void CommandExecutor::Draw()
{
    if (m_ExecuteMode == ExecuteMode::Selective)
    {
        DrawSelectiveMode();
    }
    else
    if (m_ExecuteMode == ExecuteMode::Continuous)
    {
        DrawContinuousMode();
    }

}
