﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * インクルード
 */
#include "QREL.h"

/****************************************************************
 * 外部関数
 ****************************************************************/
/*
 * 関数名 QREL_DataAnalyze
 * 機能   型番、誤り訂正レベル、モード指示子およびデータを分析し摘要
 *          可能な型番を求める。
 * 引数   versioninfo(IN/OUT)     型番情報へのポインタ
 *                                型番情報が0のときは型番未指定とする。
 *                                型番情報が型番未指定のとき最適な型番
 *                                を返す。
 *        ecclevel(IN)            誤り訂正レベルへのポインタ
 *        mode(IN)                モード指示子へのポインタ
 *        userdata(IN)            ユーザデータへのポインタ
 *          chaininfo(IN)         連結情報のポインタ
 *                                連結モードでないときはNULLを指定する。
 * 戻り値 正常終了時0以外を異常終了時0を返す
 *          異常終了時のエラーコードはQREL_GetLastError関数で取得できる。
 *          エラーコードは以下。
 *          ERR_ILLEGALPARAM      指定されたモード指示子が不正
 *          ERR_VERSIONINFOERROR  指定された型番が不正
 *          ERR_TOOSMALL          指定されたシンボルにデータが格納できない
 * 説明   型番、誤り訂正レベル、モード指示子およびデータを分析し摘要
 *          可能な型番を求める。
 *          型番が指定されたときはデータが指定された型番、誤り訂正レベ
 *          ル、モード指示子で作成シンボルに格納できるか否かを調べる。
 *          型番が指定されていないときは誤り訂正レベル、モード指示子で
 *          データを格納しうる最小のシンボルの型番を求める。
 */
bool QREL_DataAnalyze(VERSIONINFO *versioninfo, ECCLEVEL *ecclevel, MODE *mode,
                       USERDATA *userdata,
                       CHAININFO *chaininfo)
{
    int32_t ncodes;
    int8_t *code = NULL;
    bool status = 0;

    QREL_SetLastError(ERR_NOERROR, true);
    if (versioninfo->version != 0) {
        status = QREL_DataEncode(versioninfo, ecclevel, mode,
                                  userdata, &ncodes,
                                  reinterpret_cast<uint8_t **>(&code),
                                  chaininfo);
        QRE_Free(code);
        code = NULL;
    }
    else {
        int32_t version;
        int32_t maxversion = VER_GetMaxVersion();
        for (version = 1; version <= maxversion; version++) {
            VERSIONINFO tempver;
            ERRNO err;
            tempver.version = version;
            code = NULL;
            status = QREL_DataEncode(&tempver, ecclevel, mode,
                                      userdata, &ncodes,
                                      reinterpret_cast<uint8_t **>(&code),
                                      chaininfo);
            QRE_Free(code);
            code = NULL;
            if (status) {
                versioninfo->version = tempver.version;
                break;
            }
            err = QREL_GetLastError();
            if (err != ERR_TOOSMALL)
                break;
        }
    }
    return status;
}
