﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/****************************************************************
 * QRC生成ライブラリヘッダファイル
 ****************************************************************/

#pragma once

#include <stddef.h>
#include <stdint.h>

/*
 * 定数定義
 */

enum ConstantValue_QREL
{
    /* マスクパターン参照子 */
    Mask000 = 0,
    Mask001 = 1,
    Mask010 = 2,
    Mask011 = 3,
    Mask100 = 4,
    Mask101 = 5,
    Maslk110 = 6,
    Mask111 = 7,
    MaskAny = 0xFF,

    /*
     * 型番情報テーブル用構造体
     */
    PopMaxNum = 46,            /* 最大位置合わせパターン数 */
    HighBlock = 0,               /* RSブロック情報上段 */
    LowBlock = 1,               /* RSブロック情報上段 */
    DefaultQuietZoneSize = 4,
    Mm = 8,                   /* RS code over GF(2**MM) - change to suit */
};
 /*
 * マクロ定義
 */
#define NN ((1 << Mm) - 1)      /* MM=8ならば255 */

/*
 * データ定義
 */
/* エラー番号 */
typedef enum {
    ERR_NOERROR = 0,            /* エラーなし */
    ERR_ILLEGALPARAM = 1,       /* 不正なパラメータ */
    ERR_TOOSMALL = 2,           /* シンボルのデータ領域不足 */
    ERR_NOMEMORY = 3,           /* ヒープ領域不足 */
    ERR_FILENOTFOUND = 4,       /* ファイルが見つからない */
    ERR_FILEREADERROR = 5,      /* ファイル読み込みエラー */
    ERR_FILEWRITEERROR = 6,     /* ファイル書き込みエラー */
    ERR_VERSIONINFOERROR = 7,   /* 型番情報取得エラー */
    ERR_DATAFORMATERROR = 8,    /* データフォーマットエラー */
    ERR_IMAGE2BIG = 9,
    ERR_ARGUMENT_ILLEGAL = 10
} ERRNO;

/* 誤り訂正レベル */
typedef enum {
    ECCL_L = 1,                 /* L 01 */
    ECCL_M = 0,                 /* M 00 */
    ECCL_Q = 3,                 /* Q 11 */
    ECCL_H = 2                  /* H 10 */
} ECCLEVEL;

/* 型番情報 */
typedef struct {
    int32_t version;            /* 型番 7, 8, 9, ... 10 */
} VERSIONINFO;

/* 形式情報 */
typedef struct {
    int8_t mp;                  /* マスクパターン */
    ECCLEVEL ecclevel;          /* 誤り訂正レベル */
} FORMATINFO;

/* 座標 (Coordinate) int32_t版 */
typedef struct {
    int32_t x;                      /* x座標 */
    int32_t y;                      /* y座標 */
} XYCOORD;

/* シンボル符号 */
typedef struct {
    int32_t ncodes;                 /* シンボル符号の個数 */
    uint8_t *code;                  /* シンボル符号の配列 */
    uint8_t **matrix;               /* シンボル符号の２次元配列 */
} SYMBOLCODE;

/* (個別の)RSブロック */
typedef struct {
    int32_t length;                 /* 1RSブロックのデータ数 */
    int32_t datalen;                /* データコード語数 */
    int32_t nblock;                 /* このデータが飛ばすべきブロック数 */
    int32_t num;                    /* このブロック内での位置*/
    uint8_t *data;                  /* データワード (length) */
} SINGLERSBLOCK;

/* RSブロック */
typedef struct {
    int32_t ndatawords;             /* 総データワード数 */
    int32_t nblocks;                /* RSブロック数 */
    uint8_t *data;                  /* データワード (総データワード数) */
    SINGLERSBLOCK *rsb;             /* (個別の)RSブロック */
} RSBLOCK;

/* RSブロック型情報 */
typedef struct {
    uint8_t AllCodeNum;             /* 総コード語数 */
    uint8_t DataCodeNum;            /* データコード語数 */
    int8_t ErrorCorrectNum;         /* 誤り訂正数 */
} BLOCKUNIT;

/* 誤り訂正レベル毎のデータ */
typedef struct {
    short ErrcorCodeNum;            /* 誤り訂正コード語数 */
    int8_t RSBlockNum;              /* 総ＲＳブロック数 */
    int8_t HiblockCount;            /* 上段ブロックの数 */
    int8_t LoblockCount;            /* 下段ブロックの数 */
    BLOCKUNIT BlockUnit[2];         /* 各段のRSブロック情報 */
} ERRCORUNIT;

/* モード指示子 */
typedef enum {
    MODE_ECI = 0x07,                /* ECI */
    MODE_NESTB = 0x0e,              /* 入れ子開始 */
    MODE_NESTE = 0x0f,              /* 入れ子終了 */
    MODE_FNCI1ST = 0x05,            /* FNCI １番目の位置 */
    MODE_FNCI2ND = 0x09,            /* FNCI ２番目の位置 */
    MODE_CHAIN = 0x03,              /* 連結 */
    MODE_NUM = 0x01,                /* 数字 */
    MODE_ALPHANUM = 0x02,           /* 英数字 */
    MODE_8BITSBYTE = 0x04,          /* ８ビットバイト */
    MODE_KANJI = 0x08,              /* 漢字 */
    MODE_TERM = 0x00,               /* 終端 */
    MODE_ANY = 0xFF                 /* 自動 */
} MODE;

/* データ列 */
typedef struct {
    int32_t length;
    int8_t *data;
} USERDATA;

/* 連結情報 */
typedef struct {
    int32_t seqno;                  /* 通し番号 1, 2, 3 */
    int32_t totalcount;             /* 全シンボル数 */
    uint8_t parity;                 /* パリティデータ */
} CHAININFO;

/* シンボル情報 */
typedef struct {
    int32_t cellsize;               /* セルサイズ */
    int32_t ncells;                 /* 1辺のセルの個数 */
    VERSIONINFO versioninfo;        /* 型番情報 */
    SYMBOLCODE symbolcode;          /* シンボル符号 */
    RSBLOCK rsblock;                /* RSブロック */
    USERDATA userdata;              /* ユーザーデータ */
    FORMATINFO formatinfo;          /* 形式情報 */
} SYMBOLINFO;

extern int32_t E;                   /* E = number of iparity symbols */

/*
 * 関数
 */
#ifdef __cplusplus
extern "C" {
#endif

/****************************************************************
 * ライブラリ関数
 ****************************************************************/

/* ※ここでDATASTREAMにユーザーデータを入れておく
 * 連結モードのパリティを計算する */
bool QREL_CalcChainParity(USERDATA *userdata, uint8_t *parity);
/* データ分析 ※型番、モード指示子、誤り訂正レベルは指定する場合代入しておく */
bool QREL_DataAnalyze(VERSIONINFO *versioninfo, ECCLEVEL *ecclevel, MODE *mode,
                       USERDATA *userdata,
                       CHAININFO *chaininfo);
/* データ符号化 */
bool QREL_DataEncode(VERSIONINFO *versioninfo, ECCLEVEL *ecclevel, MODE *mode,
                      USERDATA *userdata, int32_t *ncodes, uint8_t **code,
                      CHAININFO *chaininfo);
/* 誤り訂正符号の追加・・・データコード語列の生成（符号化）*/
bool QREL_AddErrorCorrectData(VERSIONINFO versioninfo, ECCLEVEL ecclevel,
                               uint8_t *code);
/* マトリックスデータに展開
 * ・・・最終データ列をマトリックスに展開する→マトリックス */
bool QREL_LoadToMatrix(VERSIONINFO versioninfo, int32_t ncodes,
                        uint8_t *code, uint8_t ***matrix);
/*
 * マスク処理
 * ※マスクパターンを指定する場合はここで代入しておく
 * ・・・マトリックスにマスクをかける→仮シンボル
 * マスクパターンの評価ロジック修正 */
bool QREL_SetMask(VERSIONINFO versioninfo, int8_t *mp, ECCLEVEL ecclevel,
                   uint8_t **matrix);
/* シンボル完成・・・最終シンボルの完成（２次元配列） */
bool QREL_GenerateQRCode(VERSIONINFO versioninfo, FORMATINFO formatinfo,
                          uint8_t **matrix);

/****************************************************************
 * 型番情報テーブルアクセス関数（VER_XXX）
 ****************************************************************/

/* 型番に定義された総データワード数を返す */
bool VER_GetAllDataWord(int32_t version, int32_t *pAllDataWordNum);
/* 型番に定義された位置合わせパターン情報を返す */
bool VER_GetPOPCoord(int32_t version, int32_t *popnum, int32_t *xarray, int32_t *yarray);
/* 型番・誤り訂正レベル毎に定義されたRSブロック情報を返す */
bool VER_GetRSInfo(int32_t version, ECCLEVEL ecclevel, ERRCORUNIT *rsinfo);
/* 型番毎に定義された一辺のセル数情報を返す*/
bool VER_GetnCellNum(int32_t version, int32_t *cellnum);
/* 最大型番を取得する */
int32_t VER_GetMaxVersion();

MODE AnalyzeMode(uint8_t *, uint32_t);
int32_t CalcDataBitLength(int32_t version, MODE mode, int32_t datalen);
int32_t CalcCharLengthFieldLength(int32_t version, MODE mode);

/****************************************************************
 * 共通関数（QREL_XXX）
 ****************************************************************/

/* エラー番号を取得する */
ERRNO QREL_GetLastError();
/* エラー番号を設定する */
void QREL_SetLastError(ERRNO lasterror, bool enforce);

void InitLastError();

/* encode BCH */
void encode_bch(int32_t al, int32_t dl, int8_t *d, int8_t *p);

void init_rs(int32_t e, int32_t b0);
void encode_rs(uint8_t data[], uint8_t bb[]);

#undef ISP_MEMORY_DEBUG
#ifdef ISP_MEMORY_DEBUG

#define QRE_Malloc(i) ISPMalloc(i, __FILE__, __LINE__, __func__)
#define QRE_Realloc(i,p) ISPRealloc(i, p, __FILE__, __LINE__, __func__)
#define QRE_Free(p) ISPFree(p, __FILE__, __LINE__, __func__)

void *ISPMalloc(size_t, const int8_t *, int32_t, const int8_t *);
void *ISPRealloc(void *, size_t, const int8_t *, int32_t, const int8_t *);
void ISPFree(void *, const int8_t *, int32_t, const int8_t *);

#else

void *QRE_Malloc(size_t);
void *QRE_Realloc(void *, size_t);
void QRE_Free(void *);

#endif

void InitHeap();

inline size_t round4(size_t l)
{
    return ((l + ((size_t)3)) & ~((size_t)3));
}

#ifdef __cplusplus
} /* extern "C" */
#endif
