﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.h"

#include <cstdlib>
#include <cstring>
#include <glv_viewutility.h>
#include <nn/nn_Assert.h>

#include "QCIT_ModalView.h"
#include "QCIT_RootSurface.h"
#include "QCIT_Image.h"

namespace qcit
{

/**
 * @brief       モーダルビューの定義です。
 */
namespace {

    // 実行情報の初期数
    static const nn::TimeSpan ZeroSpan;

} // end of unnamed namespace

ModalView::ModalView(const glv::Rect& rect) NN_NOEXCEPT
    :   glv::ViewContainer(rect),
        m_Timeout(),
        m_BeginTick(0),
        m_IsRunning(false),
        m_IsExiting(false)
{
    enable(glv::Visible | glv::DrawBack | glv::DrawBorder);
    anchor(glv::Place::CC);
}

ModalView& ModalView::SetTimeout(nn::TimeSpan timeout) NN_NOEXCEPT
{
    m_Timeout = timeout;
    return *this;
}

ModalView& ModalView::ExtendTimeout(nn::TimeSpan timeout) NN_NOEXCEPT
{
    m_Timeout = timeout;
    m_BeginTick = nn::os::GetSystemTick();
    return *this;
}

bool ModalView::IsExpired() const NN_NOEXCEPT
{
    bool expired = false;
    if( (m_IsRunning != false) && (m_Timeout > ZeroSpan) )
    {
        nn::os::Tick current = nn::os::GetSystemTick();
        nn::TimeSpan elapsed = nn::os::ConvertToTimeSpan(current - m_BeginTick);
        expired = (elapsed > m_Timeout);
    }
    return expired;
}

bool ModalView::IsTimeoutValid() const NN_NOEXCEPT
{
    return m_Timeout > ZeroSpan;
}

void ModalView::StartModal() NN_NOEXCEPT
{
    NN_ASSERT( m_IsRunning == false );

    if( m_IsRunning == false )
    {
        m_IsRunning = true;
        m_IsExiting = false;
        m_BeginTick = nn::os::GetSystemTick();
        enable(glv::Visible);
        OnActivate();
    }
}

void ModalView::EndModal() NN_NOEXCEPT
{
    NN_ASSERT( m_IsRunning != false );

    if( m_IsRunning != false )
    {
        m_IsRunning = false;
        m_IsExiting = false;
        OnDeactivate();
    }
}

/**
 * @brief       メッセージビューの定義です。
 */
namespace {

    // 実行情報の初期数
    static const int InitialActionCapacity = 4;

    // パッディング
    const glv::space_t MessageViewPaddingX = 32.f;
    const glv::space_t MessageViewPaddingY = 32.f;
    const glv::space_t MessageButtonPaddingX = 32.f;
    const glv::space_t MessageButtonPaddingY = 16.f;

} // end of unnamed namespace

MessageView::MessageView() NN_NOEXCEPT
    :   ModalView(glv::Rect(0)),
        m_MessagesTable("<", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ButtonsTable("", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ImageTable("", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ViewTable("<,x,>", 0, 0, glv::Rect(0)),
        m_IsClosable(true),
        m_IsCenteringButton(false)
{
    Initialize();
}

MessageView::MessageView(bool isClosable, bool isCenteringButton) NN_NOEXCEPT
    :   ModalView(glv::Rect(0)),
        m_MessagesTable("<", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ButtonsTable("", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ImageTable("0", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ViewTable("<,x,>", 0, 0, glv::Rect(0)),
        m_IsClosable(isClosable),
        m_IsCenteringButton(isCenteringButton)
{
    Initialize();
}

void MessageView::Initialize() NN_NOEXCEPT
{
    enable(glv::Property::DrawBorder | glv::Property::DrawBack);
    disable(glv::Property::FocusHighlight);
    m_Actions.reserve(InitialActionCapacity);

    if( m_IsClosable != false )
    {
        auto pLabel = new glv::Label("B: Close with doing nothing");
        pLabel->size(CommonValue::InitialFontSize);

        auto pTable = new glv::Table(">", MessageViewPaddingX, 0.0f, glv::Rect(0));
        *pTable << pLabel;
        pTable->arrange();

        m_ViewTable.arrangement("<,x,>,>");
        if (m_IsCenteringButton)
        {
            m_ViewTable.arrangement("<,x,>,x");
        }
        *this << (m_ViewTable << m_MessagesTable << m_ImageTable << pTable << m_ButtonsTable);
    }
    else
    {
        if (m_IsCenteringButton)
        {
            m_ViewTable.arrangement("<,x,x");
        }
        *this << (m_ViewTable << m_MessagesTable << m_ImageTable << m_ButtonsTable);
    }
}

MessageView::~MessageView() NN_NOEXCEPT
{
    m_Actions.clear();
}

void MessageView::Remove() NN_NOEXCEPT
{
    // 現状は m_ImageTable だけ remove すればいいが、子孫を全て remove する実装にしておく
    glv::View* pView = child;

    while ( nullptr != pView )
    {
        while ( nullptr != pView->child )
        {
            pView = pView->child;
        }

        glv::View* pRemovedView = pView;

        // 次に削除する要素に移動する
        if ( nullptr != pView->sibling )
        {
            pView = pView->sibling;
        }
        else if ( this != pView->parent )
        {
            pView = pView->parent;
        }
        else
        {
            pView = nullptr;
        }
        pRemovedView->remove();
    }
}

MessageView& MessageView::AddMessage(const std::string& message) NN_NOEXCEPT
{
    AddMessageT<std::string>(message);
    return *this;
}

MessageView& MessageView::AddMessage(const std::string& message, float fontSize) NN_NOEXCEPT
{
    AddMessageT<std::string>(message, fontSize);
    return *this;
}

// TORIAEZU: 名前は暫定, 不要なら後で削除する
glv::Label* MessageView::AddMessageReturnLabel(const std::string& message) NN_NOEXCEPT
{
    return AddMessageT<std::string>(message);
}

MessageView& MessageView::AddMessage(const glv::WideString& message) NN_NOEXCEPT
{
    AddMessageT<glv::WideString>(message);
    return *this;
}

template<typename StringType>
glv::Label* MessageView::AddMessageT(const StringType& message) NN_NOEXCEPT
{
    // ラベル作成
    auto pLabel = new glv::Label(message);
    pLabel->size(CommonValue::InitialFontSize);

    // レイアウト調整
    m_MessagesTable << pLabel;
    m_MessagesTable.arrange();
    m_ViewTable.arrange();
    extent(m_ViewTable.width(), m_ViewTable.height());

    return pLabel;
}

template<typename StringType>
glv::Label* MessageView::AddMessageT(const StringType& message, float fontSize) NN_NOEXCEPT
{
    // ラベル作成
    auto pLabel = new glv::Label(message);
    pLabel->size(fontSize);

    // レイアウト調整
    m_MessagesTable << pLabel;
    m_MessagesTable.arrange();
    m_ViewTable.arrange();
    extent(m_ViewTable.width(), m_ViewTable.height());

    return pLabel;
}

MessageView& MessageView::AddButton(const std::string& caption, ActionFunc pFunc, void* pParam) NN_NOEXCEPT
{
    AddButtonT<std::string>(caption, pFunc, pParam);
    return *this;
}

MessageView& MessageView::AddButton(const glv::WideString& caption, ActionFunc pFunc, void* pParam) NN_NOEXCEPT
{
    AddButtonT<glv::WideString>(caption, pFunc, pParam);
    return *this;
}

template<typename StringType>
glv::Button* MessageView::AddButtonT(const StringType& caption, ActionFunc pFunc, void* pParam) NN_NOEXCEPT
{
    // ラベル作成
    auto pLabel = new glv::Label(caption);
    pLabel->size(CommonValue::InitialFontSize).pos(glv::Place::CC).anchor(glv::Place::CC);

    // ボタン作成
    glv::Rect rect(pLabel->width() + MessageButtonPaddingX * 2, pLabel->height() + MessageButtonPaddingY * 2);
    auto pButton = new glv::Button(rect, true);
    *pButton << pLabel;
    pButton ->attach(ReceiveUpdateNotification, glv::Update::Clicked, this);

    // 配置方法指定
    const std::string align = m_ButtonsTable.arrangement() + "^";
    m_ButtonsTable.arrangement(align.c_str());

    // レイアウト調整
    m_ButtonsTable << pButton;
    m_ButtonsTable.arrange();
    m_ViewTable.arrange();
    extent(m_ViewTable.width(), m_ViewTable.height());

    // 実行情報を登録
    Action action;
    action.pButton = pButton;
    action.pFunc = pFunc;
    action.pParam = pParam;
    m_Actions.push_back(action);

    return pButton;
}

Icon* MessageView::AddImage( void* imageData, uint32_t imageDataSize, uint16_t imageWidth, uint16_t imageHeight ) NN_NOEXCEPT
{
    const size_t imagePadding = 4;
    auto pImageIcon = new Icon( imageWidth, imageHeight, imagePadding, imagePadding );
    if ( imageData != nullptr && imageDataSize > 0 )
    {
        pImageIcon->SetRawTexture( reinterpret_cast< unsigned char* >( imageData ), imageDataSize );
    }
    else
    {
        pImageIcon->ClearTexture();
    }
    // 配置方法指定
    const std::string align = m_ImageTable.arrangement() + "^";
    m_ImageTable.arrangement( align.c_str() );

    // レイアウト調整
    m_ImageTable << pImageIcon;
    m_ImageTable.arrange();
    m_ViewTable.arrange();
    extent( m_ViewTable.width(), m_ViewTable.height() );

    return pImageIcon;
}

glv::View* MessageView::focusableView() NN_NOEXCEPT
{
    if( m_Actions.size() > 0 )
    {
        auto it = m_Actions.end();
        return (--it)->pButton;
    }
    return nullptr;
}

void MessageView::OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    bool isExit = IsExpired();
    if( IsTimeoutValid() != false )
    {
        isExit = IsExpired();
    }
    else if( m_IsClosable != false )
    {
        {
            const auto& dpad = events.GetDebugPad();
            isExit = dpad.IsButtonDown<glv::DebugPadEventType::Button::B>();
        }
        if( isExit == false )
        {
            const auto& bpad = events.GetBasicPad(0);
            isExit = bpad.IsButtonDown<glv::BasicPadEventType::Button::B>();
        }
    }
    if( isExit != false )
    {
        ExitModal();
    }
}

void MessageView::ReceiveUpdateNotification(const glv::Notification& notification) NN_NOEXCEPT
{
    auto pView = notification.receiver<MessageView>();
    const void* pSender = notification.sender();

    auto& actions = pView->m_Actions;
    for ( auto it = actions.begin(); it != actions.end(); ++it )
    {
        if( pSender == reinterpret_cast<void*>(it->pButton) )
        {
            nn::TimeSpan timespan = ZeroSpan;
            ActionFunc pFunc = it->pFunc;
            if( pFunc != nullptr )
            {
                pFunc(it->pParam, timespan);
            }
            if( timespan > ZeroSpan )
            {
                pView->ExtendTimeout(timespan);
                pView->disable(glv::Visible);
            }
            else
            {
                if (it->isExitModal)
                {
                    pView->ExitModal();
                }
            }
            break;
        }
    }
}


/**
 * @brief       モーダルビューの親ビューの定義です。
 */
InvisibleWall::InvisibleWall() NN_NOEXCEPT
    :   glv::View(glv::Rect(0), glv::Place::TL),
        m_pRootContext(nullptr),
        m_pModalView(nullptr),
        m_IsDisposable(false),
        m_IsAutomaticExitAtBackgroundEnabled(false)
{
    disable(glv::Visible | glv::FocusHighlight | glv::DrawBorder);
    enable(glv::Controllable);
    stretch(1, 1);

    m_WallStyle.color.back.set(0, 0, 0, 0.9);
    style(&m_WallStyle);
}

InvisibleWall::~InvisibleWall() NN_NOEXCEPT
{
    EndModal();
}

void InvisibleWall::OnAttached(RootSurfaceContext* pContext) NN_NOEXCEPT
{
    m_pRootContext = pContext;
}

void InvisibleWall::OnDetached(RootSurfaceContext* pContext) NN_NOEXCEPT
{
    NN_UNUSED( pContext );

    EndModal();
    m_pRootContext = nullptr;
}

void InvisibleWall::DisplayModal(ModalView* pModalView) NN_NOEXCEPT
{
    NN_ASSERT( nullptr != pModalView );

    bringToFront();

    // 登録
    pModalView->pos(glv::Place::CC).anchor(glv::Place::CC);
    *this << pModalView;

    // 可視化
    enable(glv::Visible);
    pModalView->StartModal();

    // フォーカス
    auto pFocusView = pModalView->focusableView();
    if( pFocusView != nullptr )
    {
        m_pRootContext->setFocus(pFocusView);
    }
    m_pModalView = pModalView;
}

void InvisibleWall::StartModal(ModalView* pModalView, bool isDisposable, bool isDelayExec, bool isAutomaticExitAtBackgroundEnabled) NN_NOEXCEPT
{
    if( isDelayExec == false )
    {
        NN_ASSERT( m_pModalView == nullptr );
    }

    if( pModalView != nullptr )
    {
        m_IsAutomaticExitAtBackgroundEnabled = isAutomaticExitAtBackgroundEnabled;
        m_IsDisposable = isDisposable;
        m_IsDelayExec = isDelayExec;

        if (isDelayExec == false)
        {
            DisplayModal(pModalView);
        }
        else
        {
            m_pNextModalView = pModalView;
        }
    }
}

void InvisibleWall::EndModal() NN_NOEXCEPT
{
    EndModal( m_pModalView );
}

void InvisibleWall::EndModal( ModalView* pModalView ) NN_NOEXCEPT
{
    if( pModalView != nullptr )
    {
        // 不可視化
        pModalView->EndModal();
        disable(glv::Visible);

        // 破棄
        pModalView->remove();
        if( (m_IsDisposable != false) && ( pModalView->dynamicAlloc() != false) )
        {
            delete pModalView;
        }
        m_pModalView = nullptr;
        m_IsDisposable = false;
        m_IsAutomaticExitAtBackgroundEnabled = false;
    }
}

void InvisibleWall::ForceGainFocus(ModalView* pModalView) NN_NOEXCEPT
{
    // 元々何故か pModalView を null 判定にしか使用していなかったので、m_pModalView のnull判定条件追加。
    // 多分引数いらない？ もしくは、引数を利用するのが正しい？
    if( pModalView != nullptr && nullptr != ( pModalView = m_pModalView ) )
    {
        // フォーカス
        auto pFocusView = pModalView->focusableView();
        if( pFocusView != nullptr )
        {
            m_pRootContext->setFocus(pFocusView);
        }
    }
}

void InvisibleWall::OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    auto pModalView = m_pModalView;
    if( nullptr != pModalView )
    {
        pModalView->OnLoopBeforeSceneRenderer(context, events);
    }
}

void InvisibleWall::OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    auto pModalView = m_pModalView;
    if( pModalView != nullptr )
    {
        pModalView->OnLoopAfterSceneRenderer(context, events);
        if( pModalView->IsRunnable() == false )
        {
            EndModal( pModalView );
            if ( m_IsDelayExec == true )
            {
                ModalView *pNextView;
                m_IsDelayExec = false;
                if ( nullptr != ( pNextView = m_pNextModalView ) )
                {
                    m_pNextModalView = nullptr;
                    DisplayModal( pNextView );
                }
            }
        }
    }
}

bool InvisibleWall::onEvent(glv::Event::t events, glv::GLV& context) NN_NOEXCEPT
{
    // フォーカス移動の操作
    int flags = glv::toMoveFlags(events, context);
    if( (flags & glv::MoveArrowMask) != 0 )
    {
        View* prev = context.focusedView();
        View* next = nullptr;
        if( (glv::isVisible(prev) == false) || (glv::isFocusable(prev) == false) )
        {
            prev = nullptr;
        }

        // 現在フォーカス表示されていない
        if( prev == nullptr )
        {
            auto pModalView = m_pModalView;
            if( pModalView != nullptr )
            {
                next = pModalView->focusableView();
            }
            if( next != nullptr )
            {
                context.setFocus(next);
            }
        }
    }
    // InvisibleWall で処理された際は親にイベントを移譲しない
    return false;
}

bool InvisibleWall::IsModalViewRunning() NN_NOEXCEPT
{
    return nullptr != m_pModalView;
}

void InvisibleWall::OnChangeIntoBackground() NN_NOEXCEPT
{
    ModalView* pModalView;
    if ( m_IsAutomaticExitAtBackgroundEnabled && nullptr != ( pModalView = m_pModalView ) )
    {
        EndModal( pModalView );
    }
}

void InvisibleWall::OnChangeIntoForeground() NN_NOEXCEPT
{
}

} // ~namespace qcit
