﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <glv_CustomVerticalListView.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/os/os_ThreadTypes.h>

#include "QCIT_RootSurface.h"
#include "QCIT_InstallContentInfo.h"
#include "QCIT_CustomModalView.h"
#include "QCIT_AppInfo.h"

namespace qcit
{
    class Refreshable
    {
    private:
        std::atomic< bool > m_RefreshOnNext;
    protected:
        Refreshable() NN_NOEXCEPT
            : m_RefreshOnNext(false)
        {
        }
    public:
        void RequireRefresh() NN_NOEXCEPT
        {
            m_RefreshOnNext.store(true);
        }
            bool IsRefreshRequired() const NN_NOEXCEPT
        {
            return m_RefreshOnNext.load();
        }
            void ClearRefreshRequest() NN_NOEXCEPT
        {
            m_RefreshOnNext.store(false);
        }
        virtual void Refresh() NN_NOEXCEPT = 0;
    };

    class Scene
        : public glv::Group
        , public Refreshable
    {
    public:
        glv::View* GetFirstFocusedView() const NN_NOEXCEPT
        {
            return m_pFirstFocusedView;
        }

        void SetLastFocusedView(View* pView) NN_NOEXCEPT
        {
            NN_ASSERT(nullptr != pView);
            m_pLastFocusedView = pView;
        }

        glv::View* GetLastFocusedView() const NN_NOEXCEPT
        {
            return m_pLastFocusedView;
        }

        void SetBbuttonCallback(std::function< void() > BbuttonCallback) NN_NOEXCEPT
        {
            m_BbuttonCallback = BbuttonCallback;
        }

    protected:

        explicit Scene(const glv::Rect& rect, std::function< void() > BbuttonCallback = nullptr) NN_NOEXCEPT
            : glv::Group(rect), m_pFirstFocusedView(nullptr), m_pLastFocusedView(nullptr), m_BbuttonCallback(BbuttonCallback)
        {
            // Set this scene to be able to detect clicks
            this->enable(glv::HitTest | glv::GestureDetectability);
            // Attach a Click handler to detect if the B button was pressed
            this->attach([](const glv::Notification& n)->void {
                auto p = n.sender< Scene >();

                auto& g = reinterpret_cast<glv::GLV&>(p->root());
                if (g.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::B >() ||
                    g.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::B >())
                {
                    // Call the given function when the B button is detected
                    if (p->m_BbuttonCallback != nullptr)
                    {
                        p->m_BbuttonCallback();
                    }
                }
            },
                glv::Update::Clicked, this);

        }

        // シザーボックスの領域設定
        static const int HeaderRegion = 32;
        static const int FooterRegion = 32;

        glv::View*  m_pFirstFocusedView; // 表示開始後に最初にフォーカスするビュー
        glv::View*  m_pLastFocusedView;  // 別シーンに遷移する直前のフォーカス
        std::function< void() > m_BbuttonCallback;
    };
};

namespace {
    const int64_t MaxDirectoryNameLength = 300;  //!< アプリケーション名の最大文字数（終端文字を含む）

    enum InstallState {
        InstallState_NotStarted,
        InstallState_Running,
        InstallState_Committed,
        InstallState_Completed,
        InstallState_Failed,
        InstallState_WaitUserAction,
        InstallState_Uninstall,
    };
}

namespace qcit
{

/**
 * @brief SD カード内のアプリケーションプロパティ型です。
 */
struct FilePropertyType
{
    bool* checkedPtr;
    glv::WideCharacterType id[ 19 ];
    glv::WideCharacterType type[ 4 ];
    glv::WideCharacterType name[ MaxDirectoryNameLength ];
    glv::WideCharacterType size[ 32 ];

    FilePropertyType() : checkedPtr(nullptr) {}
};
/**
 * @brief リストビュー
 */
class ContentListView : public glv::CustomVerticalListView< FilePropertyType >
{
public:

    //----------------------------------------------------------

    typedef glv::CustomVerticalListView< FilePropertyType > ParentType;

    static const glv::space_t HorizontalMargin_Checked;
    static const glv::space_t HorizontalMargin_Id;
    static const glv::space_t HorizontalMargin_Type;
    static const glv::space_t HorizontalMargin_Name;  //!< リスト要素中の[ プログラムName ]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_Name;  //!< リスト要素中の[ プログラムName ]表示横幅( ピクセル単位 )

    /**
     * @brief コンストラクタです。
     */
    explicit ContentListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT
        : CustomVerticalListView( parentClipRegion )
    {
        SetTouchAndGo( true );
        glv::Style* pStyle = new glv::Style();
        pStyle->color = glv::Style::standard().color;
        pStyle->color.selection.set( 0.1f, 0.85f, 0.2f );
        style( pStyle );
        font().size(21.f);

        BuildUtf16< 6 >(m_CheckOffStr, "[   ]");
        BuildUtf16< 6 >(m_CheckOnStr, "[ * ]");
    }

private:
    glv::WideCharacterType m_CheckOffStr[6];
    glv::WideCharacterType m_CheckOnStr[6];

protected:
    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
     */
    virtual void OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT NN_OVERRIDE
    {
        this->font().getBounds( outWidth, outHeight, item.name );
        outWidth = this->width();
    }

    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
     */
    virtual void OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( index );
        glv::Font& font = this->font();

        glv::space_t outWidth, outHeight;

        glv::WideCharacterType* checkStrPtr = m_CheckOffStr;
        if (item.checkedPtr != nullptr && *(item.checkedPtr) == true)
        {
            checkStrPtr = m_CheckOnStr;
        }

        glv::space_t posCheck = contentRegion.left() + HorizontalMargin_Checked;
        font.render(checkStrPtr, posCheck, contentRegion.top());

        glv::space_t posId = contentRegion.left() + HorizontalMargin_Id;
        font.render(item.id, posId, contentRegion.top());

        glv::space_t posType = contentRegion.left() + HorizontalMargin_Type;
        font.render(item.type, posType, contentRegion.top());

        glv::space_t posName = contentRegion.left() + HorizontalMargin_Name;
        font.render( item.name, posName, contentRegion.top() );

        font.getBounds( outWidth, outHeight, item.size );
        const glv::space_t sizeExpect = contentRegion.right() - ( outWidth + ( paddingX() * 2 ) + 4.f );
        const glv::space_t sizeLimit = posName + HorizontalLength_Name + 12.f;
        font.render( item.size, ( sizeExpect < sizeLimit ) ? sizeLimit : sizeExpect, contentRegion.top() );
    }
};

/**
 * @brief アプリケーションインストールシーン
 */
class InstallScene : public Scene
{
public:
    /**
     * @brief コンストラクタです。
     */
    InstallScene( RootSurfaceContext* pRoot, Scene* pParentScene = nullptr) NN_NOEXCEPT;

    void FinalizeProperties() NN_NOEXCEPT;

    virtual void Refresh() NN_NOEXCEPT;

    /**
    * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
    */
    glv::View* GetFocusableChild() NN_NOEXCEPT;

    nn::Result MakeNspList() NN_NOEXCEPT;

    /**
     * @brief       インストール進捗表示を更新します。
     */
    void UpdateInstallProgress() NN_NOEXCEPT;

    void InitialCheckProcess() NN_NOEXCEPT;

private:
    /**
     * @brief アプリケーションプロパティコレクションを登録します。
     * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
     */
    void EntryProperties( const ContentListView::CollectionType* pNsps ) NN_NOEXCEPT;

    nn::Result EnumerateFiles( std::vector< FilePropertyType >& entries, const char* directoryPath ) NN_NOEXCEPT;

    /**
     * @brief ダイアログを表示して確認を取った後、全 nsp のインストールを行います。
     */
    void ConfirmInstall( const FilePropertyType* pProperty = nullptr ) NN_NOEXCEPT;

    /**
     * @brief 選択された nsp ファイルのインストールを行います。
     */
    void InstallNspFile() NN_NOEXCEPT;

    /**
     * @brief インストール進捗画面の Close ボタンや Next ボタン押下時のコールバック関数です。
     */
    void CloseButtonCallback() NN_NOEXCEPT;

    /**
     * @brief インストール進捗画面の Cancel ボタン押下時のコールバック関数です。
     */
    void CancelButtonCallback() NN_NOEXCEPT;

    /**
     * @brief インストール進捗画面の Next ボタン押下時のコールバック関数です。
     */
    void NextButtonCallback() NN_NOEXCEPT;

    /**
     * @brief nsp ファイルをインストールします。
     */
    nn::Result ExecuteInstall( const char* filePath ) NN_NOEXCEPT;

    /**
     * @brief InstallTask 実行の事前準備を行います。
     */
    void RecreateApplicationRecord( nn::ncm::ApplicationId applicationId, bool isRecordChanged ) NN_NOEXCEPT;

    /**
     * @brief インストール結果を設定します。
     */
    void SetInstallResult( nn::Result result ) NN_NOEXCEPT;

    /**
     * @brief インストールのトリガをかけるスレッドのメイン関数です。
     */
    static void InstallTriggerThreadFunc( void* args ) NN_NOEXCEPT;

    /**
     * @brief インストールトリガスレッドを破棄します。
     */
    void DestroyInstallTriggerThread() NN_NOEXCEPT;

    /**
     * @brief インストール処理を別スレッドで実行します。
     */
    nn::Result StartAndWaitForExitInstallExecutionThread( nn::ncm::InstallTaskBase* task ) NN_NOEXCEPT;

    void ShowImageHash() NN_NOEXCEPT;

    void UninstallContents() NN_NOEXCEPT;

    using SkipIndexList = std::map<nn::Bit64, std::set<int>>;
    nn::Result UninstallContentsDetail(nn::Bit64 inQCITId,
        std::set<nn::Bit64>& inSkipIdSet, SkipIndexList& inSkipIndexList) NN_NOEXCEPT;

    void UninstallAocOnly(nn::Bit64 inTargetId, std::set<int>& inSkipIndexSet) NN_NOEXCEPT;

    void InstallProcess() NN_NOEXCEPT;

    void CheckAllItems() NN_NOEXCEPT;

    void ConfirmReboot() NN_NOEXCEPT;

    void SetKeyAssign() NN_NOEXCEPT;

    void CheckSdCardStatus() NN_NOEXCEPT;
    void CheckDebugMode() NN_NOEXCEPT;

    void FormatSdCardAndRebootProcess() NN_NOEXCEPT;

    void ReadApplicationInfoData() NN_NOEXCEPT;

    // (SIGLO-76759) NAND空き容量チェック関連のメンバ関数の追加
    void SetRequiredNandSpaceSizeSetting() NN_NOEXCEPT;
    int64_t GetNandFreeSpaceSize() NN_NOEXCEPT;
    void UninstallContent(const std::string& inFileName) NN_NOEXCEPT;

    glv::space_t SetPartsForHeaderArea() NN_NOEXCEPT;
    void SetContentListArea(glv::space_t inTopBeginPos) NN_NOEXCEPT;
    void SetPartsForFooterArea() NN_NOEXCEPT;

private:
    qcit::Button*                                m_pButtonAllCheckboxes;
    qcit::Button*                                m_pButtonCheckedInstall;
    ContentListView::CollectionType*             m_pNsps;
    ContentListView*                             m_pNspListView;
    glv::Label*                                  m_pLabelNoNsp;

    // 内部プロパティ用に定義
    struct InternalFilePropertyType
    {
        bool* checkedPtr;
        std::string id;
        std::string type;
        std::string name;
        std::string size;
        std::string applicationId;

        InternalFilePropertyType() : checkedPtr(nullptr) {}
    };
    std::vector< InternalFilePropertyType >      m_FileList;

    RootSurfaceContext*                          m_pRootSurface;

    // TORIAEZU: インストール状況表示関連は全て InstallScene のメンバ変数にしておく
    nn::os::ThreadType                           m_InstallTriggerThread;
    nn::os::ThreadType                           m_InstallExecThread;
    nn::ns::ApplicationInstallTask*              m_pInstallTask;
    nn::fs::FileHandle                           m_Handle;
    bool                                         m_IsInstallAll;
    nn::os::Tick                                 m_StartTick;
    InstallView*                                 m_pInstallView;         //!< インストールの進捗表示用のモーダル
    InstallState                                 m_InstallState;         //!< インストール状態
    nn::ncm::StorageId                           m_TargetStorageId;
    bool                                         m_IsLastTarget;
    bool                                         m_RequestsCancel;       //!< キャンセル要求(InstallAll 時のみ)
    bool                                         m_IsSdCardStatusInvalid;
    bool                                         m_IsAbortCheckQuestDevice;
    bool                                         m_DebugMode;
    bool                                         m_IsNandFreeSpaceSizeCheckRequired;

    nn::os::Mutex                                m_CheckItemMutex;

    std::string m_UninstallingContentName;
    size_t m_AllContentCountForDeleting;
    size_t m_UninstalledContentCount;

    ContentDataContainer m_ContentInfoList;
    std::set<std::string> m_QuestMenuFileNameSet;

    int64_t m_AllContentSizeForInstalling;
    int64_t m_InstalledContentSize;

    int64_t m_RequiredNandFreeSpaceSize;

    AppInfoData m_ApplicationInfoData;
};

} // ~namespace qcit
