﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.h"
#include <memory>
#include <sstream>
#include <mutex> // for std::lock_guard<>

#include <nn/nn_BitTypes.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Directory.h>
#include <nn/ns/ns_ApplicationEntitySystemApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/ns/ns_ApplicationContentMetaApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/rid/rid_StartUpApi.h>

#include <nn/oe/oe_PowerStateControlApi.private.h>

#include "QCIT_ApplicationInstaller.h"
#include "DevMenuCommand_InstallUtil.h"

#include "QCIT_FsUtilities.h"
#include "QCIT_StringUtil.h"

#include "QrCodeEncoder/QCIT_CreateQrCode.h"


namespace qcit
{
    const float CommonValue::InitialFontSize = 25.0f;

// ----------------------------------------------------------------------------------------------

    std::string GetDelimitedNumberString(int64_t number) NN_NOEXCEPT
    {
        std::vector< int > separateNumber;

        while (number / 1000)
        {
            separateNumber.push_back(number % 1000);
            number /= 1000;
        }

        std::stringstream sstream;
        sstream << number;
        for (std::vector< int >::reverse_iterator i = separateNumber.rbegin();
            i < separateNumber.rend(); i++)
        {
            char temp[8];
            nn::util::SNPrintf(temp, sizeof(temp), ",%03lld", *i);
            sstream << temp;
        }

        return sstream.str();
    }

// ----------------------------------------------------------------------------------------------

    class Stopwatch
    {
    public:
        Stopwatch() : m_IsStarted(false)
        {
        }

        ~Stopwatch()
        {
        }

        void Start()
        {
            if (m_IsStarted == false)
            {
                m_IsStarted = true;
                m_Start = nn::os::GetSystemTick().ToTimeSpan();
            }
        }

        void Stop()
        {
            if (m_IsStarted == true)
            {
                m_Stop = nn::os::GetSystemTick().ToTimeSpan();
                m_IsStarted = false;
            }
        }

        int64_t GetMilliSecond()
        {
            if (m_IsStarted == true)
            {
                return -1;
            }

            return (m_Stop - m_Start).GetMilliSeconds();
        }

        int64_t GetSecond()
        {
            if (m_IsStarted == true)
            {
                return -1;
            }

            return (m_Stop - m_Start).GetSeconds();
        }

    private:
        bool m_IsStarted;
        nn::TimeSpan m_Start;
        nn::TimeSpan m_Stop;
    };

// ----------------------------------------------------------------------------------------------

const glv::space_t ContentListView::HorizontalMargin_Checked = 2.0f;
const glv::space_t ContentListView::HorizontalMargin_Id = 45.5f;
const glv::space_t ContentListView::HorizontalMargin_Type = 304.0f;
const glv::space_t ContentListView::HorizontalMargin_Name = 355.0f;
const glv::space_t ContentListView::HorizontalLength_Name = 700.0f;

namespace {
    const size_t InstallBufferSize = 16 * 1024 * 1024;
    const size_t MaxComfirmNum = 8;

    const size_t MaxModalMessageLength    = 50;
    const size_t MaxCatalogFileNameLength = 46;

    // リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
    static const glv::space_t ListMarginL = 16.0f;
    static const glv::space_t ListMarginR = 16.0f;
    static const glv::space_t ListMarginY = 2.0f;

    /**
     * @brief       ファイル名に指定した拡張子( '.' を含む)が含まれるかをチェックします。
     */
    bool HasExtension( const char* str, const char* extension ) NN_NOEXCEPT
    {
        if ( strlen( str ) <= strlen( extension ) + 1 )
        {
            return false;
        }
        return ( nn::util::string_view( str ).substr( strlen( str ) - strlen( extension ) ) == nn::util::string_view( extension ) );
    }
}

// インストールスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_InstallStack[ 32 * 1024 ];

void ConvertWideStringToString( char* pOutStr, const glv::WideCharacterType* fileName ) NN_NOEXCEPT
{
    int fileListLength = 0;
    const uint16_t* pFileList = reinterpret_cast< const uint16_t* >( fileName );
    nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8( &fileListLength, pFileList, MaxDirectoryNameLength );
    nn::util::ConvertStringUtf16NativeToUtf8( pOutStr, fileListLength, pFileList );
}

/**
 * @brief コンストラクタです。
 */
InstallScene::InstallScene(RootSurfaceContext* pRoot, Scene* pParentScene) NN_NOEXCEPT
: Scene(pRoot->GetSurfacePageRect()),
    m_pButtonAllCheckboxes( nullptr ),
    m_pButtonCheckedInstall( nullptr ),
    m_pNsps( nullptr ),
    m_pNspListView( nullptr ),
    m_pLabelNoNsp( nullptr ),
    m_pRootSurface(pRoot),
    m_pInstallTask( nullptr ),
    m_IsInstallAll( false ),
    m_pInstallView( nullptr ),
    m_InstallState( InstallState_NotStarted ),
    m_IsLastTarget( false ),
    m_RequestsCancel( false ),
    m_IsSdCardStatusInvalid( false ),
    m_IsAbortCheckQuestDevice( true ),
    m_DebugMode( false ),
    m_CheckItemMutex(false),
    m_AllContentCountForDeleting(1),
    m_UninstalledContentCount(0),
    m_AllContentSizeForInstalling(0),
    m_InstalledContentSize(0)

{
    // ヘッダーエリアの構築
    auto bottomPos = this->SetPartsForHeaderArea();

    // コンテンツリストエリアの構築
    this->SetContentListArea(bottomPos);

    // フッターエリアの構築
    this->ReadApplicationInfoData();
    this->SetPartsForFooterArea();

    m_pFirstFocusedView = m_pNspListView;

    this->SetKeyAssign();

    this->CheckSdCardStatus();
    this->CheckDebugMode();
    // (SIGLO-76759) 空き容量指定の確認
    this->SetRequiredNandSpaceSizeSetting();

    this->MakeNspList();
}

glv::space_t InstallScene::SetPartsForHeaderArea() NN_NOEXCEPT
{
    const auto buttonYPos = 10.0f;

    // 全コンテンツ選択ボタン
    auto pButtonAllCheckboxes = new qcit::Button("L: Check All Items", [&] { this->CheckAllItems(); }, CommonValue::InitialFontSize, 16.0f);
    {
        pButtonAllCheckboxes->anchor(glv::Place::TL).pos(glv::Place::TL, 60.0f, buttonYPos);
        m_pButtonAllCheckboxes = pButtonAllCheckboxes;
        *this << pButtonAllCheckboxes;
    }

    // インストール実行ボタン
    auto pButtonCheckedInstall = new qcit::Button("Y: Install Checked Item", [&] { m_IsInstallAll = false; ConfirmInstall(); }, CommonValue::InitialFontSize, 16.0f);
    {
        pButtonCheckedInstall->anchor(glv::Place::TL).pos(glv::Place::TL, pButtonAllCheckboxes->right() + 30.0f, buttonYPos);
        m_pButtonCheckedInstall = pButtonAllCheckboxes;
        *this << pButtonCheckedInstall;
    }

    // ハッシュ表示ボタン
    auto pHashInfoButton = new qcit::Button("X: Show Selected Hash", [&] { ShowImageHash(); }, CommonValue::InitialFontSize, 16.0f);
    {
        pHashInfoButton->anchor(glv::Place::TL).pos(glv::Place::TL, pButtonCheckedInstall->right() + 70.0f, buttonYPos);
        *this << pHashInfoButton;
    }

    // 再起動ボタン
    auto pRebootButton = new qcit::Button("+: Reboot", [&] { ConfirmReboot(); }, CommonValue::InitialFontSize - 4.0f, 16.0f);
    {
        pRebootButton->anchor(glv::Place::TL).pos(glv::Place::TL, pHashInfoButton->right() + 50.0f, buttonYPos);
        *this << pRebootButton;
    }

    return pButtonAllCheckboxes->bottom();
}

void InstallScene::SetContentListArea(glv::space_t inTopBeginPos) NN_NOEXCEPT
{
    // シザーボックスの領域設定
    static const glv::space_t InstallSceneHeaderRegion = 80.0f;
    static const glv::space_t InstallSceneFooterRegion = 48.0f;

    // ラベルヘッダ
    auto pHeaderId = new glv::Label("ID", glv::Label::Spec(glv::Place::TL, ListMarginL + ContentListView::HorizontalMargin_Id + 5.0f, inTopBeginPos + 4.0f, CommonValue::InitialFontSize));
    auto pHeaderType = new glv::Label("Type", glv::Label::Spec(glv::Place::TL, ListMarginL + ContentListView::HorizontalMargin_Type, inTopBeginPos + 4.0f, CommonValue::InitialFontSize - 4.0f));
    auto pHeaderName = new glv::Label("FileName", glv::Label::Spec(glv::Place::TL, ListMarginL + ContentListView::HorizontalMargin_Name + 8.0f, inTopBeginPos + 4.0f, CommonValue::InitialFontSize));
    auto pHeaderSize = new glv::Label("Size(Byte)", glv::Label::Spec(glv::Place::TR, -(ListMarginR + 16.0f), pHeaderName->top(), CommonValue::InitialFontSize));
    {
        *this << pHeaderId;
        *this << pHeaderType;
        *this << pHeaderName;
        *this << pHeaderSize;
    }

    // リスト
    auto pListContainer = new glv::ScissorBoxView(0, InstallSceneHeaderRegion, width(), height() - (InstallSceneHeaderRegion + InstallSceneFooterRegion));
    {
        glv::Rect clipRegion(ListMarginL, ListMarginY, pListContainer->width() - (ListMarginL + ListMarginR), pListContainer->height() - (ListMarginY * 2));
        auto pNspListView = new ContentListView(clipRegion);

        pNspListView->attach(
            [](const glv::Notification& n)->void {
            auto pScene = n.receiver< InstallScene >();
            auto* pProperty = n.data< ContentListView::ItemType >();
            if (nullptr != pScene && nullptr != pProperty)
            {
                std::lock_guard< nn::os::Mutex > guard(pScene->m_CheckItemMutex);
                if (pProperty->checkedPtr != nullptr)
                {
                    // 値を反転させる
                    *(const_cast<FilePropertyType*>(pProperty)->checkedPtr) = !(*(pProperty->checkedPtr));
                }
            }
        },
            glv::Update::Action,
            this
            );

        m_pNspListView = pNspListView;
        *pListContainer << pNspListView;

        // NSP ファイルなしメッセージ
        auto pLabelNoNsp = new glv::Label("No nsp is found.", glv::Label::Spec(glv::Place::CC, 0.f, 0.f, 32.f));
        {
            m_pLabelNoNsp = pLabelNoNsp;
            *pListContainer << pLabelNoNsp;
        }
        *this << pListContainer;
    }
}

void InstallScene::SetPartsForFooterArea() NN_NOEXCEPT
{
    std::string applicationIdLabelStr = "Application ID: " + m_ApplicationInfoData.applicationId;
    auto pAppIdLabel = new glv::Label(applicationIdLabelStr, glv::Label::Spec(glv::Place::BL, ListMarginL, -10.f, CommonValue::InitialFontSize - 4.0f));
    *this << pAppIdLabel;

    std::string versionLabelStr = "Version: " + m_ApplicationInfoData.version;
    auto pVersionLabel = new glv::Label(versionLabelStr, glv::Label::Spec(glv::Place::BL, pAppIdLabel->right() + 22.f, -10.f, CommonValue::InitialFontSize - 4.0f));
    *this << pVersionLabel;

    auto pAButtonOperationLabel = new glv::Label("A: Check Item", glv::Label::Spec(glv::Place::BR, -(ListMarginL + 16.f), -8.f, CommonValue::InitialFontSize));
    *this << pAButtonOperationLabel;
}

void InstallScene::ReadApplicationInfoData() NN_NOEXCEPT
{
    const auto qcitInfoPath = std::string(RomMountName) + ":/QCIT_ID.txt";
    ApplicationInfo info;
    info.ReadInfo(qcitInfoPath);
    info.GetData(m_ApplicationInfoData);
}

void InstallScene::SetKeyAssign() NN_NOEXCEPT
{
    // キー操作のアサイン
    this->attach([](const glv::Notification& n)->void {
        auto p = n.sender< InstallScene >();
        auto& g = reinterpret_cast<glv::GLV&>(p->root());
        if (g.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::X >() ||
            g.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::X >())
        {
            p->ShowImageHash();
        }
        else if (g.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::Y >() ||
            g.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::Y >())
        {
            p->m_IsInstallAll = false;
            p->ConfirmInstall();
        }
        else if (g.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::L >() ||
            g.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::L >())
        {
            p->CheckAllItems();
        }
        else if (g.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::Start >() ||
            g.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::Start >())
        {
            p->ConfirmReboot();
        }
    },
        glv::Update::Clicked, this);
}

void InstallScene::CheckSdCardStatus() NN_NOEXCEPT
{
    auto result = nn::ns::CheckSdCardMountStatus();
    if (nn::ns::ResultSdCardNoOwnership::Includes(result) ||
        nn::ns::ResultSdCardFileSystemCorrupted::Includes(result) ||
        nn::ns::ResultSdCardDatabaseCorrupted::Includes(result))
    {
        // SD カードが挿入されていて、SDカードが利用できない状態であれば、
        // フォーマットを実行し、再起動を促すメッセージを表示するフラグを設定する
        m_IsSdCardStatusInvalid = true;
    }
}

void InstallScene::CheckDebugMode() NN_NOEXCEPT
{
    const auto debugModeFilePath = std::string(RomMountName) + ":/DebugMode.txt";
    if (fsutil::IsExistPath(debugModeFilePath.c_str()))
    {
        // ひとまずファイルの存在有無のみで判断する
        m_DebugMode = true;
    }
}

// (SIGLO-76759) 空き容量指定のファイル読み込み処理
void InstallScene::SetRequiredNandSpaceSizeSetting() NN_NOEXCEPT
{
    m_IsNandFreeSpaceSizeCheckRequired = false;
    m_RequiredNandFreeSpaceSize = 0;

    const auto requiredSizeFilePath = std::string(RomMountName) + ":/RequiredFreeNandSize.txt";
    if (fsutil::IsExistPath(requiredSizeFilePath.c_str()) == false)
    {
        // 指定パスが存在しない場合、Nandの空き容量チェック処理は実施しない
        return;
    }

    std::string dataStr;
    auto result = fsutil::GetDataOfFile(requiredSizeFilePath, dataStr);
    if (result.IsFailure())
    {
        // ファイルの読み込み処理に失敗した場合、容量チェック処理は実施しない
        NN_LOG("[QCIT][Error] ReadRequireNandSpaceSizeSetting() : GetDataOfFile() Failed (result = 0x%08x): path = %s\n",
            result.GetInnerValueForDebug(), requiredSizeFilePath.c_str());
        return;
    }

    // 末尾に改行文字があれば取り除く
    if (*dataStr.rbegin() == '\r')
    {
        dataStr.erase(dataStr.size() - 1);
    }

    // 文字列型から数値型に変換
    auto requiredSizeNum = static_cast<int64_t>(StringUtil::ToBit64(dataStr));
    // MB(メガバイト) 単位で設定しているので、バイト単位のスケールにしておく
    m_RequiredNandFreeSpaceSize = requiredSizeNum * 1024 * 1024;
    m_IsNandFreeSpaceSizeCheckRequired = true;
}

int64_t InstallScene::GetNandFreeSpaceSize() NN_NOEXCEPT
{
    int64_t size;
    auto result = nn::ns::GetFreeSpaceSize(&size, nn::ncm::StorageId::BuildInUser);
    if (result.IsFailure())
    {
        // NAND の空き容量の取得に失敗した場合は 0 を返しておく
        NN_LOG("[QCIT][Error] GetNandFreeSpaceSize() Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        size = 0;
    }
    return size;
}

// ToDo: どこで使うのかが決まっていない
void InstallScene::FinalizeProperties() NN_NOEXCEPT
{
    // プロパティデータの解放
    ContentListView::CollectionType* pNsps = nullptr;
    if ( nullptr != ( pNsps = m_pNsps ) )
    {
        m_pNsps = nullptr;
        delete pNsps;
    }
}

void InstallScene::Refresh() NN_NOEXCEPT
{

}

nn::Result InstallScene::EnumerateFiles( std::vector< FilePropertyType >& entries, const char* directoryPath ) NN_NOEXCEPT
{
    nn::fs::DirectoryHandle directoryHandle;
    NN_RESULT_DO( nn::fs::OpenDirectory( &directoryHandle, directoryPath, nn::fs::OpenDirectoryMode_All ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory( directoryHandle );
    };

    while ( NN_STATIC_CONDITION( true ) )
    {
        int64_t entryCount = 0;
        nn::fs::DirectoryEntry entry;
        NN_RESULT_DO( nn::fs::ReadDirectory( &entryCount, &entry, directoryHandle, 1 ) );
        if ( 0 >= entryCount )
        {
            break;
        }
        if ( entry.directoryEntryType == nn::fs::DirectoryEntryType_File )
        {
            if ( HasExtension( entry.name, ".nsp" ) || HasExtension( entry.name, ".nspu" ) )
            {
                FilePropertyType filePropery;
                entries.push_back( filePropery );
            }
        }

        else if ( entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory )
        {
            char childDirectoryPath[ MaxDirectoryNameLength ];
            nn::util::SNPrintf( childDirectoryPath, sizeof( childDirectoryPath ), "%s%s/", directoryPath, entry.name );

            // 子ディレクトリに対して再帰的に関数を呼び出す
            NN_RESULT_DO( EnumerateFiles( entries, childDirectoryPath ) );
        }
    }
    NN_RESULT_SUCCESS;
}

std::string ConvertDisplayType(const std::string& inTypeString) NN_NOEXCEPT
{
    struct TypeConvertPair
    {
        std::string typeStr;
        std::string displayStr;
    } typeConvert[] = {
        { "Application", "App" },
        { "AddOnContent", "AoC" },
        { "Patch", "Pth" },
    };

    auto returnValue = inTypeString;
    for (auto convert : typeConvert)
    {
        if (convert.typeStr == inTypeString)
        {
            returnValue = convert.displayStr;
            break;
        }
    }
    return returnValue;
}

nn::ncm::ContentMetaType ConvertContentMetaType(const std::string& inTypeString) NN_NOEXCEPT
{
    struct TypeConvertPair
    {
        std::string typeStr;
        nn::ncm::ContentMetaType metaType;
    } typeConvert[] = {
        { "App", nn::ncm::ContentMetaType::Application },
        { "AoC", nn::ncm::ContentMetaType::AddOnContent },
        { "Pth", nn::ncm::ContentMetaType::Patch },
    };

    auto returnValue = nn::ncm::ContentMetaType::Unknown;
    for (auto convert : typeConvert)
    {
        if (convert.typeStr == inTypeString)
        {
            returnValue = convert.metaType;
            break;
        }
    }
    return returnValue;
}

/**
 * @brief SD カードをマウントしファイルリストを作成。
 */
nn::Result InstallScene::MakeNspList() NN_NOEXCEPT
{
    ContentListView::CollectionType* pNsps = nullptr;
    m_pNsps = pNsps = new ContentListView::CollectionType();
    m_FileList.clear();

    auto contentInfoPath = std::string(RomMountName) + ":/InstallContentInfoList.txt";
    this->m_ContentInfoList.ReadFromContentInfoCSV(contentInfoPath);
    auto infoMap = this->m_ContentInfoList.GetInfoMap();

    m_QuestMenuFileNameSet.clear();
    pNsps->resize(infoMap.size());
    size_t i = 0;
    for (auto& content : infoMap)
    {
        InternalFilePropertyType ifp;
        {
            std::lock_guard< nn::os::Mutex > guard(m_CheckItemMutex);
            ifp.checkedPtr = new bool(false);
        }
        auto& record = content.second;
        auto displayTypeStr = ConvertDisplayType(record.type);

        ifp.id = record.id;
        ifp.type = displayTypeStr;
        ifp.name = record.nspFileName;
        auto sizeStr = GetDelimitedNumberString(StringUtil::ToBit64(record.dataSize));
        ifp.size = sizeStr;
        ifp.applicationId = record.applicationId;

        m_FileList.push_back(ifp);

        {
            std::lock_guard< nn::os::Mutex > guard(m_CheckItemMutex);
            pNsps->at(i).checkedPtr = ifp.checkedPtr;
        }
        BuildUtf16< 19 >(pNsps->at(i).id, "%s", record.id.c_str());
        BuildUtf16< 4 >(pNsps->at(i).type, "%s", displayTypeStr.c_str());
        BuildUtf16< MaxDirectoryNameLength >(pNsps->at(i).name, "%s", ClipMessage(record.nspFileName.c_str(), MaxCatalogFileNameLength).c_str());
        BuildUtf16< 32 >(pNsps->at(i).size, "%s", sizeStr.c_str());

        // Quest メニューの アプリID が含まれているファイル名を保持しておく
        if (record.applicationId == QuestMenuId)
        {
            m_QuestMenuFileNameSet.insert(record.nspFileName);
        }

        ++i;
    }

    EntryProperties(pNsps);

    NN_RESULT_SUCCESS;
}

/**
 * @brief SD カード内のプロパティコレクションを登録します。
 * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
 */
void InstallScene::EntryProperties( const ContentListView::CollectionType* pNsps ) NN_NOEXCEPT
{
    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    auto* const pLabelNoNsp = m_pLabelNoNsp;
    if ( nullptr != m_pNspListView )
    {
        m_pNspListView->EntryCollection( *pNsps );
        if ( true == pNsps->empty() )
        {
            m_pNspListView->disable( focusableProperty );
            pLabelNoNsp->enable( glv::Property::t::Visible );
            pLabelNoNsp->bringToFront();
        }
        else
        {
            m_pNspListView->enable( focusableProperty );
            pLabelNoNsp->disable( glv::Property::t::Visible );
            m_pFirstFocusedView = m_pNspListView;
        }
    }
}

/**
 * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
 */
glv::View* InstallScene::GetFocusableChild() NN_NOEXCEPT
{
    return m_pButtonAllCheckboxes;
}

/**
 * @brief ダイアログを表示して確認を取った後、インストールを行います。
 */
void InstallScene::ConfirmInstall( const FilePropertyType* pProperty ) NN_NOEXCEPT
{
    //NN_LOG("[Trace] Call InstallScene::ConfirmInstall()\n");

    auto pView = new ConfirmView();
    auto overSize = 0;
    int fileListSize = 0;

    std::set<std::string> fileNameSet;
    // 重複ファイル名がないかの確認も兼ねる
    for (auto item : m_FileList)
    {
        if ((m_IsInstallAll == true) || (item.checkedPtr != nullptr && *(item.checkedPtr) == true))
        {
            fileNameSet.insert(item.name);
        }
    }
    fileListSize = fileNameSet.size();

    if (fileListSize == 0)
    {
        pView->AddMessage("No checked item. Select at least one item.");
        pView->AddButton("Close",
            [this](void* pParam, nn::TimeSpan& timespan)
        {
            m_pRootSurface->setFocus(m_pNspListView);
        }
        );
    }
    else
    {
        // 確認表示時のファイル数が 1 つ
        if ( fileListSize == 1 )
        {
            pView->AddMessage( "Install the selected file?" );
        }
        // 確認表示時のファイル数が複数
        else
        {
            pView->AddMessage( "Install these files?" );
            // 上限を MaxComfirmNum 個までとする。
            if ( fileListSize > MaxComfirmNum )
            {
                overSize = fileListSize - MaxComfirmNum;
                fileListSize = MaxComfirmNum;
            }
        }

        // ファイル名を表示
        std::string fileNamesStr( "" );

        int i = 0;
        for (auto name : fileNameSet)
        {
            // モーダルにインストールするファイル名を表示、文字数が多い場合は最初と最後から数文字だけを表示
            fileNamesStr += (" - " + ClipMessage(name, MaxModalMessageLength));

            ++i;
            if (i < fileListSize)
            {
                fileNamesStr += '\n';
            }
            else
            {
                break;
            }
        }

        pView->AddMessage( fileNamesStr );

        // 列挙するファイル数が MaxComfirmNum を超えた場合、残りの数を表示して省略
        if ( overSize > 0 )
        {
            if ( overSize == 1 )
            {
                pView->AddMessage( "and another file." );
            }
            else
            {
                char message[MaxModalMessageLength] = { 0 };
                nn::util::SNPrintf(message, MaxModalMessageLength, "and the other %d files.", overSize);
                pView->AddMessage( message );
            }
        }

        // 開始ボタンを追加
        pView->AddButton(
            "Start",
            [ this ]( void* pParam, nn::TimeSpan& timespan )
            {
                InstallNspFile();
            }
        );
        pView->AddButton( "Cancel",
            [this](void* pParam, nn::TimeSpan& timespan)
            {
                m_pRootSurface->setFocus(m_pNspListView);
            }
        );
    }

    m_pRootSurface->StartModal(pView, true);
}

void InstallScene::ConfirmReboot() NN_NOEXCEPT
{
    auto pView = new ConfirmView();
    pView->AddMessage("Are you sure you want to reboot device ?");

    // 開始ボタンを追加
    pView->AddButton(
        "Reboot",
#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        [](void* pParam, nn::TimeSpan& timespan)
#else
        [this, pView](void* pParam, nn::TimeSpan& timespan)
#endif
    {
        // 再起動処理を始める API を叩く
#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        // NX環境の場合は oe の API を叩く
        nn::oe::RequestToReboot();
#else
        // Generic では特に何もしない
        m_pRootSurface->setFocus(m_pNspListView);
#endif
    }
    );
    pView->AddButton("Cancel",
        [this](void* pParam, nn::TimeSpan& timespan)
    {
        m_pRootSurface->setFocus(m_pNspListView);
    }
    );

    m_pRootSurface->StartModal(pView, true);
}

/**
 * @brief 選択された nsp ファイルのインストールを行います。
 */
void InstallScene::InstallNspFile() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread( &m_InstallTriggerThread, InstallTriggerThreadFunc, this, g_InstallStack, sizeof( g_InstallStack ), nn::os::DefaultThreadPriority )
    );

    // インストール進捗ビューを登録
    m_pInstallView = new InstallView([&]{ CloseButtonCallback(); }, [&]{ CancelButtonCallback(); }, [&]{ NN_LOG("Push Cleanup Button"); }, [&]{ NextButtonCallback(); });
    m_pRootSurface->StartModal(m_pInstallView, true, true);

    nn::os::StartThread( &m_InstallTriggerThread );
}

/**
 * @brief インストール進捗画面の Close ボタン押下時のコールバック関数です。
 */
void InstallScene::CloseButtonCallback() NN_NOEXCEPT
{
    m_InstallState = InstallState_NotStarted;
    DestroyInstallTriggerThread();

    m_pRootSurface->setFocus(m_pNspListView);
}

/**
 * @brief インストール進捗画面の Cancel ボタン押下時のコールバック関数です。
 */
void InstallScene::CancelButtonCallback() NN_NOEXCEPT
{
    m_RequestsCancel = true;
    m_InstallState = InstallState_NotStarted;
}

/**
 * @brief インストール進捗画面の Next ボタン押下時のコールバック関数です。
 */
void InstallScene::NextButtonCallback() NN_NOEXCEPT
{
    m_pInstallView->RecoverErrorToProgressMessage();
    if (m_IsLastTarget == false)
    {
        m_InstallState = InstallState_NotStarted;
    }
    else
    {
        m_InstallState = InstallState_Completed;
    }
}

/**
 * @brief インストールスレッドのメイン関数です。
 */
void InstallScene::InstallTriggerThreadFunc( void* args ) NN_NOEXCEPT
{
    InstallScene* self = reinterpret_cast< InstallScene* >( args );

    NN_ASSERT_NOT_NULL( self );
    NN_ASSERT_EQUAL( self->m_InstallState, InstallState_NotStarted );

    self->InstallProcess();
}

void InstallScene::InstallProcess() NN_NOEXCEPT
{
    nn::Result result;

    std::set<std::string> fileNameSet;
    // 指定されたコンテンツのファイル名をリスト化
    // 重複ファイル名がないかの確認も兼ねる
    for (auto item : m_FileList)
    {
        if ((m_IsInstallAll == true) || (item.checkedPtr != nullptr && *(item.checkedPtr) == true))
        {
            fileNameSet.insert(item.name);
        }
    }
    auto installFileListSize = fileNameSet.size();
    if (installFileListSize == 0)
    {
        // ありえないはずだが、インストール対象のファイルが 0 の場合は何もしない
        return;
    }

    // まずはアンインストール処理を実施 (何もしない場合有り)
    this->UninstallContents();
    m_InstallState = InstallState_NotStarted;

    m_RequestsCancel = false;

    // まずはインストール対象のファイル名を登録しておく
    m_pInstallView->SetInstallFileNameList(fileNameSet);

    // インストール対象の全体ファイルサイズを算出
    {
        this->m_InstalledContentSize = 0;
        this->m_AllContentSizeForInstalling = 0;
        for (auto& file : fileNameSet)
        {
            const auto filePath = std::string(RomMountName) + ":/" + file;
            this->m_AllContentSizeForInstalling += fsutil::GetFileSize(filePath);
        }
    }

    Stopwatch watch;
    int64_t totalSec = 0;
    size_t i = 0;
    for (auto& file : fileNameSet)
    {
        while (InstallState_NotStarted != m_InstallState)
        {
            // 状態が変化するまでポーリング
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }

        if (m_RequestsCancel) // Cancel ボタン押下
        {
            // Cancel ボタンを押すとモーダルが消えてしまうため、その対策処理
            if (!m_pRootSurface->IsModalViewRunning())
            {
                m_pRootSurface->StartModal(m_pInstallView, true);
            }
            m_pInstallView->SetTotalElapsedSecond(totalSec);
            // こちらは念のため
            m_pRootSurface->setFocus(nullptr);
            // インストール結果一覧の画面表示にする
            m_InstallState = InstallState_Completed;
            // インストール処理のループから抜ける
            break;
        }

        //　インストール先のストレージは Auto 扱いにしておく
        m_TargetStorageId = nn::ncm::StorageId::Any;
        if (m_QuestMenuFileNameSet.find(file) != std::end(m_QuestMenuFileNameSet))
        {
            // Quest メニューのファイルは、インストール先を NAND 固定にする
            m_TargetStorageId = nn::ncm::StorageId::BuildInUser;
        }

        const auto filePath = std::string(RomMountName) + ":/" + file;
        m_IsLastTarget = (i == (installFileListSize - 1));

#ifdef NN_BUILD_CONFIG_OS_WIN
        // Generic 版ではファイルが存在しない場合がありえるので、存在しない場合はスキップさせる
        if (fsutil::IsExistPath(filePath.c_str()) == false)
        {
            NN_LOG("[QCIT][Warning] Not Exist File : Path = %s\n", filePath.c_str());
            if (m_IsLastTarget)
            {
                m_InstallState = InstallState_Completed;
            }
            ++i;
            continue;
        }
#endif // NN_BUILD_CONFIG_OS_WIN

        watch.Start();
        result = ExecuteInstall(filePath.c_str());
        watch.Stop();

        // (SIGLO-76759) インストール後の NAND 空き容量確認処理の追加
        if (m_IsNandFreeSpaceSizeCheckRequired && result.IsSuccess())
        {
#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
            // NX 版では実際の NAND の空き容量と設定された空き容量のサイズとを比較する
            if (this->GetNandFreeSpaceSize() < m_RequiredNandFreeSpaceSize)
#endif // defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
            {
                // Generic 版は NAND の空き容量が確認できないので、強制エラーとしておく(デバック用途)
                // (RequireFreeSpace.txt を rom ディレクトリに配置していた場合のみ)

                // インストールしたコンテンツをアンインストールしておく
                this->UninstallContent(file);
                // 強制的に容量不足エラーとする
                result = nn::ncm::ResultNotEnoughInstallSpace();
            }
        }

        m_pInstallView->AddInstallResultList(file, result, watch.GetSecond());
        totalSec += watch.GetSecond();
        if (m_IsLastTarget)
        {
            m_pInstallView->SetTotalElapsedSecond(totalSec);
        }
        ++i;

        // インストール済のファイルサイズを加算していく
        this->m_InstalledContentSize += fsutil::GetFileSize(filePath);

        this->SetInstallResult(result);
    }
}

/**
 * @brief インストール結果を設定します。
 */
void InstallScene::SetInstallResult( nn::Result result ) NN_NOEXCEPT
{
    m_pInstallView->SetInstallResult(result);
    if ( result.IsSuccess() )
    {
        if ( m_IsLastTarget )
        {
            m_InstallState = InstallState_Completed;
        }
        else
        {
            m_InstallState = InstallState_NotStarted;
        }
    }
    else
    {
        m_InstallState = InstallState_Failed;
    }
}

/**
 * @brief nsp ファイルをインストールします。
 */
nn::Result InstallScene::ExecuteInstall( const char* filePath ) NN_NOEXCEPT
{
    const auto pRootSurface = this->m_pRootSurface;
    if ( !pRootSurface->IsModalViewRunning() )
    {
       pRootSurface->StartModal( m_pInstallView, true );
    }

    //NN_LOG("[Trace] ExecuteInstall() fileneme = %s\n", filePath);
    m_pInstallView->SetProgressFileName( ClipMessage( filePath, MaxModalMessageLength ).c_str() );

    // ターゲットが SD カードの場合は状態をチェックする
    if ( nn::ncm::StorageId::SdCard == m_TargetStorageId )
    {
        NN_RESULT_DO( nn::ns::CheckSdCardMountStatus() );
    }

    NN_RESULT_DO( nn::fs::OpenFile( &m_Handle, filePath, nn::fs::OpenMode::OpenMode_Read ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile( m_Handle );
    };

    // 不要なアプリケーション実体を削除
    NN_RESULT_DO( nn::ns::DeleteRedundantApplicationEntity() );

    std::unique_ptr< char[] > buffer( new char[ InstallBufferSize ] );
    nn::ns::ApplicationInstallTask task;

    // TORIAEZU: Pad からの入力で Progress の ModalView が消えないようするための Work Aruond
    // ひとつ前の ModalView が EndModal してから呼ぶ必要があるが、タイミングは適当
    m_pRootSurface->setFocus(nullptr);

    // インストールタスクを初期化して処理を開始
    const bool isNspuFile = HasExtension( filePath, ".nspu" );
    NN_RESULT_DO( task.Initialize( m_Handle, m_TargetStorageId, buffer.get(), InstallBufferSize, isNspuFile ) );
    NN_LOG("[QCIT] Preparing to install %s...\n", filePath);

    NN_RESULT_DO( task.Prepare() );

    bool needsCleanup = true;
    NN_UTIL_SCOPE_EXIT
    {
        if ( needsCleanup == true )
        {
            task.Cleanup();
        }
    };

    nn::ncm::ApplicationContentMetaKey applicationKey;

    const int maxContentMetaCount = 2048;
    int contentMetaKeyCount;
    int applicationMetaKeyCount;
    std::unique_ptr< nn::ncm::StorageContentMetaKey[] > keys( new nn::ncm::StorageContentMetaKey[ maxContentMetaCount ] );

    // インストール前にインストーラブルなコンテンツが入っているかをチェックする
    {
        NN_RESULT_DO( task.ListContentMetaKey( &contentMetaKeyCount, keys.get(), maxContentMetaCount, 0 ) );
        NN_RESULT_DO( task.ListApplicationContentMetaKey( &applicationMetaKeyCount, &applicationKey, 1, 0 ) );

        // key が 1 つも返って来ない == application が含まれていないシステムコンテンツの可能性あり
        if ( applicationMetaKeyCount == 0 )
        {
            needsCleanup = true;
            NN_RESULT_THROW( nn::ncm::ResultContentNotFound() );
        }
        const nn::ncm::ContentMetaType type = applicationKey.key.type;

        // インストールできるコンテンツでなければエラーを投げる
        if ( nn::ncm::ContentMetaType::Application != type && nn::ncm::ContentMetaType::AddOnContent != type && nn::ncm::ContentMetaType::Patch != type )
        {
            needsCleanup = true;
            NN_RESULT_THROW( nn::ncm::ResultContentNotFound() );
        }

        // 必要な情報は取得したので、いったん cleanup
        needsCleanup = false;
        NN_RESULT_DO( task.Cleanup() );
    }

    const nn::ncm::ApplicationId applicationId = applicationKey.applicationId;

    // アプリケーション記録の再生成要求管理
    bool requestsRecreateRecord = false;
    NN_UTIL_SCOPE_EXIT
    {
        RecreateApplicationRecord( applicationId, requestsRecreateRecord );
    };

    // 既にインストールされているかどうかチェックする
    {
        for ( int i = 0; i < contentMetaKeyCount; ++i )
        {
            auto key = keys[ i ];

            // すでに同じまたは新たなバージョンのコンテンツがインストール済かどうかの確認
            bool existsHigherOrEqualVersion;
            NN_RESULT_DO( devmenuUtil::ExistsHigherOrEqualVersion( &existsHigherOrEqualVersion, applicationId, key.key ) );
            if ( existsHigherOrEqualVersion )
            {
                if (key.key.type == nn::ncm::ContentMetaType::AddOnContent)
                {
                    // Aoc は従来通り上書きインストール処理を行う
                    // 新しいものを削除し、アプリケーション記録の再生成要求をしておく
                    NN_RESULT_TRY( devmenuUtil::DeleteBasedOnDatabase( key.key.type, key.key.id, key.storageId ) )
                        NN_RESULT_CATCH(nn::ncm::ResultContentMetaNotFound) {}
                    NN_RESULT_END_TRY
                    requestsRecreateRecord = true;
                }
                else
                {
                    // Aoc 以外は上書きインストールを行わずエラーを返す
                    NN_RESULT_THROW(nn::ncm::ResultContentAlreadyExists());
                }
            }
        }
        NN_RESULT_DO( task.Prepare() );
        needsCleanup = true;
    }

    // TORIAEZU: 念のため、もう一度呼んでおく
    m_pRootSurface->setFocus(nullptr);

    m_pInstallTask = &task;
    NN_RESULT_DO( StartAndWaitForExitInstallExecutionThread( &task ) );
    m_pInstallView->SetLastNspTotalSize( task.GetProgress().totalSize );
    NN_RESULT_DO( task.Commit() );
    requestsRecreateRecord = true;
    needsCleanup = false;

    nn::ns::InvalidateApplicationControlCache( applicationId );

    m_InstallState = InstallState_Committed;

    m_pInstallTask = nullptr;

    NN_RESULT_SUCCESS;
}

/**
 * @brief InstallTask 実行の事前準備を行います。
 */
void InstallScene::RecreateApplicationRecord( nn::ncm::ApplicationId applicationId, bool requestsRecreateRecord ) NN_NOEXCEPT
{
    nn::Result result;
    if ( requestsRecreateRecord )
    {
        result = devmenuUtil::PushApplicationRecordBasedOnDatabase( applicationId );
        if ( result.IsFailure() )
        {
            NN_LOG( "PushApplicationRecord failed: %0x08x\n", result.GetInnerValueForDebug() );
        }
    }

    result = nn::ns::CleanupUnrecordedApplicationEntity( applicationId );
    if ( result.IsFailure() )
    {
        NN_LOG("Cleanup failed: %0x08x\n", result.GetInnerValueForDebug());
    }
}

/**
 * @brief インストールトリガスレッドを破棄します。
 */
void InstallScene::DestroyInstallTriggerThread() NN_NOEXCEPT
{
    nn::os::WaitThread( &m_InstallTriggerThread );
    nn::os::DestroyThread( &m_InstallTriggerThread );
}

/**
 * @brief インストール処理を別スレッドで実行します。
 */
nn::Result InstallScene::StartAndWaitForExitInstallExecutionThread( nn::ncm::InstallTaskBase* task ) NN_NOEXCEPT
{
    static NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 s_Stack[ 16 * 1024 ];
    static nn::Result s_Result = nn::ResultSuccess();

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &m_InstallExecThread, []( void* p ) {
        s_Result = reinterpret_cast< nn::ncm::InstallTaskBase* >( p )->Execute();
    }, task, s_Stack, sizeof( s_Stack ), nn::os::DefaultThreadPriority ) );
    nn::os::StartThread( &m_InstallExecThread );
    m_InstallState = InstallState_Running;

    while ( NN_STATIC_CONDITION( true ) )
    {
        auto isFinished = nn::os::TryWaitAny( &m_InstallExecThread ) >= 0;
        if ( isFinished )
        {
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 250 ) );
    }
    nn::os::WaitThread( &m_InstallExecThread );
    nn::os::DestroyThread( &m_InstallExecThread );

    return s_Result;
}

/**
 * @brief インストール状況を更新します。
 */
void InstallScene::UpdateInstallProgress() NN_NOEXCEPT
{
    if ( InstallState_Running == m_InstallState )
    {
        const int64_t pastTimeMills = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() - m_StartTick ).GetMilliSeconds();
        if ( 750 < pastTimeMills )
        {
            if (m_pInstallTask == nullptr)
            {
                m_StartTick = nn::os::GetSystemTick();
                return;
            }

            auto progress = m_pInstallTask->GetProgress();
            if (progress.totalSize == 0)
            {
                m_StartTick = nn::os::GetSystemTick();
                return;
            }
            char progressStr[64] = { 0 };
            auto installedSizeStr = GetDelimitedNumberString( progress.installedSize );
            auto totalSizeStr = GetDelimitedNumberString( progress.totalSize );

            float progressRate = static_cast< float >( progress.installedSize ) / static_cast< float >( progress.totalSize );
            float progressRatePercent = 100 * progressRate;
            nn::util::SNPrintf( progressStr, sizeof( progressStr ), "%s / %s (%.1f%)", installedSizeStr.c_str(), totalSizeStr.c_str(), progressRatePercent );
            m_pInstallView->SetProgressLabel( progressStr );
            m_pInstallView->SetProgressBarValue( progressRate );
            m_pInstallView->SetProgressCaption("[Install Progress]");
            NN_LOG("%s\n", progressStr);

            // 全体進捗の更新処理
            {
                if (m_AllContentSizeForInstalling <= 0)
                {
                    m_StartTick = nn::os::GetSystemTick();
                    return;
                }
                auto totalProgressRate = static_cast<double>(m_InstalledContentSize) / static_cast<double>(m_AllContentSizeForInstalling);
                auto totalProgressRatePercent = 100 * totalProgressRate;
                char totalProgressStr[64] = { 0 };
                nn::util::SNPrintf(totalProgressStr, sizeof(totalProgressStr), "Total Progress : %s / %s (%.1f%)",
                    GetDelimitedNumberString(m_InstalledContentSize).c_str(),
                    GetDelimitedNumberString(m_AllContentSizeForInstalling).c_str(),
                    totalProgressRatePercent);
                m_pInstallView->SetTotalProgressLabel(totalProgressStr);
                m_pInstallView->SetTotalProgressBarValue(totalProgressRate);
            }

            m_StartTick = nn::os::GetSystemTick();
        }
    }
    else if ( InstallState_Committed == m_InstallState )
    {
        if ( true == m_IsInstallAll )
        {
            // ToDo: 進捗 100% を表示をするなど必要なら対応する
            m_InstallState = InstallState_NotStarted;
        }
    }
    else if ( InstallState_Completed == m_InstallState )
    {
        // インストール処理の結果画面を表示する
        m_pInstallView->SetInstallResultMessageView();

        m_InstallState = InstallState_NotStarted;
    }
    else if ( InstallState_Failed == m_InstallState )
    {
        m_pInstallView->SetErrorMessageView( m_IsLastTarget );
        m_InstallState = InstallState_WaitUserAction;
    }
    else if ( InstallState_WaitUserAction == m_InstallState )
    {
        // ボタン押下待ち
    }
    else if (InstallState_Uninstall == m_InstallState)
    {
        char progressStr[48] = { 0 };
        auto totalSizeStr = GetDelimitedNumberString(m_pInstallView->GetLastNspTotalSize());
        auto progressRate = static_cast< float >(m_UninstalledContentCount) / static_cast< float >(m_AllContentCountForDeleting);
        float progressRatePercent = 100 * progressRate;
        nn::util::SNPrintf(progressStr, sizeof(progressStr), "%d / %d (%.1f%)", m_UninstalledContentCount, m_AllContentCountForDeleting, progressRatePercent);

        m_pInstallView->SetProgressCaption("[Uninstall Content]");
        m_pInstallView->SetProgressFileName(this->m_UninstallingContentName.c_str());
        m_pInstallView->SetProgressLabel(progressStr);
        m_pInstallView->SetProgressBarValue(progressRate);
    }
}

void InstallScene::InitialCheckProcess() NN_NOEXCEPT
{
    if (m_IsAbortCheckQuestDevice)
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        // DebugMode が無効の場合、チェックを実施する
        if (m_DebugMode == false)
        {
            // Quest 化されている本体かどうかの確認
            // Quest 化されていなければ強制的に Abort する
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::rid::CheckRetailInteractiveDisplayDevice());
        }
#endif // defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        // (Generic 版では実質的に何もしない)

        // 念のためフラグは落としておく
        m_IsAbortCheckQuestDevice = false;
    }

    if (m_IsSdCardStatusInvalid)
    {
        this->FormatSdCardAndRebootProcess();
        // 念のためフラグは落としておく
        m_IsSdCardStatusInvalid = false;
    }
}

void InstallScene::FormatSdCardAndRebootProcess() NN_NOEXCEPT
{
    auto pView = new MessageView(false, true);

    pView->AddMessage("[ Caution ]");
    pView->AddMessage("SD card status is Invalid.");
    pView->AddMessage("Pressing the OK button, SD card format and device restart are done Automatically.");

    pView->AddButton(
        "OK",
        [](void* pParam, nn::TimeSpan& timespan)
    {
        // SD カードをフォーマットする
        auto result = nn::ns::FormatSdCard();
        if (result.IsFailure())
        {
            if (nn::ns::ResultSdCardFormatUnexpected::Includes(result))
            {
                result = nn::ns::GetLastSdCardFormatUnexpectedResult();
            }
            NN_LOG("[QCIT][Error] FormatSdCard Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        }

        // 再起動処理を始める API を叩く
#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        // NX 環境の場合は対応する oe の API を叩く
        nn::oe::RequestToReboot();
#else // defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        // Generic では特に何もしない
#endif // defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    }
    );

    m_pRootSurface->StartModal(pView, true);
}

void InstallScene::ShowImageHash() NN_NOEXCEPT
{
    auto pView = new MessageView(false);

    auto selectedPtr = m_pNspListView->GetSelectedValue();
    if (selectedPtr == nullptr)
    {
        NN_LOG("[Error] Not Selected File Item\n");
        return;
    }

    char id[19] = { 0 };
    ConvertWideStringToString(id, selectedPtr->id);

    std::string idStr(id);

    auto infoMap = this->m_ContentInfoList.GetInfoMap();
    auto itr = infoMap.find(StringUtil::ToBit64(idStr));
    if (itr == std::end(infoMap))
    {
        NN_LOG("[Error] Not Fount ID Item : %s\n", idStr.c_str());
        return;
    }

    // モーダルにファイル名を表示、文字数が多い場合は最初と最後から数文字だけを表示
    std::string fileNameStr = itr->second.nspFileName;
    const std::string fileNameLineFeedSpace = "\n        ";
    if (fileNameStr.size() > 60 && fileNameStr.size() <= 120)
    {
        // 60文字を超える場合は折り返す
        fileNameStr.insert(59, fileNameLineFeedSpace);
    }
    else if (fileNameStr.size() > 120 && fileNameStr.size() <= 180)
    {
        // 120文字を超える場合は短縮系で表示する
        fileNameStr.insert(59, fileNameLineFeedSpace);
        fileNameStr.insert(119 + fileNameLineFeedSpace.size(), fileNameLineFeedSpace);
    }
    else if (fileNameStr.size() > 180)
    {
        // 180文字を超える場合は短縮系で表示する
        fileNameStr = ClipMessage(fileNameStr, 55);
    }

    pView->AddMessage("ID: " + idStr + "\nFile: " + fileNameStr, 19.0f);

    const std::string titleStr("[Content Image Hash]\n  ");
    std::string diplayHashStr, qrCodeHashStr;
    diplayHashStr = qrCodeHashStr = itr->second.digest;
    diplayHashStr.insert((qrCodeHashStr.size() / 2), "\n  ");

    pView->AddMessage(titleStr + diplayHashStr);

    mw::qre::ImageInfo imageInfo = { 0, 0, nullptr, mw::qre::Rect() };
    uint32_t imageSize = 0;
    const auto hashSize = qrCodeHashStr.size();
    std::unique_ptr< uint8_t[] > hashData(new uint8_t[hashSize]);
    memcpy(hashData.get(), qrCodeHashStr.c_str(), hashSize);
    qcit::CreateQrCode(&imageInfo, &imageSize, hashData.get(), hashSize);
    pView->AddImage(imageInfo.rgbData, imageSize, imageInfo.width, imageInfo.height);
    delete[] imageInfo.rgbData;

    pView->AddButton("Close",
        [this](void* pParam, nn::TimeSpan& timespan)
        {
            m_pRootSurface->setFocus(m_pNspListView);
        }
    );
    m_pRootSurface->StartModal(pView, true);
}

void InstallScene::UninstallContents() NN_NOEXCEPT
{
    const auto syncListPath = std::string(RomMountName) + ":/SyncListInfo.txt";
    if (fsutil::IsExistPath(syncListPath.c_str()) == false)
    {
        // Sync リストが設定されていない場合、アンインストール処理は実行しない
        return;
    }

    m_InstallState = InstallState_Uninstall;

    auto result = this->m_ContentInfoList.ReadFromSyncListCSV(syncListPath);
    if (result.IsFailure())
    {
        NN_LOG(" [Error] ReadFromSyncListCSV Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        return;
    }

    auto syncList = m_ContentInfoList.GetSyncList();
    std::set<nn::Bit64> skipIdSet;
    SkipIndexList skipIndexList;
    for (auto rd : syncList)
    {
        // レコードから ApplicationId の数値を取得
        const auto setId = StringUtil::ToBit64(rd.id);
        if (rd.type == "Demo")
        {
            // Demo の場合は skipIdSet に単純に追加
            skipIdSet.insert(setId);
        }
        else if (rd.type == "Asset")
        {
            // Asset の場合は、
            auto iter = skipIndexList.find(setId);
            if (iter == std::end(skipIndexList))
            {
                // skipIndexList に登録されていないIDであれば登録して、
                std::pair<nn::Bit64, std::set<int>> p;
                p.first = setId;
                iter = skipIndexList.insert(p).first;
            }

            // 登録している ID に対して index の値を追加する
            int setIndex = StringUtil::ToBit32(rd.index);
            iter->second.insert(setIndex);
        }
    }

    result = nn::ns::DeleteRedundantApplicationEntity();
    if (result.IsFailure())
    {
        NN_LOG(" [Error] DeleteRedundantApplicationEntity Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        return;
    }

    auto selfAppId = StringUtil::ToBit64(m_ApplicationInfoData.applicationId);

    result = this->UninstallContentsDetail(selfAppId, skipIdSet, skipIndexList);
    if (result.IsFailure())
    {
        NN_LOG(" [Error] UninstallContentsDetail Failed : result = 0x%08x\n", result.GetInnerValueForDebug());
        return;
    }

    auto& refInfoMap = m_ContentInfoList.GetInfoMap();
    for (auto& item : refInfoMap)
    {
        auto iter = skipIdSet.find(item.first);
        if (iter != std::end(skipIdSet))
        {
            skipIdSet.erase(iter);
        }
    }

    // インストール結果ダイアログに Sync リストの失敗した ID を登録(何もしない場合もあり)
    m_pInstallView->SetSyncFailIdList(skipIdSet);
}

nn::Result InstallScene::UninstallContentsDetail(nn::Bit64 inQCITId, std::set<nn::Bit64>& inSkipIdSet, SkipIndexList& inSkipIndexList) NN_NOEXCEPT
{
    nn::Result result;

    const size_t bufferSize = 1024;
    std::unique_ptr<nn::ns::ApplicationRecord[]> recordBuffer(new nn::ns::ApplicationRecord[bufferSize]);
    auto recordList = recordBuffer.get();

    std::vector<nn::ns::ApplicationView> viewList;

    int offset = 0;
    int count = 1;
    while (count > 0)
    {
        count = nn::ns::ListApplicationRecord(recordList, bufferSize, offset);
        for (int i = 0; i < count; ++i)
        {
            auto& record = recordList[i];
            nn::ns::ApplicationView view;
            result = nn::ns::GetApplicationView(&view, &record.id, 1);
            if (result.IsFailure())
            {
                NN_LOG(" [Error] GetApplicationView Failed : 0x%016llx\n", record.id);
                return result;
            }

            viewList.push_back(view);
        }

        offset += count;
    }

    this->m_AllContentCountForDeleting = viewList.size();
    this->m_UninstalledContentCount = 0;

    for (auto& v : viewList)
    {
        {
            auto iter = inSkipIndexList.find(v.id.value);
            if (iter != std::end(inSkipIndexList))
            {
                // Aoc のみの削除処理なので、別処理とする
                UninstallAocOnly(iter->first, iter->second);
                // 削除処理が終われば次のIDへ
                continue;
            }
        }

        if (v.id.value == inQCITId)
        {
            // 自分自身のアプリID は対象外
            continue;
        }

        //NN_LOG(" [Trace] UninstallContents  ViewList : ID = 0x%016llx\n", v.id.value);
        auto iter = inSkipIdSet.find(v.id.value);
        if (iter != std::end(inSkipIdSet))
        {
            inSkipIdSet.erase(iter);
            // Sync リストに記載されている ID は削除対象としない
            //NN_LOG(" [Trace] UninstallContents Skip : ID = 0x%016llx\n", v.id.value);
            continue;
        }

        char appId[32] = { 0 };
        nn::util::SNPrintf(appId, sizeof(appId), "  ID : 0x%016llx", v.id.value);
        this->m_UninstallingContentName = appId;
        //NN_LOG(" [Trace] UninstallContents : ID = 0x%016llx\n", v.id.value);

        result = nn::ns::DeleteApplicationCompletely(v.id);
        if (result.IsFailure())
        {
            NN_LOG(" [Error] DeleteApplicationCompletely Failed : result = 0x%08x, id = 0x%016llx\n", result.GetInnerValueForDebug(), v.id.value);
            return result;
        }

        ++m_UninstalledContentCount;
    }

    return result;
}

void InstallScene::UninstallAocOnly(nn::Bit64 inTargetId, std::set<int>& inSkipIndexSet) NN_NOEXCEPT
{
    // 対象アプリケーションIDのコンテンツメタの数を取得
    nn::ncm::ApplicationId appId = { inTargetId };
    int metaCount = 0;
    auto result = nn::ns::CountApplicationContentMeta(&metaCount, appId);
    if (result.IsFailure())
    {
        NN_LOG(" [Error] CountApplicationContentMeta() Failed : result = 0x%08x, id = 0x%016llx\n", result.GetInnerValueForDebug(), inTargetId);
        return;
    }

    // 対象アプリケーションIDのコンテンツメタ設定の一覧を取得
    std::unique_ptr<nn::ns::ApplicationContentMetaStatus[]> buffer(new nn::ns::ApplicationContentMetaStatus[metaCount]);
    auto listPtr = buffer.get();

    int outCount = 0;
    result = nn::ns::ListApplicationContentMetaStatus(&outCount, listPtr, metaCount, appId, 0);
    if (result.IsFailure())
    {
        NN_LOG(" [Error] ListApplicationContentMetaStatus() Failed : result = 0x%08x, id = 0x%016llx\n", result.GetInnerValueForDebug(), inTargetId);
        return;
    }

    for (int i = 0; i < outCount; ++i)
    {
        auto& metaStatus = listPtr[i];
        // コンテンツメタはAocのみ着目
        if (metaStatus.type == nn::ncm::ContentMetaType::AddOnContent)
        {
            // スキップ対象のインデックス値かどうかの比較処理
            auto indexValue = static_cast<int>(metaStatus.id & 0x0000000000000FFF);
            auto iter = inSkipIndexSet.find(indexValue);
            if (iter == std::end(inSkipIndexSet))
            {
                // スキップリストに指定されていないので、コンテンツを削除する
                devmenuUtil::DeleteBasedOnDatabase(metaStatus.type, metaStatus.id, metaStatus.installedStorage);
            }
            else
            {
                // 指定しているならば、スキップリストから削除しておく (SyncFailure用処理)
                inSkipIndexSet.erase(iter);
            }
        }
    }

    if (inSkipIndexSet.empty())
    {
        // 空の場合は以降の処理は無駄なので返っておく
        return;
    }

    auto& refInfoList = m_ContentInfoList.GetDataList();
    for (auto& rd : refInfoList)
    {
        if (StringUtil::ToBit64(rd.applicationId) == inTargetId && rd.type == "AddOnContent")
        {
            int indexValue = static_cast<int>(StringUtil::ToBit64(rd.id) & 0x0000000000000FFF);
            inSkipIndexSet.erase(indexValue);
        }
    }

    // インストール結果ダイアログに Sync リストの失敗した ID を登録(何もしない場合もあり)
    m_pInstallView->SetSyncFailAocIndexList(inTargetId, inSkipIndexSet);
}

void InstallScene::UninstallContent(const std::string& inFileName) NN_NOEXCEPT
{
    std::vector< InternalFilePropertyType > uninstallList;
    for (auto& item : m_FileList)
    {
        // インストール処理中に呼ばれる想定なので、チェック済のファイルのみが対象となる
        if (item.checkedPtr != nullptr && *(item.checkedPtr) == true)
        {
            if (inFileName == item.name)
            {
                // 同じファイル名のプロパティを保持
                // 1ファイルに複数コンテンツが含まれている可能性があるためすぐに break しない
                uninstallList.push_back(item);
            }
        }
    }

    if (uninstallList.empty())
    {
        // ありえないと思うが、該当のファイルが存在しない場合を考慮
        NN_LOG("[QCIT][Error] UninstallContent() : Not Found, inFileName=%s\n", inFileName.c_str());
        return;
    }

    nn::ns::DeleteRedundantApplicationEntity();

    // 運用上ありえないと思うが、1ファイルに複数の(Aoc)コンテンツが含まれている場合を考慮
    for (auto& content : uninstallList)
    {
        auto metaType = ConvertContentMetaType(content.type);
        auto setIdStr = (nn::ncm::ContentMetaType::AddOnContent == metaType) ? content.id : content.applicationId;
        nn::ncm::ApplicationId contentId = { StringUtil::ToBit64(setIdStr) };
        // 削除する対象ストレージは運用上ひとまず NAND 固定となる
        nn::Result result;
        if (nn::ncm::ContentMetaType::AddOnContent == metaType)
        {
            // Aoc の削除は value 指定でないと駄目な模様。。
            result = devmenuUtil::DeleteBasedOnDatabase(metaType, contentId.value, nn::ncm::StorageId::BuildInUser);
        }
        else
        {
            result = devmenuUtil::DeleteBasedOnDatabase(metaType, contentId, nn::ncm::StorageId::BuildInUser);
        }

        if (result.IsFailure())
        {
            NN_LOG("[QCIT][Error] UninstallContent() DeleteBasedOnDatabase failed: 0x%08x, setid = %s\n", result.GetInnerValueForDebug(), setIdStr.c_str());
        }
    }

    // Push するのは 1 番目のプロパティのみとしておく (運用上はひとまず問題ないはず)
    nn::ncm::ApplicationId appId = { StringUtil::ToBit64(uninstallList.at(0).applicationId) };
    auto result = devmenuUtil::PushApplicationRecordBasedOnDatabase(appId);
    if (result.IsFailure())
    {
        NN_LOG("[QCIT][Error] UninstallContent() PushApplicationRecordBasedOnDatabase failed: 0x%08x, appId = %s\n",
            result.GetInnerValueForDebug(), uninstallList.at(0).applicationId.c_str());
    }
}

void InstallScene::CheckAllItems() NN_NOEXCEPT
{
    std::lock_guard< nn::os::Mutex > guard(m_CheckItemMutex);
    bool isAllChecked = true;
    for (auto& item : m_FileList)
    {
        if ((item.checkedPtr != nullptr) && (*item.checkedPtr == false))
        {
            // 未チェック項目をチェック済にする
            *item.checkedPtr = true;
            // オールクリアのフラグを無効にする
            isAllChecked = false;
        }
    }

    if (isAllChecked)
    {
        // 全てチェック済だった場合のみオールクリア処理を実施する
        for (auto& item : m_FileList)
        {
            if (item.checkedPtr != nullptr)
            {
                *item.checkedPtr = false;
            }
        }
    }
}

} // ~namespace qcit
