﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/repair.h>

#include <nn/repair/repair_LabelButton.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_ShutdownButton.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Ptm.h>
#include <nn/util/util_FormatString.h>

#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>

#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>

namespace {

const char* ToolName = "NX Reinitialize Fuel Gauge";
const int ToolMajorVersion = 1;
const int ToolMinorVersion = 2;

const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;
glv::Label s_ToolInformationLabel;
glv::Label s_ProgressLabel;
nn::repair::StreamTextView* s_pStreamText;
nn::repair::LabelButton* s_pLabelButton;
nn::os::ThreadType s_WorkerThread;

const int OperationStackSize = 50 * 4096;
NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t s_OperationStack[OperationStackSize];

glv::GLV s_GlvRootView;

void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

void Deallocate(void* address, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(address);
}

void* Reallocate(void* address, size_t size, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(address, size);
}

void SetupPeripherals() NN_NOEXCEPT
{
    nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
    glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
}

//! 本体設定上の電池情報を削除します。
nn::Result EraseBatteryLotFromSystem() NN_NOEXCEPT
{
    nn::settings::factory::BatteryLot blankLot;

    memset(&(blankLot.string[0]), '\0', sizeof(blankLot.string));

    nn::settings::system::SetPtmBatteryLot(blankLot);
    nn::settings::system::GetPtmBatteryLot(&blankLot);

    NN_RESULT_SUCCESS;
}

void DisplayErrorMessage(nn::Result result) NN_NOEXCEPT
{
    const int BufferSize = 200;
    char buffer[BufferSize];
    nn::util::SNPrintf(buffer, BufferSize, "Operation Failed. Error code = 0x%08x.\n", result.GetInnerValueForDebug());

    s_pStreamText->AppendValue(std::string(buffer));
}

void OperationFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    // 本体設定上の電池情報を削除します。
    nn::Result result = EraseBatteryLotFromSystem();
    if ( result.IsFailure() )
    {
        DisplayErrorMessage(result);
    }
    else
    {
        s_ProgressLabel.setValue("Progress: Finished");
        s_pStreamText->AppendValue("[[ SUCCESS ]]\n");
    }
}

void StartRepairOperation() NN_NOEXCEPT
{
    nn::Result result = nn::os::CreateThread(&s_WorkerThread, OperationFunction, nullptr, s_OperationStack, OperationStackSize, 30);

    if ( result.IsFailure() )
    {
        DisplayErrorMessage(result);
    }
    else
    {
        nn::os::StartThread(&s_WorkerThread);
    }
}

void DisplayToolInformation(glv::GLV* const pGlvRootView) NN_NOEXCEPT
{
    nn::repair::GetToolInformationLabel(&s_ToolInformationLabel, ToolName, ToolMajorVersion, ToolMinorVersion);

    *pGlvRootView << s_ToolInformationLabel;
}

nn::Result Initialize() NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    s_pStreamText = new nn::repair::StreamTextView(glv::Rect(1024, 440), 28.0f);
    s_pStreamText->pos(120, 164);
    *pGlvRootView << s_pStreamText;

    // シャットダウンボタン(FW 3.0.0 以降で有効)
    glv::Button* shutdownButton = new nn::repair::ShutdownButton();
    shutdownButton->pos(800,640);
    *pGlvRootView << shutdownButton;

    DisplayToolInformation(pGlvRootView);

    s_ProgressLabel.setValue("Progress: Waiting for the operation\n");

    s_pLabelButton = new nn::repair::LabelButton("START", [&] { StartRepairOperation(); });
    s_GlvRootView << s_pLabelButton->pos(glv::Place::BC, 0, -30).anchor(glv::Place::BC);

    NN_RESULT_SUCCESS;
}

void Main(glv::Window& window) NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    nn::Result result = Initialize();
    if ( result.IsFailure() )
    {
        DisplayErrorMessage(result);
    }

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    window.setGLV(*pGlvRootView);
    glv::Application::run();
}

} // namespace

extern "C" void nnMain()
{
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_ABORT_UNLESS_NOT_NULL(window);

    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    Main(*window);
}
