﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @page PageTargetToolRepairAssistCup
 * @tableofcontents
 *
 * @brief
 * 修理、リファービッシュ工程において CUP を補助するプログラムの解説です。
 *
 * @section PageTargetToolRepairAssistCup_SectionFileStructure ファイル構成
 * 本プログラムは @link ../../../Programs/Iris/Sources/TargetTools/RepairTools/RepairAssistCup
 *  @endlink 以下にあります。
 *
 * @section PageTargetToolRepairAssistCup_SectionNecessaryEnvironment 必要な環境
 * Ocean 実行環境
 *
 * @section PageTargetToolRepairAssistCup_SectionHowToOperate 操作方法
 * FW 2.0.0 未満であれば画面下のボタンを押すことで
 * 初回起動シーケンスを省略するための仮設定ができます
 *
 * @section PageTargetToolRepairAssistCup_SectionPrecaution 注意事項
 *
 * @section PageTargetToolRepairAssistCup_SectionHowToExecute 実行手順
 * ビルドし、実行してください。
 *
 * @section PageTargetToolRepairAssistCup_SectionDetail 解説
 * 特記事項無し
 */

#include <iostream>
#include <iomanip>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <sstream>
#include <nn/repair/repair_Api.h>
#include <nn/repair/repair_LabelButton.h>
#include <nn/repair/repair_ShutdownButton.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/repair/repair_Sdcard.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_NetworkSettingsParser.h>

// デバイス情報取得
#include <nn/util/util_BitFlagSet.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/time/time_Api.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#if defined( NN_SDK_BUILD_LIBRARY )
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#else
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#endif

//#define GLV_ENABLE_LOG_DEBUG

//!--------------------------------------------------------------------------------------
//! @brief SDKライブラリビルドでのアサート/ログマクロ制限.
//!--------------------------------------------------------------------------------------
#if defined( NN_SDK_BUILD_LIBRARY )
#define NN_ASSERT NN_SDK_ASSERT
#define GLV_LOG(...) NN_SDK_LOG( "[GLV] " __VA_ARGS__ )
#else
#define GLV_LOG(...) NN_LOG( "[GLV] " __VA_ARGS__ )
#endif // defined( NN_SDK_BUILD_LIBRARY )


namespace {
    const char* ToolName             = "NX Cup Assistant Tool";
    const int   ToolMajorVersion     = 1;
    const int   ToolMinorVersion     = 0;

    const char* DispMsgStartupDone   = "Initial Setup\t\t\tAlready Done";
    const char* DispMsgStartupNotYet = "Initial Setup\t\t\tNot Yet";

    const char* DispMsgAboutFirmware1  = "Auto-Cup is NOT available! ";
    const char* DispMsgAboutFirmware1B = "You need to start CUP-Card manually.";
    const char* DispMsgAboutFirmware2 = "Auto-Cup is available! Set CUP-Card and Reboot System.";

    const char* DispMsgSkipDone      = "Made-up settings has been saved! Reboot System.";
    const char* DispMsgButtonSkip    = "Skip Initial-Startup"; // or Initial Setup Sequence

    static nn::repair::StreamTextView*    s_pTextView;
    static glv::Button*                   s_skipButton = nullptr;

    static const size_t AppletHeapSize = 256 * 1024 * 1024;                 //!< アプリケーション予約ヒープメモリサイズ
    static const size_t AppletAllocatableSize = 256 * 1024 * 1024;          //!< アプリケーション稼働ヒープ上限メモリサイズ
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024; //!< NVNグラフィクス稼働予約メモリ領域サイズ
#endif
    static bool  IsCompletionStartup  = true;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)

    void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, size);
    }

    void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        free(addr);
    }

    void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
#endif

    //!--------------------------------------------------------------------------------------
    //! @brief GLV用ユーザ定義アロケータ.
    //!--------------------------------------------------------------------------------------
    void* glvMemoryAllocator( const size_t size, const size_t beginAlignment ) NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        return ::_aligned_malloc( size, beginAlignment );
#else
        return ::aligned_alloc( beginAlignment, size );
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief GLV用ユーザ定義デアロケータ.
    //!--------------------------------------------------------------------------------------
    void glvMemoryDeallocator( void* address ) NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        ::_aligned_free( address );
#else
        ::free( address );
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ペリフェラルセットアップ
    //!--------------------------------------------------------------------------------------
    static void SetupPeripherals() NN_NOEXCEPT
    {
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        // this memory allocation will be used from the nvn graphics systems at runtime.
        nv::SetGraphicsAllocator( NvAllocate, NvFree, NvReallocate, nullptr );
        nv::InitializeGraphics( ::malloc( GraphicsSystemReservedMemorySize ), GraphicsSystemReservedMemorySize );

#if NN_GFX_IS_TARGET_NVN
        // this memory allocation interface will be used when compiling of shader code at runtime.
        glslcSetAllocator( NvAllocate, NvFree, NvReallocate, nullptr );
#endif
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief HID設定初期値
    //!--------------------------------------------------------------------------------------
    static const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration( glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary );

} // namespace

//!--------------------------------------------------------------------------------------
//! nninitStartup() is invoked before calling nnMain().
//! 256MB確保
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::SetMemoryHeapSize( AppletHeapSize ) );
    uintptr_t address;
    const size_t MallocMemorySize = AppletAllocatableSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::AllocateMemoryBlock( &address, MallocMemorySize ) );
    nn::init::InitializeAllocator( reinterpret_cast< void* >( address ), MallocMemorySize );

    // GLV内部で利用されるルートメモリアロケータを上書きします.
    // 上書きしない場合は C++11 の std::aligned_alloc 及び std::free が利用されます.
    glv::detail::ApplicationMemoryAllocator::AttachUserAllocator( glvMemoryAllocator, glvMemoryDeallocator );
}

void MyMouseDownHandlerSkip() NN_NOEXCEPT
{
    // 初回起動フラグを 真 にセット
    // EULA Version を １ にセット
    // Region を Australia にセット
    {
        nn::settings::system::InitialLaunchSettings ils;
        bool completionFlag = true;

        nn::settings::system::GetInitialLaunchSettings(&ils);
        ils.flags.Set<nn::settings::system::InitialLaunchFlag::IsCompleted>(completionFlag);
        nn::settings::system::SetInitialLaunchSettings(ils);
        nn::settings::system::SetRegionCode(nn::settings::system::RegionCode_Australia);

        nn::settings::system::EulaVersion eulaver;

        // 上位16ビットがメジャーバージョン、下位16ビットがマイナーバージョン
        eulaver.version = 0x10000;

        // nn::settings::system::GetRegionCode は別権限(set)が必要
        eulaver.regionCode = nn::settings::system::RegionCode_Australia;

        eulaver.clockType = nn::settings::system::EulaVersionClockType_SteadyClock;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&eulaver.steadyClock));

        nn::settings::system::SetEulaVersions(&eulaver, 1);
    }

    s_pTextView->AppendValue(DispMsgSkipDone);
    s_pTextView->AppendValue("\n");

    s_pTextView->AppendValue(DispMsgAboutFirmware1B);
    s_pTextView->AppendValue("\n");

    if(s_skipButton != nullptr)
      s_skipButton -> disable(glv::Property::Visible);
}

//!--------------------------------------------------------------------------------------
//! @brief ルートサーフェイスコンテキスト
//!--------------------------------------------------------------------------------------
class RootSurfaceContext : public glv::Window, public glv::GLV, public glv::ApplicationLoopCallback
{
private:

public:
    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ.
    //! @details 指定の幅、高さの領域に対して、ページセレクトヘッダバーとページサーフェイスを追加します.
    //!--------------------------------------------------------------------------------------
    RootSurfaceContext( const unsigned width, const unsigned height ) NN_NOEXCEPT
         : glv::Window( width, height, "Main Window" ), glv::GLV()
    {
        // this->setGLV( *this );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief デストラクタ.
    //! @attention
    //! DropDownオブジェクトは内部で ListViewを利用していますが、親による自動解放が正しく動かないバグがある模様のため、
    //! 明示的に親( m_pSurfaceTable )が解放される前にデストラクションしています.
    //! m_pSurfaceTableもついでに.
    //! ※尚、GLVのViewは glv::SmartObjectクラスにより動的確保したオブジェクトは、親から外される際に自動的に delete されます.
    //!   明示的解放の場合には、親子関係の解除も含めるようにしてください.
    //!--------------------------------------------------------------------------------------
    virtual ~RootSurfaceContext() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ランタイムエンジンにアタッチされた際に呼ばれます.
    //! @see glv::ApplicationLoopCallback::OnLoopAttached( ApplicationLoopContext& )
    //!--------------------------------------------------------------------------------------
    virtual void OnLoopAttached( glv::ApplicationLoopContext& context ) NN_NOEXCEPT NN_OVERRIDE
    {
        ApplicationLoopCallback::OnLoopAttached( context );
        GLV_LOG( "OnLoopAttached\n" );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ランタイムエンジンからデタッチされた際に呼ばれます.
    //! @see glv::ApplicationLoopCallback::OnLoopDetached( ApplicationLoopContext& )
    //!--------------------------------------------------------------------------------------
    virtual void OnLoopDetached( glv::ApplicationLoopContext& context ) NN_NOEXCEPT NN_OVERRIDE
    {
        GLV_LOG( "OnLoopDetached\n" );
        ApplicationLoopCallback::OnLoopDetached( context );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ループ中の呼び出し.@n
    //! @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます.@n
    //! この時点ではまだ glvコンテキストのレンダリングは開始していません.@n
    //! また、このメソッドが呼び出されるフレームは OnLoopAfterSceneRendererと同じです.@n
    //! @return reserved.@n
    //! 現在はRequiredRestoration::RequireRestrationNothing を返却してください.
    //! @see glv::ApplicationLoopCallback::OnLoopBeforeSceneRenderer( ApplicationLoopContext&, const HidEvents& )
    //!--------------------------------------------------------------------------------------
    virtual const glv::RequiredRestoration OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( events );
        NN_UNUSED( context );
        if ( events.GetAvailableBasicPadCount() > 0 )
        {
            auto& bpad = events.GetBasicPad( 0 );
            if ( bpad.IsButtonDown<glv::BasicPadEventType::Button::R>() || bpad.IsButtonRepeat<glv::BasicPadEventType::Button::R>() )
            {
                GLV_LOG( "OnLoopBeforeSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }
        }
        return glv::RequiredRestoration::RequireRestrationNothing;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ループ中の呼び出し.@n
    //! @details glvシーンレンダラのレンダリングが終わった後に呼び出されます.@n
    //! また、このメソッドが呼び出されるフレームは OnLoopBeforeSceneRendererと同じです.@n
    //! @return reserved.@n
    //! 現在はRequiredRestoration::RequireRestrationNothing を返却してください.
    //! @see glv::ApplicationLoopCallback::OnLoopAfterSceneRenderer( ApplicationLoopContext&, const HidEvents& )
    //!--------------------------------------------------------------------------------------
    virtual const glv::RequiredRestoration OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( events );
        NN_UNUSED( context );
        const glv::DebugPadEventType& dpad = events.GetDebugPad();
        if ( true == dpad.HasAnyEvent() )
        {
            // SELECT + START 同時押し( 後押し許可版 )
            const glv::DebugPadEventType::ButtonSetType exit( nn::hid::DebugPadButton::Start::Mask | nn::hid::DebugPadButton::Select::Mask );
            if ( exit == ( dpad.GetButtons() & exit )  )
            {
                // アプリケーションループ退場要求
                glv::ApplicationFrameworkExit();
            }
            if ( dpad.IsButtonDown<nn::hid::DebugPadButton::L>() )
            {
                GLV_LOG( "OnLoopAfterSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }
        }
        else if ( events.GetAvailableBasicPadCount() > 0 )
        {
            const glv::BasicPadEventType& bpad = events.GetBasicPad( 0 );
            // SELECT + START 同時押し( 後押し許可版 )
            const glv::BasicPadEventType::ButtonSetType exit( glv::BasicPadEventType::Button::Start::Mask | glv::BasicPadEventType::Button::Select::Mask );
            if ( exit == ( bpad.GetButtons() & exit )  )
            {
                // アプリケーションループ退場要求
                glv::ApplicationFrameworkExit();
            }

            if ( bpad.IsButtonDown<glv::BasicPadEventType::Button::L>() )
            {
                GLV_LOG( "OnLoopAfterSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }

        }

        return glv::RequiredRestoration::RequireRestrationNothing;
    }

private:
};

void ShowToolInformation(RootSurfaceContext* context)
{
    // タイトルラベル表示
    auto toolInformationLabel = new glv::Label;
    nn::repair::GetToolInformationLabel(toolInformationLabel, ToolName, ToolMajorVersion, ToolMinorVersion);
    *context << toolInformationLabel;
}

void ShowFirmwareVersion(nn::repair::StreamTextView* view )
{
    glv::Label firmwareVersionLabel;
    nn::repair::GetFirmwareVersionLabel(&firmwareVersionLabel);
    view->AppendValue(firmwareVersionLabel.getValue() + "\n");
}

void ShowSerialNumber(nn::repair::StreamTextView* view)
{
    glv::Label serialNumberLabel;
    nn::repair::GetSerialNumberLabel(&serialNumberLabel);
    view->AppendValue(serialNumberLabel.getValue() + "\n");
}

void ShowInitialLaunchTime(nn::repair::StreamTextView* view, bool* pIsCompletionStartup)
{
    ::nn::Result result;
    std::ostringstream oss;
    std::string txt;

    nn::settings::system::InitialLaunchSettings ils;
    nn::settings::system::GetInitialLaunchSettings(&ils);

    // 初回起動済みか否か
    if(ils.flags.Test<nn::settings::system::InitialLaunchFlag::IsCompleted>())
    {
        *pIsCompletionStartup = true;
        view->AppendValue(DispMsgStartupDone);
    }
    else
    {
        *pIsCompletionStartup = false;
        view->AppendValue(DispMsgStartupNotYet);
    }

    view->AppendValue("\n");
}

void ShowDeviceId(nn::repair::StreamTextView* view)
{
    ::nn::Result result;

    // device id の定義 ： デバイス名 = prefix(2byte) + device id(16 byte 16進数文字表記) + "-" + ["1" or "0" (0がPROD)]
    glv::Label deviceIdLabel;
    nn::repair::GetDeviceIdLabel(&deviceIdLabel);

    view->AppendValue(deviceIdLabel.getValue() + "\n");
}

//!--------------------------------------------------------------------------------------
//! @brief メイン
//!--------------------------------------------------------------------------------------

void MyMain() NN_NOEXCEPT
{
    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    RootSurfaceContext* context = new RootSurfaceContext( width, height );

    // タイトルラベル表示
    ShowToolInformation(context);

    // 修理ライブラリ初期化
    nn::repair::Initialize();

    // time:u 権限必要
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    // ログビュー表示
    s_pTextView = new nn::repair::StreamTextView(glv::Rect(1024, 440), 28.f);
    s_pTextView->pos(120, 164);
    *context << s_pTextView;

    // serial number (set:cal 権限必要)
    ShowSerialNumber(s_pTextView);

    // device id を得る
    ShowDeviceId(s_pTextView);

    // firmware version
    ShowFirmwareVersion(s_pTextView);

    // 初回起動日時
    ShowInitialLaunchTime(s_pTextView, &IsCompletionStartup);

    // 一行あける
    s_pTextView->AppendValue("\n");

    if ( nn::repair::IsAvailableAutoCup() )
    {
        s_pTextView->AppendValue(DispMsgAboutFirmware2);
    }
    else
    {
        s_pTextView->AppendValue(DispMsgAboutFirmware1);

        if(!IsCompletionStartup)
        {
            // スキップボタンの表示
            s_skipButton = new nn::repair::LabelButton(DispMsgButtonSkip, MyMouseDownHandlerSkip );
            s_skipButton->pos(300, 640);
            *context << s_skipButton;
        }
        else
        {
            s_pTextView->AppendValue("\n");
            s_pTextView->AppendValue(DispMsgAboutFirmware1B);
        }
    }
    s_pTextView->AppendValue("\n");

    // シャットダウンボタン(FW 3.0.0 以降で有効)
    glv::Button* shutdownButton = new nn::repair::ShutdownButton();
    shutdownButton->pos(800,640);
    *context << shutdownButton;

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    context -> setGLV(*context);

    // メインループコールバックを登録.
    ApplicationFrameworkRegisterLoopCallback( context );

    glv::Application::run();

    // 修理ライブラリ終了
    nn::repair::Finalize();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());

    if ( nullptr != context )
    {
        delete context;
    }
}

//!--------------------------------------------------------------------------------------
//! @brief プロセスエントリポイント
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nnMain() NN_NOEXCEPT
{
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(LocalHidConfiguration);

    MyMain();
}
