﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "RepairAbortMigration.h"
#include <nn/util/util_CharacterEncoding.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_Account.h>
#include <nn/settings/system/settings_SystemApplication.h>

#ifdef FUNCTIONAL_DEBUG

void ShowSystemSaveDataList() NN_NOEXCEPT
{
    nn::fs::SaveDataSpaceId m_SpaceId = nn::fs::SaveDataSpaceId::System;
    std::unique_ptr<nn::fs::SaveDataIterator> iter;

    nn::fs::OpenSaveDataIterator(&iter, m_SpaceId);

    NN_LOG("====================================================\n");

    for ( ; ; )
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        iter->ReadSaveDataInfo(&count, &info, 1);

        if (count == 0)
        {
            break;
        }

        // ボリューム情報(for DEBUG)
        NN_LOG( "\nSaveData %d, %llX, %llX, %llX, %llX, %llX-%llX",
                info.saveDataType, info.saveDataId, info.systemSaveDataId,
                info.applicationId.value, info.saveDataSize, info.saveDataUserId._data[0], info.saveDataUserId._data[1]);

        uint32_t keepFlags1 = nn::fs::SaveDataFlags_KeepAfterResettingSystemSaveData;
        uint32_t keepFlags2 = nn::fs::SaveDataFlags_KeepAfterRefurbishment;
        uint32_t keepFlags3 = nn::fs::SaveDataFlags_KeepAfterResettingSystemSaveDataWithoutUserSaveData; // ユーザセーブデータを残す初期化で削除されない
        uint32_t flags;

        auto result = nn::fs::GetSaveDataFlags(&flags, info.saveDataId);
        if( !result.IsFailure() )
        {
            if ((flags & keepFlags1) == 0)
            {
                NN_LOG(" -"); // delete
            }
            else
            {
                NN_LOG(" L");
            }

            if ((flags & keepFlags2) == 0)
            {
                NN_LOG(" -"); // delete
            }
            else
            {
                NN_LOG(" L");
            }

            if ((flags & keepFlags3) == 0)
            {
                NN_LOG(" -"); // delete
            }
            else
            {
                NN_LOG(" L");
            }
        }

        if( info.saveDataType == nn::fs::SaveDataType::System )
        {
#if 0
            // mount してみる(FsAccessControlDescriptor SaveDataBackup 権限が必要)
            nn::Result result = nn::fs::MountSystemSaveData(g_VolumeSavedata.c_str(), m_SpaceId, info.systemSaveDataId, info.saveDataUserId);

            if (result.IsSuccess())
            {
                NN_LOG(" o");
                nn::fs::Unmount(g_VolumeSavedata.c_str());
            }
            else
            {
                NN_LOG(" x");
                NN_LOG("\n MountSystemSaveData %08x\n", result.GetInnerValueForDebug());
            }
#endif

        }

    }
    NN_LOG("\n");
    NN_LOG("====================================================\n");

}

void ShowUserSaveDataList() NN_NOEXCEPT
{
    nn::fs::SaveDataSpaceId m_SpaceId = nn::fs::SaveDataSpaceId::User;
    std::unique_ptr<nn::fs::SaveDataIterator> iter;

    nn::fs::OpenSaveDataIterator(&iter, m_SpaceId);

    NN_LOG("====================================================\n");

    for ( ; ; )
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        iter->ReadSaveDataInfo(&count, &info, 1);

        if (count == 0)
        {
            break;
        }

        // ボリューム情報(for DEBUG)
        NN_LOG( "\nSaveData %d, %llX, %llX, %llX, %llX, %llX-%llX",
                info.saveDataType, info.saveDataId, info.systemSaveDataId,
                info.applicationId.value, info.saveDataSize, info.saveDataUserId._data[0], info.saveDataUserId._data[1]);

    }
    NN_LOG("\n");
    NN_LOG("====================================================\n");
}

void CreateMigrtionSaveDataForDebug(uint64_t savedataid, uint64_t process_id,
                                    MigrationRole role, nn::account::Uid uid) NN_NOEXCEPT
{
    // abort 禁止
    nn::fs::ScopedAutoAbortDisabler scopedAbortDisabler;

    // 一旦消す
    nn::Result result;
    result = DeleteSystemSaveData(nn::fs::SaveDataSpaceId::System , savedataid, nn::fs::InvalidUserId);
    NN_LOG("\n DeleteSystemSaveData %08x\n", result.GetInnerValueForDebug());

    // volume 作る
    result = nn::fs::CreateSystemSaveData(nn::fs::SaveDataSpaceId::System
                                          , savedataid
                                          , nn::fs::InvalidUserId
                                          , process_id
                                          , 0x100000
                                          , 0x100000
                                          , 0);

    NN_LOG("\n CreateSystemSaveData %08x\n", result.GetInnerValueForDebug());

    result = nn::fs::MountSystemSaveData( g_VolumeSavedata.c_str() ,
                                          nn::fs::SaveDataSpaceId::System ,
                                          savedataid,
                                          nn::fs::InvalidUserId);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(g_VolumeSavedata.c_str());
    };

    NN_LOG("\n MountSystemSaveData %08x\n", result.GetInnerValueForDebug());

    const std::string targetFilePath = g_VolumeSavedata + ":/" + "migrationInfo.bin";
    result = nn::fs::CreateFile(targetFilePath.c_str(), 64);
    NN_LOG("\n CreateFile %08x\n", result.GetInnerValueForDebug());

    {
        nn::fs::FileHandle handle;

        result = nn::fs::OpenFile(&handle, targetFilePath.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        NN_LOG("\n OpenFile %08x\n", result.GetInnerValueForDebug());
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        struct nn::migration::user::ClientInfo destInfo;
        struct nn::migration::user::ServerInfo originInfo;
        size_t write_size = 0;
        const size_t BufferSize = 64;
        std::unique_ptr<char[]> buffer(new char[BufferSize]);

        // format check
        // fwVersion may have to be checked in the future.(currently ignored)
        if (role == Role_Origin)
        {
            originInfo.user = uid; // 使うのはこのデータだけ
            originInfo.fwVersion = 0x12345678;
            originInfo.sessionId.FromString("DEAD0000-1234-5678-0000-00000000BEEF");

            write_size = sizeof(originInfo);
            std::memcpy(buffer.get(), &originInfo, write_size);
        }
        else if (role == Role_Destination)
        {
            destInfo.serverInfo.user = uid; // 使うのはこのデータだけ
            destInfo.serverInfo.fwVersion = 0x12345678;
            destInfo.serverInfo.sessionId.FromString("DEAD0000-1234-5678-0000-00000000BEEF");

            write_size = sizeof(destInfo);
            std::memcpy(buffer.get(), &destInfo, write_size);
        }

        result = nn::fs::WriteFile(handle, 0, buffer.get(), write_size, nn::fs::WriteOption());
        NN_LOG("\n WriteFile %08x\n", result.GetInnerValueForDebug());

        result = nn::fs::FlushFile(handle);
        NN_LOG("\n FlushFile %08x\n", result.GetInnerValueForDebug());
    }

    result = nn::fs::CommitSaveData(g_VolumeSavedata.c_str());
    NN_LOG("\n CommitSaveData %08x\n", result.GetInnerValueForDebug());

    // Ocean migration flag をオン
    nn::settings::system::AppletLaunchFlagSet flags;
    nn::settings::system::GetAppletLaunchFlags(&flags);

    flags.Set<nn::settings::system::AppletLaunchFlag::Migration>(true);

    nn::settings::system::SetAppletLaunchFlags(flags);


}

void PickupRole8State(MigrationRole * pRole, MigrationState *pState) NN_NOEXCEPT
{

    // *pState = State_None;
    // *pState = State_Initialize;
    // *pState = State_Transfer;
    *pState = State_BeforeFinalize_PreDefined;
    // *pState = State_BeforeFinalize_Delete;
    // *pState = State_Finalize;
    // *pState = State_OceanFinalize;

    if( *pRole == Role_Unknown )
    {
        *pRole = Role_Origin;
        // migration_role  = Role_Destination;
    }


}


void PickupUserFromAccountList(nn::account::Uid *puid) NN_NOEXCEPT
{
    static const int arraysize = 32;
    nn::account::Uid users[arraysize];
    int usercount = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::account::ListAllUsers(&usercount, users, arraysize));

    for(int i=0;i<usercount;i++)
    {
        NN_LOG("user %d: %016llx %016llx \n", i, users[i]._data[0], users[i]._data[1] );
    }

    if( puid != nullptr )
    {
        for(int i=0;i<usercount;i++)
        {
            if( nn::ec::system::HasDeviceLink(users[i]) )
            {
                *puid = users[i];
                return;
            }
        }

        if( usercount >= 2 )
        {
            *puid = users[1];
        }
        else if (usercount > 0)
        {
            *puid = users[0];
        }
    }
}

#endif
