﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{LdnStation.cpp,PageSampleLdnStation}
 *
 * @brief LDN ライブラリのサンプルプログラム（ステーション）
 */

/**
 * @page PageSampleLdnStation ステーションのサンプルプログラム（削除予定）
 *
 * @tableofcontents
 *
 * @brief LDN ライブラリのサンプルプログラム（ステーション）の解説です。
 *
 * @section PageSampleLdnStation_SectionBrief 概要
 * LDN ライブラリのステーションのサンプルです。
 * アクセスポイント側のサンプルは @link PageSampleLdnAccessPoint LdnAccessPoint @endlink です。
 *
 * @attention
 * LdnAccessPoint と LdnStation は削除が予定されています。
 * 代わりに新しく追加された @link PageSampleLdnStation LdnSimple @endlink を参照してください。
 *
 * @section PageSampleLdnStation_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/LdnBasic/LdnStation
 * Samples/Sources/Applications/LdnBasic/LdnStation @endlink 以下にあります。
 *
 * @section PageSampleLdnStation_SectionHowToOperate 操作方法
 * サンプルプログラムを実行するとログメッセージで操作方法が表示されます。
 * この操作方法に従い、デバッグパッドで操作してください。
 *
 * @section PageSampleLdnStation_SectionPrecaution 注意事項
 * このサンプルプログラムは 2.4GHz 帯を使用して無線通信を行いますので、
 * 同じ周波数帯を使用する無線通信機器が多数存在する環境は避けてください。
 * 電波状況が悪いと以下のような問題が多発します。
 *
 * - 接続対象の LDN ネットワークが見つからない
 * - LDN ネットワークへの接続に失敗する
 * - UDP 通信のパケットロス率が高くなる
 *
 * 事情があって電波状況を改善できない場合、サンプルプログラムを修正することで 5GHz 帯を使用できます。
 * nn::ldn::Scan() の引数として渡す無線チャンネルを
 * nn::ldn::AutoChannel から 36, 40, 44, 48 のいずれかに変更してください。
 * 併せて @link PageSampleLdnAccessPoint LdnAccessPoint @endlink サンプルも変更が必要になります。
 * 詳細は @link PageSampleLdnAccessPoint LdnAccessPoint @endlink サンプルの解説を参照してください。
 *
 * @section PageSampleLdnStation_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleLdnStation_SectionDetail 解説
 * LDN ライブラリは、中継器やインターネットを介さずに複数のターゲットを接続し、
 * それぞれのターゲット上で動作するアプリケーション間で通信する機能を提供します。
 *
 * LDN ライブラリを使用すると最大 8 台のターゲットを接続して通信できます。
 * 最初に 1 台のターゲットが「アクセスポイント」となり、LDN ネットワークを構築し、
 * 他のターゲットは「ステーション」として LDN ネットワークを探索して接続します。
 *
 * このサンプルプログラムではステーションとして LDN ネットワークに接続します。
 * 事前に @link PageSampleLdnAccessPoint LdnAccessPoint @endlink
 * サンプルでアクセスポイントを起動して LDn ネットワークを構築しておいてください。
 * LDN ネットワークに接続した後は SOCKET ライブラリを用いて UDP 通信を行います。
 * 全てのターゲットは独自にカウンタをもっており、そのカウンタの値を全員で共有します。
 * 具体的には、ステーションは自身のカウンタの値をユニキャストでアクセスポイントに送信し、
 * アクセスポイントは自身を含む全員のカウンタの値をブロードキャストでステーションに配信します。
 */

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include "../Common/Definition.h"
#include "../Common/Util.h"

nn::ldn::NetworkInfo g_ScanResult[nn::ldn::ScanResultCountMax];

namespace
{
    const int testNum = 16;
    const int retryCount = 1;
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

NN_ALIGNAS(4096) char g_SendThreadStack[16384];
NN_ALIGNAS(4096) char g_LdnThreadStack[8192];
NN_ALIGNAS(4096) char g_MonitorThreadStack[8192];

int SendThreadPriority      = nn::os::DefaultThreadPriority + 4;
int LdnThreadPriority       = nn::os::DefaultThreadPriority + 2;
int MonitorThreadPriority   = nn::os::DefaultThreadPriority + 1;

nn::os::Mutex g_LogMutex(false);
bool SendRecieveThreadResult = false;

enum ApplicationState
{
    ApplicationState_Initialized,
    ApplicationState_Connecting,
    ApplicationState_Connected,
    ApplicationState_Disconnecting,
    ApplicationState_Disconnected
};

struct ApplicationResource
{
    nn::os::ThreadType          monitorThread;
    nn::os::ThreadType          sendThread;
    nn::os::ThreadType          receiveThread;
    nn::os::SystemEventType     stateChangeEvent;
    nn::os::EventType           cancelEvent;
    nn::os::MutexType           mutex;
    nn::ldn::NetworkInfo        networkInfo;
    nn::ldn::Ipv4Address        ipv4Address;
    nn::ldn::SubnetMask         subnetMask;
    uint32_t                    myCounter;
    uint32_t                    counter[nn::ldn::NodeCountMax];
    int                         socket;
    ApplicationState            state;
};

void MonitorThread(void* arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(arg);

    nn::Result result;
    nn::ldn::NetworkInfo networkInfo;

    // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::MultiWaitType       multiWait;
    nn::os::MultiWaitHolderType stateChangeHolder;
    nn::os::MultiWaitHolderType cancelHolder;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &app.stateChangeEvent);
    nn::os::InitializeMultiWaitHolder(&cancelHolder, &app.cancelEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);
    while (nn::os::WaitAny(&multiWait) != &cancelHolder)
    {
        // WaitAny では必ず手動でシグナルをクリアしなければなりません。
        nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

        // ネットワークが破棄された場合には監視を終了します。
        nn::ldn::State ldnState = nn::ldn::GetState();
        if (ldnState != nn::ldn::State_StationConnected)
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }

        // 現在のネットワークの状態を取得します。
        result = nn::ldn::GetNetworkInfo(&networkInfo);
        if (result <= nn::ldn::ResultInvalidState())
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        nn::os::LockMutex(&app.mutex);
        std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
        nn::os::UnlockMutex(&app.mutex);
    }

    nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
    nn::os::UnlinkMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWait(&multiWait);
}

void SendRecieveThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);

    // ブロードキャスト送信用のアドレスを生成します。
    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);
    addr.sin_addr.S_addr = nn::socket::InetHtonl(nn::ldn::BroadcastIpv4Address.raw);

    // 16回パケットを送信して受信する
    bool result[testNum] = {};
    for(int i = 0; i < testNum; i++)
    {
        // 自身のカウンタをインクリメントします。
        nn::os::LockMutex(&app.mutex);
        app.counter[0] += 1;

        // 送信データを生成します。
        AccesspointData data;
        nn::os::GenerateRandomBytes(data.counter, dataSize);

        // 現在の状態をログ出力します。
        {
            std::lock_guard<nn::os::Mutex> lock(g_LogMutex);
            NN_LOG("[Counter]\n");
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (app.networkInfo.ldn.nodes[i].isConnected)
                {
                    NN_LOG("Node %d : %u\n", i, app.counter[i]);
                }
            }
            NN_LOG("\n");
        }

        // パケットを送信します。
        nn::socket::SendTo(
            app.socket,
            &data,
            sizeof(data),
            nn::socket::MsgFlag::Msg_None,
            reinterpret_cast<nn::socket::SockAddr*>(&addr),
            sizeof(addr));

        // アクセスポイントからパケットを受信するためのアドレスを生成します。
        const auto& ap = app.networkInfo.ldn.nodes[0];
        nn::socket::SockAddrIn receiveAddr;
        receiveAddr.sin_family = nn::socket::Family::Af_Inet;
        receiveAddr.sin_port = nn::socket::InetHtons(UdpPort);
        receiveAddr.sin_addr.S_addr = nn::socket::InetHtonl(ap.ipv4Address.raw);;

        // パケットの受信まで待機します。失敗したらその時点でスレッドを終了します。
        nn::socket::SockLenT length = sizeof(receiveAddr);
        AccesspointData recvData;
        auto size = nn::socket::RecvFrom(
            app.socket,
            &recvData,
            sizeof(recvData),
            nn::socket::MsgFlag::Msg_None,
            reinterpret_cast<nn::socket::SockAddr*>(&receiveAddr),
            &length);
        if (size == 0 || size == -1)
        {
            break;
        }

        // 一致していることを確認
        result[i] = std::memcmp(data.counter, recvData.counter, dataSize) == 0;
        nn::os::UnlockMutex(&app.mutex);
    }

    for (int i = 0; i < testNum; i++)
    {
        if (!result[i])
        {
            SendRecieveThreadResult = false;
            NN_LOG("Send-Receive test failed\n");
            return;
        }
    }
    SendRecieveThreadResult = true;
    NN_LOG("Send-Receive test succeeded\n");
}

nn::Result Scan(ApplicationResource* pApp, int* pOutScanResultCount) NN_NOEXCEPT
{
    // IntentId が一致する LDN ネットワークのみを探索対象とします。
    nn::ldn::ScanFilter filter = { };
    filter.networkType = nn::ldn::NetworkType_Ldn;
    filter.networkId.intentId = BasicSampleIntentId;
    filter.flag = nn::ldn::ScanFilterFlag_IntentId | nn::ldn::ScanFilterFlag_NetworkType;

    // LDN ネットワークを探索します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::Scan(
        g_ScanResult, pOutScanResultCount, nn::ldn::ScanResultCountMax,
        filter, nn::ldn::AutoChannel));

    return nn::ResultSuccess();
}

nn::Result Connect(ApplicationResource* pApp, const nn::ldn::NetworkInfo& network) NN_NOEXCEPT
{
    // 接続に必要なセキュリティパラメータを設定します。
    nn::ldn::SecurityConfig security = {};
    nn::ldn::UserConfig     user     = {};
    security.securityMode   = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Any);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    std::strncpy(user.userName, "Station", nn::ldn::UserNameBytesMax);

    // LDN ネットワークに接続します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::Connect(
        network, security, user, 0, nn::ldn::ConnectOption_None));

    // ネットワークの情報を取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetNetworkInfo(&pApp->networkInfo));

    // 自身の IPv4 アドレスとサブネットマスクを取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetIpv4Address(&pApp->ipv4Address, &pApp->subnetMask));

    // ネットワークの状態を監視するスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->monitorThread, MonitorThread, pApp,
        g_MonitorThreadStack, sizeof(g_MonitorThreadStack),
        MonitorThreadPriority));
    nn::os::StartThread(&pApp->monitorThread);

    return nn::ResultSuccess();
}

void Disconnect(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ネットワークの状態を監視するスレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->monitorThread);
    nn::os::DestroyThread(&pApp->monitorThread);

    // ネットワークから切断します。
    nn::ldn::Disconnect();
}

void CreateSocket(ApplicationResource* pApp) NN_NOEXCEPT
{
    // データの送受信に使用する socket を生成します。
    pApp->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    NN_ABORT_UNLESS_NOT_EQUAL(pApp->socket, -1);
    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);
    addr.sin_addr.S_addr = nn::socket::InAddr_Any;
    NN_ABORT_UNLESS_EQUAL(nn::socket::Bind(
        pApp->socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)), 0);

    // ブロードキャスト通信を有効化します。
    int isEnabled = 1;
    nn::socket::SetSockOpt(
        pApp->socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &isEnabled, sizeof(isEnabled));

    // データ送信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->sendThread, SendRecieveThread, pApp,
        g_SendThreadStack, sizeof(g_SendThreadStack),
        SendThreadPriority));
    nn::os::StartThread(&pApp->sendThread);

    // スレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->sendThread);
    nn::os::DestroyThread(&pApp->sendThread);
    NN_LOG("CreateSocket finished\n");

    nn::os::SignalEvent(&pApp->cancelEvent);
}

void DestroySocket(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ソケットを破棄します。
    nn::socket::Shutdown(pApp->socket, nn::socket::ShutdownMethod::Shut_RdWr);
    nn::socket::Close(pApp->socket);
}

void LdnStation(void* arg) NN_NOEXCEPT
{
    nn::Result result;
    auto& app = *static_cast<ApplicationResource*>(arg);
    int scanResultCount = 0;

    for (int i = 0; i < retryCount; i++)
    {
        // ステーションとして起動します。
        result = nn::ldn::OpenStation();
        if (result <= nn::ldn::ResultInvalidState())
        {
            NN_LOG("OpenStation failed\n");
            continue;
        }

        // LDN ネットワークを探索します。
        NN_LOG("LDN ネットワークを探索中です・・・。\n\n");

        result = Scan(&app, &scanResultCount);
        if (result <= nn::ldn::ResultInvalidState())
        {
            NN_LOG("Scan failed");
            continue;
        }
        else if (scanResultCount == 0)
        {
            NN_LOG("接続対象のネットワークが見つかりませんでした。\n\n");
            nn::ldn::CloseStation();
            app.state = ApplicationState_Disconnected;
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
            continue;
        }
    }

    if (scanResultCount == 0)
    {
        NN_LOG("接続対象のネットワークが見つかりませんでした2。\n");
        app.state = ApplicationState_Disconnected;
        return;
    }

    // LDN ネットワークへの接続を試行します。
    result = Connect(&app, g_ScanResult[0]);
    if (result <= nn::ldn::ResultInvalidState() ||
        result <= nn::ldn::ResultConnectionFailed() ||
        result <= nn::ldn::ResultIncompatibleVersion())
    {
        NN_LOG("LDN ネットワークへの接続に失敗しました。\n\n");
        Disconnect(&app);
        nn::ldn::CloseStation();
        app.state = ApplicationState_Disconnected;
        return;
    }

    {
        std::lock_guard<nn::os::Mutex> lock(g_LogMutex);
        NN_LOG("LDN ネットワークへの接続に成功しました。\n");
        NN_LOG("Name (AP)        : %s\n", app.networkInfo.ldn.nodes[0].userName);
        NN_LOG("IP Address (AP)  : %s/%s\n",
            ConvertToString(app.networkInfo.ldn.nodes[0].ipv4Address),
            ConvertToString(app.subnetMask));
        NN_LOG("IP Address (STA) : %s/%s\n\n",
            ConvertToString(app.ipv4Address), ConvertToString(app.subnetMask));
    }
    app.state = ApplicationState_Connected;

    // データの送受信を開始します.
    CreateSocket(&app);

    // LDN ネットワークから切断し、データの送受信を終了します。
    DestroySocket(&app);
    Disconnect(&app);
    nn::ldn::CloseStation();
    app.state = ApplicationState_Disconnected;
    NN_LOG("LDN ネットワークから切断されました。\n\n");
}

}

bool LdnStationMain()
{
    nn::Result result;

    // LDN ライブラリを初期化します。
    result = nn::ldn::Initialize();
    if (result <= nn::ldn::ResultDeviceOccupied())
    {
        // 他の通信機能を使用中のためローカル通信機能を利用できません。
        // 通信機能を終了してから再度 LDN ライブラリを初期化してください。
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // socket ライブラリを初期化します。
    result = nn::socket::Initialize(g_SocketConfigWithMemory);

    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // アプリケーションが使用するリソースを初期化します。
    ApplicationResource app;
    std::memset(&app, 0, sizeof(app));

    // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
    nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);

    // メインループです。
    nn::os::ThreadType ldnThread;

    app.state = ApplicationState_Connecting;
    NN_LOG("LDN ネットワークをスキャンして接続します。\n\n");
    nn::os::InitializeEvent(
        &app.cancelEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMutex(&app.mutex, false, 0);
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &ldnThread, LdnStation, &app,
        g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
    nn::os::StartThread(&ldnThread);

    app.state = ApplicationState_Initialized;
    nn::os::WaitThread(&ldnThread);
    nn::os::DestroyThread(&ldnThread);
    nn::os::FinalizeMutex(&app.mutex);
    nn::os::FinalizeEvent(&app.cancelEvent);

    // socket ライブラリの使用を終了します。
    result = nn::socket::Finalize();
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ライブラリの使用を終了します。
    nn::ldn::Finalize();

    return SendRecieveThreadResult;
}
