﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <chrono>
#include <functional>

#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/util/util_FormatString.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/oe/oe_PowerStateControlApi.private.h>

#include "PreinstallAppWriter_LabelButton.h"
#include "PreinstallAppWriter_Status.h"
#include "PreinstallAppWriter_Progress.h"

namespace PreinstallAppWriter
{

nn::Result GetInstallResult();
InstallStatus GetInstallStatus();
std::string GetPreinstallVersion();
std::string GetErrorMessage();

namespace {

static const char* ToolName = "NX Preinstall App Writer";
const int ToolMajorVersion = 1;
const int ToolMinorVersion = 0;

static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;
glv::Label g_ToolInformationLabel;
glv::Label g_ProgressLabel;
glv::Label g_VersionLabel;
glv::Label g_SpeedLabel;
glv::Label g_ResultLabel;
glv::Label g_ErrorMessageLabel;
LabelButton* s_pLabelButton;
nn::os::ThreadType g_OperationThread;

const int OperationStackSize = 50 * 4096;
NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_OperationStack[OperationStackSize];

glv::Slider* g_pProgressBar;

void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

void Deallocate(void* address, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(address);
}

void* Reallocate(void* address, size_t size, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(address, size);
}

void SetupPeripherals() NN_NOEXCEPT
{
    nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
    glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
}


glv::GLV s_GlvRootView;

void DisplayToolInformation(glv::GLV* const pGlvRootView) NN_NOEXCEPT
{
    char buffer[1024];
    nn::util::SNPrintf(buffer, 1024, "%s\t\t\t\t\t\tVer.%d.%d", ToolName, ToolMajorVersion, ToolMinorVersion);
    g_ToolInformationLabel.setValue(std::string(buffer));
    g_ToolInformationLabel.pos(10, 25);
    g_ToolInformationLabel.size(32.0f);
    *pGlvRootView << g_ToolInformationLabel;
}

void ShutdownOperation()
{
    nn::oe::RequestToShutdown();
}

void UpdateSpeedInfo(std::chrono::system_clock::time_point start, ReadOnlyProgressInfo* progressInfo)
{
    auto now = std::chrono::system_clock::now();
    auto elapsed = std::chrono::duration_cast<std::chrono::seconds>( now - start ).count();
    char speedStr[256];
    auto mBPerSec = progressInfo->GetInstallThroughput().elapsedTime.GetMilliSeconds() != 0 ?
            static_cast<double>( progressInfo->GetInstallThroughput().installed ) / static_cast<double>( 1024 )
            / static_cast<double>( 1024 ) / static_cast<double>( progressInfo->GetInstallThroughput().elapsedTime.GetMilliSeconds() / static_cast<double>(1000) )
            : 0;
    nn::util::SNPrintf(
        speedStr, sizeof( speedStr ), "Elapsed: %d seconds, %.1f MB/s", static_cast<int>( elapsed ), mBPerSec);
    g_SpeedLabel.setValue( std::string( speedStr ) );
}

void UpdateTimeInfo(std::chrono::system_clock::time_point start)
{
    auto now = std::chrono::system_clock::now();
    auto elapsed = std::chrono::duration_cast<std::chrono::seconds>( now - start ).count();
    char timeStr[256];
    nn::util::SNPrintf(
        timeStr, sizeof( timeStr ), "Elapsed: %d seconds", static_cast<int>( elapsed ));
    g_SpeedLabel.setValue( std::string( timeStr ) );
}

void DisplayResult()
{
    const int BufferSize = 200;
    char buffer[BufferSize];
    if(GetInstallStatus() == Error)
    {
        auto styleRed = new glv::Style();
        styleRed->color.text.set( 1.0, 0.0, 0.0 );
        g_ResultLabel.style( styleRed );
        g_ErrorMessageLabel.style( styleRed );

        nn::util::SNPrintf( buffer, BufferSize, "Fail. Error Code = 0x%08x.\n",
                            GetInstallResult().GetInnerValueForDebug() );
        g_ResultLabel.setValue( std::string( buffer ) );
        g_ErrorMessageLabel.setValue(GetErrorMessage());
    }
    else if(GetInstallStatus() == Done)
    {
        auto styleGreen = new glv::Style();
        styleGreen->color.text.set( 0.0, 1.0, 0.0 );
        g_ResultLabel.style( styleGreen );

        g_ResultLabel.setValue("Success");
    }
}

void Work(void* p)
{
    auto progressInfo = reinterpret_cast<ReadOnlyProgressInfo*>(p);
    auto start = std::chrono::system_clock::now();
    for(;;)
    {
        if(GetPreinstallVersion().length() != 0)
        {
            char versionStr[256];
            nn::util::SNPrintf( versionStr, sizeof( versionStr ), "Version %s",
                                GetPreinstallVersion().c_str() );
            g_VersionLabel.setValue( std::string( versionStr ) );
        }

        if(GetInstallStatus() == Prepare)
        {
            g_ProgressLabel.setValue("Connect AC Adapter");
        }

        if(GetInstallStatus() == Installing)
        {
            char progressStr[256];
            auto progressRatio = progressInfo->GetInstallTotalSize() != 0 ?
                static_cast<double>( progressInfo->GetInstalledSize() + progressInfo->GetCurrentInstallSize())
                / static_cast<double>( progressInfo->GetInstallTotalSize())
                : 0;
            nn::util::SNPrintf(
                progressStr, sizeof( progressStr ), "Installing (%d of %d) %lld / %lld (%.1f%)",
                progressInfo->GetInstalledCount() + 1,
                progressInfo->GetInstallTotalCount(),
                progressInfo->GetInstalledSize() + progressInfo->GetCurrentInstallSize(),
                progressInfo->GetInstallTotalSize(),
                progressRatio * 100);
            g_ProgressLabel.setValue( std::string( progressStr ) );
            g_pProgressBar->setValue( progressRatio );

            UpdateSpeedInfo( start, progressInfo );
        }

        if(GetInstallStatus() == Verifying)
        {
            char progressStr[256];
            auto progressRatio = progressInfo->GetVerifyTotalSize() ?
                static_cast<double>( progressInfo->GetVerifiedSize() + progressInfo->GetCurrentVerifySize())
                / static_cast<double>( progressInfo->GetVerifyTotalSize())
                : 0;
            nn::util::SNPrintf(
                progressStr, sizeof( progressStr ), "Verifying (%d of %d) %lld / %lld (%.1f%)",
                progressInfo->GetVerifiedCount() + 1,
                progressInfo->GetVerifyTotalCount(),
                progressInfo->GetVerifiedSize() + progressInfo->GetCurrentVerifySize(),
                progressInfo->GetVerifyTotalSize(),
                progressRatio * 100);
            g_ProgressLabel.setValue( std::string( progressStr ) );
            g_pProgressBar->setValue( progressRatio );

            UpdateTimeInfo( start );
        }

        if( GetInstallStatus() == Error ||
            GetInstallStatus() == Done)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    DisplayResult();
    g_ProgressLabel.setValue("Done");
    s_pLabelButton = new LabelButton("Shutdown", [&] { ShutdownOperation(); });
    s_GlvRootView << s_pLabelButton->pos(glv::Place::BC, 0, -30).anchor(glv::Place::BC);
}

nn::Result Initialize(ReadOnlyProgressInfo* progressInfo) NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    DisplayToolInformation(pGlvRootView);

    //進捗
    g_ProgressLabel.size(28);
    g_ProgressLabel.setValue("Start");
    *pGlvRootView << g_ProgressLabel.pos(glv::Place::TL, 120, 100);
    g_VersionLabel.size(28);
    *pGlvRootView << g_VersionLabel.pos(glv::Place::TL, 120, 130);
    g_SpeedLabel.size(28);
    *pGlvRootView << g_SpeedLabel.pos(glv::Place::TL, 120, 160);
    g_pProgressBar = new glv::Slider(glv::Rect(1040, 100));
    auto styleGreen = new glv::Style();
    styleGreen->color.fore.set(static_cast<double>(0) / static_cast<double>(255),
        static_cast<double>(211) / static_cast<double>(255),
        static_cast<double>(40) / static_cast<double>(255));
    styleGreen->color.back.set(0, 0, 0);
    g_pProgressBar->style(styleGreen);
    *pGlvRootView << g_pProgressBar->pos(glv::Place::TL, 120, 310);
    g_ErrorMessageLabel.size(28);
    *pGlvRootView << g_ErrorMessageLabel.pos(glv::Place::TL, 120, 490);
    g_ResultLabel.size(50);
    *pGlvRootView << g_ResultLabel.pos(glv::Place::TL, 120, 550);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread( &g_OperationThread, Work, progressInfo, g_OperationStack, OperationStackSize, nn::os::DefaultThreadPriority ));
    nn::os::StartThread( &g_OperationThread );

    NN_RESULT_SUCCESS;
}

void Main( glv::Window& window, ReadOnlyProgressInfo* progressInfo ) NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    nn::Result result = Initialize(progressInfo);
    if (result.IsFailure())
    {
        const int BufferSize = 200;
        char buffer[BufferSize];
        nn::util::SNPrintf(buffer, BufferSize, "Initialization Failed. Error code = 0x%08x.\n", result.GetInnerValueForDebug());
        g_ResultLabel.setValue(std::string(buffer));
        auto styleRed = new glv::Style();
        styleRed->color.text.set(1.0, 0.0, 0.0);
        g_ResultLabel.style(styleRed);
    }

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set( 0.5 );

    window.setGLV(*pGlvRootView);
    glv::Application::run();
}

} // namespace

void View(void* p)
{
    auto progressInfo = reinterpret_cast<ReadOnlyProgressInfo*>(p);
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_ABORT_UNLESS_NOT_NULL(window);

    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    Main(*window, progressInfo);
}

}
