﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/mem.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/audio.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <string>
#include "PreinstallAppWriter_WavFormat.h"
#include <nn/audioctrl.h>

namespace PreinstallAppWriter {

enum class AUDIO_CHANNEL
{
    CH_L,
    CH_R,
    CH_LR
};
//---------------------------------------------------------------------------
//! @brief 正弦波、もしくはWavファイルの非同期再生を行うクラスです。
//!
//! @details
//---------------------------------------------------------------------------
class AudioOut
{
public:
    AudioOut() : m_IsInit(false), m_IsStartSoundThread(false), m_IsSoundThreadSuccess(false), m_IsLoop(true), m_AllocateHeap(nullptr),
        m_OutBuffer{0}, m_WavFileBuffer(nullptr)
    { }
    ~AudioOut()
    {
        if (m_IsStartSoundThread)
        {
            StopPlaySound();
        }

        if (m_IsInit)
        {
            Finalize();
        }
    }
    //! @brief 初期化を行います。
    //!
    //! @return 成否
    bool Initialize();
    //! @brief 終了処理を行います。
    //!
    void Finalize();
    //! @brief WAVファイルの再生を行います。
    //!        StopPlaySoundを呼ぶか、インスタンスが消えるまで再生を続けます。
    //!        loopをOFFにしている場合は無音再生を続けます。
    //!
    //! @param[in]  file    再生するWAVファイル。16bitPCM、2ch、48000Hzのみ対応
    //! @param[in]  _loop   ループ再生を行うかどうか(省略でループ再生OFF)
    //!
    //! @return 成否
    bool StartPlayWavFile(std::string file, bool loop = false);
    //! @brief 音声再生をストップします。
    //!
    void StopPlaySound();
    //! @brief Wavファイルをループ指定なしで再生した場合、再生が完了しているかを取得します。
    //!        StartPlayWavFileを呼んでいない場合の返り値は不定
    //!
    //! @return Wavファイルの再生が完了しているかどうか
    bool IsWavSoundFinished()
    {
        if (m_IsLoop == true)
        {
            return false;
        }
        if (m_RemindFileSize == 0)
        {
            return true;
        }
        else
        {
            return false;
        }
    }
private:
    static const int THREAD_STACK_SIZE = 81920;
    static const int BUFFER_SIZE = (THREAD_STACK_SIZE + (128 * 1024)) * 3;
    static const int BUFFER_NUM = 3;
    static void StartWavSoundThread(void* arg)
    {
        AudioOut* aOut = static_cast <AudioOut*>(arg);
        aOut->WavSoundThreadCommon();
    }
    void CopyWaveFileToBuffer(void* buffer, size_t outSize);
    void WavSoundThreadCommon();
    bool ReadWavFile(const char* filename);
    void InitializeFileSystem();
    bool m_IsInit;
    bool m_IsStartSoundThread;
    bool m_IsSoundThreadSuccess;
    nn::lmem::HeapHandle m_HeapHandle;
    bool m_IsLoop;
    nns::audio::WavFormat m_format;
    void* m_AllocateHeap;
    void* m_OutBuffer[BUFFER_NUM];
    uint8_t* m_WavFileBuffer;
    uint8_t* m_ThreadStack;
    uint32_t m_RemindFileSize;
    std::string m_SoundFilePath;
    nn::os::ThreadType m_SoundThreadType;
    nn::os::SystemEvent m_SystemEvent;
    nn::audio::AudioOut m_AudioOut;
};

}
