﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>

#include <nn/nfp/nfp_DebugTypes.h>
#include "../NoftWriter_Types.h"
#include "../npad/NoftWriter_NpadController.h"
#include "NoftWriter_SelectableWindow.h"

namespace noftwriter { namespace ui {

/**
 * @brief   Ntf ファイルの概要を表示し、書き込みを開始するかトリガするウィンドウです。
 */
class WriteNtfWindow : public SelectableWindow
{
public:
    WriteNtfWindow() NN_NOEXCEPT :
        m_CursorIndex(ItemForDecideStartIndex),
        m_DecideHandler(nullptr)
    {
        SetCursorVisible(true);
        SetSize(480, 232);
    }

    /**
     * @brief   カーソル位置を設定します。
     */
    virtual void SetCursorIndex(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        m_CursorIndex = std::min(std::max(index, 0), ItemCountMax - 1);
    }

    /**
     * @brief   カーソル位置を取得します。
     */
    virtual int GetCursorIndex() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_CursorIndex;
    }

    void SetNtf(uintptr_t content) NN_NOEXCEPT;

    /**
     * @brief   決定時のコールバック関数を登録します。
     */
    void SetDecideHandler(WindowHandlerTypeWithIndex handler) NN_NOEXCEPT
    {
        m_DecideHandler = handler;
    }

    /**
     * @brief   カーソル位置の項目の Y 座標を取得します。
     */
    virtual float GetSelectedItemY() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GetY() + GetDefaultLineHeight() * m_CursorIndex + GetHeaderHeight();
    }

    virtual void UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

    Ntf* GetNtf() NN_NOEXCEPT
    {
        return m_Ntf;
    }

    nn::nfp::NtfWriteType GetNtfWriteType() NN_NOEXCEPT;

protected:
    virtual float GetRenderHeight() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GetDefaultLineHeight() * ItemCountMax + GetHeaderHeight() + 8;
    }

private:
    static const int ItemCountMax;
    static const int ItemForDecideStartIndex;

private:
    void DrawCurrentInfo() NN_NOEXCEPT;

private:
    int                         m_CursorIndex;      //!< カーソル位置
    WindowHandlerTypeWithIndex  m_DecideHandler;    //!< 決定時のコールバック

    Ntf* m_Ntf;
};

}}  // noftwriter::ui
