﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <unordered_map>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/hid/hid_Npad.h>

#include "../NoftWriter_Types.h"
#include "../NoftWriter_Util.h"
#include "../graphics/NoftWriter_Renderer.h"
#include "../npad/NoftWriter_NpadController.h"
#include "NoftWriter_SelectableWindow.h"

namespace noftwriter { namespace ui {

/**
 * @brief   メニュー項目の内容を表す型です。
 */
struct FileListItem
{
    std::string         path;       //!< 項目名
    std::string         character;  //!< 項目名
    uintptr_t           argument;   //!< 決定時に渡される引数
};

struct DirectoryEntry
{
    std::shared_ptr<struct DirectoryEntry> parent;
    nn::fs::DirectoryEntryType type;
    char name[nn::fs::EntryNameLengthMax + 1];
};

struct DirectoryEntryComp;

/**
 * @brief   選択可能なメニューを表示するウィンドウです。
 */
class NtfFileListWindow : public SelectableWindow
{
    friend struct DirectoryEntryComp;
private:
    static const int VisibleItemCountMax = 19; //!< 見える項目の数

public:
    NtfFileListWindow() NN_NOEXCEPT :
        m_CursorIndex(-1),
        m_VisibleTopItemIndex(0),
        m_VisibleItemNum(0),
        m_IsBuildList(false),
        m_ScrollCounterForPath(0),
        m_ScrollCounterForCharacter(0)
    {
        SetCursorVisible(true);
    }

    void BuildList() NN_NOEXCEPT;
    void ReBuildListStart() NN_NOEXCEPT;

    void SetDecideHandler(WindowHandlerType handler) NN_NOEXCEPT;
    void SetSdAccessHandler(void (*handler)(const char* message, bool isError)) NN_NOEXCEPT;

    /**
     * @brief   カーソル位置を設定します。
     */
    virtual void SetCursorIndex(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        m_CursorIndex = std::min(std::max(index, 0), VisibleItemCountMax - 1);
        UpdateItem();
    }

    /**
     * @brief   カーソル位置を取得します。
     */
    virtual int GetCursorIndex() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_CursorIndex;
    }

    /**
     * @brief   カーソル位置の項目の Y 座標を取得します。
     */
    virtual float GetSelectedItemY() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GetY() + GetDefaultLineHeight() * GetCursorIndex() + GetHeaderHeight();
    }

    virtual void UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

protected:
    virtual float GetRenderHeight() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GetDefaultLineHeight() * VisibleItemCountMax + GetHeaderHeight() + 8;
    }

private:
    void UpdateItem() NN_NOEXCEPT;
    nn::Result SearchDirectory(std::shared_ptr<DirectoryEntry> target) NN_NOEXCEPT;
    nn::Result BuildAmiiboDataList(const char* path) NN_NOEXCEPT;
    void UpdateInputButtonDown() NN_NOEXCEPT;
    void UpdateInputButtonUp() NN_NOEXCEPT;
    void UpdateInputButtonRight() NN_NOEXCEPT;
    void UpdateInputButtonLeft() NN_NOEXCEPT;

    /**
     * @brief   指定したメニュー項目を実行します。
     */
    void Invoke(int cursorIndex, const nn::hid::NpadIdType& npadId) NN_NOEXCEPT;

private:
    FileListItem        m_Items[VisibleItemCountMax];           //!< 項目一覧
    int                 m_CursorIndex;      //!< カーソルの位置

    std::vector<std::shared_ptr<DirectoryEntry>> m_DirectoryEntryList;
    int64_t m_VisibleTopItemIndex;
    int m_VisibleItemNum;
    WindowHandlerType m_Handler;    //!< 決定時に実行する処理
    Ntf m_Ntf;
    std::unordered_map<std::string, std::string> m_AmiiboDataList;
    void (*m_SdAccessHandler)(const char* message, bool isError);
    bool m_IsBuildList;
    int m_ScrollCounterForPath;
    int m_ScrollCounterForCharacter;
};

}}  // noftwriter::ui
