﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define _USE_MATH_DEFINES

#include <cmath>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/hid.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include "../graphics/NoftWriter_Renderer.h"
#include "../npad/NoftWriter_NpadController.h"
#include "NoftWriter_NtfFileListWindow.h"

namespace noftwriter { namespace ui {

namespace {
    const char* SdMountName = "sd";
    const char*  MessageSdCardNotFound = "SD card not found.";
    const char*  MessageSdCardAccessError = "SD card access error.";
    const char*  MessageNtfFileNotFound = "NTF file not found.";
    const char*  MessageAmiiboDataListNotFound = "amiibo_DataList*.csv not found.";

    bool MatchExtension(const std::string& filename, const std::string& extension)
    {
        std::size_t commaPos = filename.find_last_of('.');
        if (commaPos == std::string::npos)
            return false;

        std::size_t delimiterPos = filename.find_last_of('/');
        if (delimiterPos != std::string::npos && delimiterPos > commaPos)
            return false;

        std::size_t len = filename.size() - commaPos - 1;
        if (len != extension.size())
            return false;

        for (std::size_t i = 0; i < len; i++)
        {
            if (nn::util::ToUpper(extension[i]) != nn::util::ToUpper(filename[commaPos + i + 1]))
                return false;
        }

        return true;
    }

    std::string MakePath(std::shared_ptr<DirectoryEntry> directoryEntry)
    {
        std::string path;
        std::shared_ptr<DirectoryEntry> entry = directoryEntry;

        if(entry != nullptr)
        {
            path = std::string(entry->name);
            entry = entry->parent;

            while(entry != nullptr)
            {
                path = std::string(entry->name) + "/" + path;
                entry = entry->parent;
            }
        }

        return path;
    }

    std::string AddScheamToPath(const char* path)
    {
        return std::string(SdMountName) + ":" + "/" + path;
    }

    std::string GetFront(noftwriter::graphics::Renderer& renderer, std::string str, float width)
    {
        for(int i = 1; i <= str.size(); i++)
        {
            if(width < renderer.CalculateTextWidth(str.substr(0, i).c_str()))
            {
                return str.substr(0, i - 1);
            }
        }
        return str;
    };

    std::string GetBackward(noftwriter::graphics::Renderer& renderer, std::string str, float width)
    {
        for(int i = 1; i <= str.size(); i++)
        {
            if(width < renderer.CalculateTextWidth(str.substr(str.size() - i, i).c_str()))
            {
                return str.substr(str.size() - i + 1, i - 1);
            }
        }
        return str;
    };
}

struct DirectoryEntryComp
{
    NtfFileListWindow* window;
    explicit DirectoryEntryComp(NtfFileListWindow* window)
    {
        this->window = window;
    }
    bool operator()(const std::shared_ptr<DirectoryEntry> &lhs, const std::shared_ptr<DirectoryEntry> &rhs)
    {
        const char* lCharacter = "";
        const char* rCharacter = "";

        std::unordered_map<std::string, std::string>::iterator it;
        it = window->m_AmiiboDataList.find(lhs->name);
        if(it != window->m_AmiiboDataList.end())
        {
            lCharacter = it->second.c_str();
        }

        it = window->m_AmiiboDataList.find(rhs->name);
        if(it != window->m_AmiiboDataList.end())
        {
            rCharacter = it->second.c_str();
        }

        auto ret = std::strcmp(lCharacter, rCharacter);
        if(ret == 0)
        {
            std::string lPath = MakePath(lhs);
            std::string rPath = MakePath(rhs);

            ret = std::strcmp(lPath.c_str(), rPath.c_str());
        }
        return ret < 0;
    }
};

void NtfFileListWindow::BuildList() NN_NOEXCEPT
{
    m_IsBuildList = true;
    m_CursorIndex = 0;
    m_VisibleTopItemIndex = 0;
    m_VisibleItemNum = 0;

    {
        nn::Result result = nn::fs::MountSdCardForDebug(SdMountName);
        if(result.IsFailure())
        {
            m_SdAccessHandler(MessageSdCardNotFound, true);
            return;
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(SdMountName);
        };

        m_DirectoryEntryList.clear();
        m_AmiiboDataList.clear();

        result = SearchDirectory(nullptr);
        if(result.IsFailure())
        {
            m_SdAccessHandler(MessageSdCardAccessError, true);
            return;
        }

        if(m_DirectoryEntryList.empty())
        {
            m_SdAccessHandler(MessageNtfFileNotFound, true);
            return;
        }
    }

    m_VisibleItemNum = (VisibleItemCountMax < m_DirectoryEntryList.size() ? VisibleItemCountMax : m_DirectoryEntryList.size());

    std::sort(m_DirectoryEntryList.begin(), m_DirectoryEntryList.end(), DirectoryEntryComp(this));

    UpdateItem();

    if(m_AmiiboDataList.empty())
    {
        m_SdAccessHandler(MessageAmiiboDataListNotFound, false);
    }
}

void NtfFileListWindow::ReBuildListStart() NN_NOEXCEPT
{
    m_IsBuildList = false;
}

nn::Result NtfFileListWindow::BuildAmiiboDataList(const char* path) NN_NOEXCEPT
{
    std::unique_ptr<nn::Bit8[]> data;
    int64_t  readSize;

    {
        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::GetFileSize(&readSize, handle));
        data.reset(new nn::Bit8[readSize]);
        NN_RESULT_DO(nn::fs::ReadFile(handle, 0, data.get(), readSize));
    }

    char* pos = reinterpret_cast<char*>(&data[0]);
    char* eof = reinterpret_cast<char*>(&data[0]) + readSize;
    bool isGotKey = false;
    std::string key;
    std::string value;

    // BOM ヘッダスキップ
    if(data[0] == 0xEF
       && data[1] == 0xBB
       && data[2] == 0xBF)
    {
        pos += 3;
    }

    isGotKey = false;
    while(NN_STATIC_CONDITION(true))
    {
        if(eof <= pos)
        {
            if(!key.empty() && !value.empty())
            {
                m_AmiiboDataList[key] = value;
            }
            break;
        }
        else if(*pos == '\r' || *pos == '\n')
        {
            if(!key.empty() && !value.empty())
            {
                m_AmiiboDataList[key] = value;
            }
            key.clear();
            value.clear();
            isGotKey = false;
        }
        else if(*pos == ',')
        {
            isGotKey = true;
        }
        else
        {
            if(!isGotKey)
            {
                key += *pos;
            }
            else
            {
                value += *pos;
            }
        }

        ++pos;
    }

    NN_RESULT_SUCCESS;
}

void NtfFileListWindow::SetDecideHandler(WindowHandlerType handler) NN_NOEXCEPT
{
    m_Handler = handler;
}

void NtfFileListWindow::SetSdAccessHandler(void (*handler)(const char* message, bool isError)) NN_NOEXCEPT
{
    m_SdAccessHandler = handler;
}


void NtfFileListWindow::UpdateItem() NN_NOEXCEPT
{
    for(decltype(m_VisibleItemNum) i = 0; i < m_VisibleItemNum; ++i)
    {
        m_Items[i].path = MakePath(m_DirectoryEntryList[m_VisibleTopItemIndex + i]);
        std::unordered_map<std::string, std::string>::iterator it = m_AmiiboDataList.find(m_DirectoryEntryList[m_VisibleTopItemIndex + i]->name);
        if(it == m_AmiiboDataList.end())
        {
            m_Items[i].character = "?";
        }
        else
        {
            m_Items[i].character = m_AmiiboDataList[m_DirectoryEntryList[m_VisibleTopItemIndex + i]->name];
        }
    }
}

void NtfFileListWindow::Update() NN_NOEXCEPT
{
    if(!m_IsBuildList)
    {
        BuildList();
    }

    Window::Update();
}

nn::Result NtfFileListWindow::SearchDirectory(std::shared_ptr<DirectoryEntry> target) NN_NOEXCEPT
{
    nn::Result result;
    nn::fs::DirectoryHandle directoryHandle;
    std::string path = AddScheamToPath(MakePath(target).c_str());

    NN_RESULT_DO(nn::fs::OpenDirectory(&directoryHandle, path.c_str(), nn::fs::OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(directoryHandle);
    };

    int64_t directoryEntryCount;
    std::unique_ptr<nn::fs::DirectoryEntry []> directoryEntryBuffer(new nn::fs::DirectoryEntry[8]);

    while(NN_STATIC_CONDITION(true))
    {
        NN_RESULT_DO(nn::fs::ReadDirectory(&directoryEntryCount, directoryEntryBuffer.get(), directoryHandle, 8));
        if(0 < directoryEntryCount)
        {
            for(decltype(directoryEntryCount) i = 0; i < directoryEntryCount; ++i)
            {
                // ntf ファイルに関係ないファイルやディレクトリであれば参照されないので消える
                std::shared_ptr<DirectoryEntry> directoryEntry(new DirectoryEntry);
                directoryEntry->parent = target;
                directoryEntry->type = static_cast<nn::fs::DirectoryEntryType>(directoryEntryBuffer[i].directoryEntryType);
                std::strcpy(directoryEntry->name, directoryEntryBuffer[i].name);

                if(directoryEntry->type == nn::fs::DirectoryEntryType_Directory)
                {
                    NN_RESULT_DO(SearchDirectory(directoryEntry));
                }
                else if(directoryEntry->type == nn::fs::DirectoryEntryType_File)
                {
                    if(MatchExtension(directoryEntry->name, "ntf"))
                    {
                        m_DirectoryEntryList.push_back(directoryEntry);
                    }
                    else if(MatchExtension(directoryEntry->name, "csv")
                            && std::strncmp(directoryEntry->name, "amiibo_DataList", std::strlen("amiibo_DataList")) == 0)
                    {
                        NN_RESULT_DO(BuildAmiiboDataList((path + "/" + directoryEntry->name).c_str()));
                    }
                }
            }
        }
        else
        {
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

void NtfFileListWindow::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
    if (!IsActive() || !IsVisible())
    {
        return;
    }

    if (controller.IsRepeated(nn::hid::NpadButton::Down::Mask))
    {
        UpdateInputButtonDown();
    }
    else if (controller.IsRepeated(nn::hid::NpadButton::Up::Mask))
    {
        UpdateInputButtonUp();
    }
    else if (controller.IsRepeated(nn::hid::NpadButton::Right::Mask))
    {
        UpdateInputButtonRight();
    }
    else if (controller.IsRepeated(nn::hid::NpadButton::Left::Mask))
    {
        UpdateInputButtonLeft();
    }

    if (controller.IsTriggered(nn::hid::NpadButton::A::Mask))
    {
        Invoke(m_CursorIndex, controller.GetNpadId());
    }

    if (controller.IsTriggered(nn::hid::NpadButton::X::Mask))
    {
        ReBuildListStart();
    }

    Window::UpdateInput(controller);
}

void NtfFileListWindow::UpdateInputButtonDown() NN_NOEXCEPT
{
    if(m_DirectoryEntryList.size() - 1 <= m_VisibleTopItemIndex + m_CursorIndex)
    {
        m_CursorIndex = 0;

        m_VisibleTopItemIndex = 0;
        UpdateItem();
    }
    else if(m_CursorIndex == VisibleItemCountMax - 1)
    {
        ++m_VisibleTopItemIndex;
        UpdateItem();
    }
    else
    {
        m_CursorIndex = m_CursorIndex + 1;
    }
    m_ScrollCounterForPath = 0;
    m_ScrollCounterForCharacter = 0;
}

void NtfFileListWindow::UpdateInputButtonUp() NN_NOEXCEPT
{
    if(m_VisibleTopItemIndex == 0 && m_CursorIndex == 0)
    {
        m_CursorIndex = m_VisibleItemNum - 1;

        m_VisibleTopItemIndex = m_DirectoryEntryList.size() - m_VisibleItemNum;
        UpdateItem();
    }
    else if(m_CursorIndex == 0)
    {
        --m_VisibleTopItemIndex;
        UpdateItem();
    }
    else
    {
        m_CursorIndex = m_CursorIndex - 1;
    }
    m_ScrollCounterForPath = 0;
    m_ScrollCounterForCharacter = 0;
}

void NtfFileListWindow::UpdateInputButtonRight() NN_NOEXCEPT
{
    if (VisibleItemCountMax < m_DirectoryEntryList.size())
    {
        if(m_DirectoryEntryList.size() - 1 < m_VisibleTopItemIndex + VisibleItemCountMax)
        {
            if(m_CursorIndex == VisibleItemCountMax - 1)
            {
                m_CursorIndex = 0;
                m_VisibleTopItemIndex = 0;
            }
            else
            {
                m_CursorIndex = VisibleItemCountMax - 1;
            }
        }
        else
        {
            int nextCursorItemIndex = m_VisibleTopItemIndex + VisibleItemCountMax + m_CursorIndex ;
            if(m_DirectoryEntryList.size() - 1 < nextCursorItemIndex)
            {
                nextCursorItemIndex = m_DirectoryEntryList.size() - 1;
            }

            if(m_DirectoryEntryList.size() - 1 < m_VisibleTopItemIndex + VisibleItemCountMax + VisibleItemCountMax)
            {
                m_VisibleTopItemIndex = m_DirectoryEntryList.size() - VisibleItemCountMax;
            }
            else
            {
                m_VisibleTopItemIndex += VisibleItemCountMax;
            }

            m_CursorIndex = nextCursorItemIndex - m_VisibleTopItemIndex;
        }

        UpdateItem();

        m_ScrollCounterForPath = 0;
        m_ScrollCounterForCharacter = 0;
    }
}

void NtfFileListWindow::UpdateInputButtonLeft() NN_NOEXCEPT
{
    if (VisibleItemCountMax < m_DirectoryEntryList.size())
    {
        if(m_VisibleTopItemIndex == 0)
        {
            if(m_CursorIndex == 0)
            {
                m_CursorIndex = VisibleItemCountMax - 1;
                m_VisibleTopItemIndex = m_DirectoryEntryList.size() - VisibleItemCountMax;
            }
            else
            {
                m_CursorIndex = 0;
            }
        }
        else
        {
            int nextCursorItemIndex = m_VisibleTopItemIndex - VisibleItemCountMax + m_CursorIndex ;
            if(nextCursorItemIndex < 0)
            {
                nextCursorItemIndex = 0;
            }

            if(m_VisibleTopItemIndex - VisibleItemCountMax < 0)
            {
                m_VisibleTopItemIndex = 0;
            }
            else
            {
                m_VisibleTopItemIndex -= VisibleItemCountMax;
            }

            m_CursorIndex = nextCursorItemIndex - m_VisibleTopItemIndex;
        }

        UpdateItem();

        m_ScrollCounterForPath = 0;
        m_ScrollCounterForCharacter = 0;
    }
}

void NtfFileListWindow::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    Window::Render();

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    renderer.SetTextScale(1.2f, 1.2f);

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    // xx/oo 表示
    {
        char value[32];
        float denominatorX;
        float denominatorY;
        renderer.SetTextColor(GetDisplayColor(graphics::Colors::Black));
        sprintf(value, "/%lu", m_DirectoryEntryList.size());
        denominatorX = rect.x + rect.width - 10 - renderer.CalculateTextWidth(value);
        denominatorY = rect.y - 32;
        renderer.DrawText(denominatorX, denominatorY, value);
        sprintf(value, "%lu", m_VisibleTopItemIndex + m_CursorIndex + 1);
        renderer.DrawText(denominatorX - renderer.CalculateTextWidth(value), denominatorY, value);
    }

    float dx = rect.x + 4;
    float dy = rect.y + 2;
    float santenWidth = renderer.CalculateTextWidth("...");
    for (int i = 0; i < m_VisibleItemNum; i++)
    {
        auto& item = m_Items[i];
        if (m_CursorIndex == i)
        {
            // 選択中の項目
            DrawCursor(dx, dy, rect.width - 4, renderer.GetTextLineHeight() - 8, GetCursorFadeColor());

            renderer.SetTextColor(GetDisplayColor(graphics::Colors::Turquoise));
        }
        else
        {
            // 通常の項目
            renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));
        }

        auto makeShortString = [&](std::string str, float width, int* counter)
        {
            float textWidth = renderer.CalculateTextWidth(str.c_str());
            if(textWidth < width)
            {
                return str;
            }

            std::string retStr;
            if (m_CursorIndex == i)
            {
                const int scale = 8;
                const std::string separator = "        ";
                int counterSize = str.size() + separator.size();
                if(counterSize == *counter / scale)
                {
                    *counter = 0;
                }
                retStr = (str + separator).substr(*counter / scale) + str;
                retStr = GetFront(renderer, retStr, width);
                *counter = *counter + 1;
            }
            else
            {
                float targetWidth = (width - santenWidth) / 2;
                retStr = GetFront(renderer, str, targetWidth);
                retStr += "...";
                retStr += GetBackward(renderer, str, targetWidth);
            }

            return retStr;
        };

        if(m_AmiiboDataList.empty())
        {
            renderer.DrawText(dx + 8, dy, "%s", makeShortString(item.path, rect.width - 8 - 32, &m_ScrollCounterForPath).c_str());
        }
        else
        {
            renderer.DrawText(dx + 8, dy, "%s", makeShortString(item.character, 250 - 8 - 16, &m_ScrollCounterForCharacter).c_str());
            renderer.DrawText(dx + 250, dy, "%s", makeShortString(item.path, rect.width - 250 - 32, &m_ScrollCounterForPath).c_str());
        }

        dy += renderer.GetTextLineHeight();
    }

    float barWidth = 10;
    DrawScrollBar(rect.x + rect.width - barWidth - 10, rect.y + 10, barWidth, rect.height - 20, m_VisibleTopItemIndex, VisibleItemCountMax, m_DirectoryEntryList.size(), graphics::Colors::White);

    renderer.SetTextColor(graphics::Colors::White);
    renderer.SetTextScale(prevScale.x, prevScale.y);
}

void NtfFileListWindow::Invoke(int cursorIndex, const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    if (cursorIndex < 0 ||
        cursorIndex >= VisibleItemCountMax)
    {
        return;
    }

    if (m_Handler != nullptr)
    {
        {
            nn::Result result;
            result = nn::fs::MountSdCardForDebug(SdMountName);
            if(result.IsFailure())
            {
                m_SdAccessHandler(MessageSdCardNotFound, true);
                return;
            }
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount(SdMountName);
            };

            nn::fs::FileHandle handle;
            result = nn::fs::OpenFile(&handle, AddScheamToPath(m_Items[cursorIndex].path.c_str()).c_str(), nn::fs::OpenMode_Read);
            if(result.IsFailure())
            {
                m_SdAccessHandler(MessageSdCardNotFound, true);
                return;
            }
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            size_t readSize;
            result = nn::fs::ReadFile(&readSize, handle, 0, m_Ntf.data, sizeof(m_Ntf.data));
            if(result.IsFailure() || readSize != sizeof(m_Ntf.data))
            {
                m_SdAccessHandler(MessageSdCardAccessError, true);
                return;
            }
        }

        m_Handler(npadId, reinterpret_cast<uintptr_t>(&m_Ntf));
    }
}

}}  // noftwriter::ui
