﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoyRight.h>

namespace noftwriter { namespace npad {

const nn::hid::NpadIdType NpadIds[] =
{
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
    nn::hid::NpadId::No5,
    nn::hid::NpadId::No6,
    nn::hid::NpadId::No7,
    nn::hid::NpadId::No8,
    nn::hid::NpadId::Handheld,
};

const int NpadIdCountMax = sizeof(NpadIds) / sizeof(nn::hid::NpadIdType);

const uint8_t IndicatorPattern[] = { 0x01, 0x03, 0x07, 0x0f, 0x09, 0x05, 0x0D, 0x06, 0x00 };

/**
 * @brief   右コン操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadRightJoy
{
    typedef nn::hid::NpadStyleJoyRight Style;
    typedef nn::hid::NpadJoyRightState State;
};

/**
 * @brief   フルキー操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadFullKey
{
    typedef nn::hid::NpadStyleFullKey Style;
    typedef nn::hid::NpadFullKeyState State;
};

/**
 * @brief   携帯機操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadHandheld
{
    typedef nn::hid::NpadStyleHandheld Style;
    typedef nn::hid::NpadHandheldState State;
};

/**
 * @brief   コントローラーに対する処理を扱うインターフェースクラスです。
 */
class INpadController
{
    NN_DISALLOW_COPY(INpadController);
    NN_DISALLOW_MOVE(INpadController);

public:
    INpadController() NN_NOEXCEPT {}
    virtual ~INpadController() NN_NOEXCEPT {}
    virtual void Initialize() NN_NOEXCEPT = 0;
    virtual void Finalize() NN_NOEXCEPT = 0;
    virtual void Update() NN_NOEXCEPT = 0;

    virtual nn::hid::NpadIdType GetNpadId() const NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンの組み合わせが押されているか
     */
    virtual bool IsPressed(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンの組み合わせが新たに押されたか
     */
    virtual bool IsTriggered(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンの組み合わせが離されたか
     */
    virtual bool IsReleased(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンがリピート入力されているか
     */
    virtual bool IsRepeated(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;
};


/**
 * @brief   操作形態ごとの処理を記述した、 INpadController の派生クラスです。
 */
template<typename Npad>
class NpadController : public INpadController
{
    NN_DISALLOW_COPY(NpadController);
    NN_DISALLOW_MOVE(NpadController);

public:
    explicit NpadController(nn::hid::NpadIdType npadId) NN_NOEXCEPT :
        m_NpadId(npadId),
        m_CurrentButtons(),
        m_PreviousButtons(),
        m_RepeatInfo(),
        m_RepeatWait(30),
        m_RepeatInterval(6) {}
    virtual ~NpadController() NN_NOEXCEPT NN_OVERRIDE
    {
        Finalize();
    }

    virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        UpdateButtons();
    }

    virtual nn::hid::NpadIdType GetNpadId() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_NpadId;
    }

    virtual bool IsPressed(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_CurrentButtons & buttons) == buttons;
    }

    virtual bool IsTriggered(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_CurrentButtons & buttons) == buttons &&
            (m_PreviousButtons & buttons) != buttons;
    }

    virtual bool IsReleased(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_CurrentButtons & buttons) != buttons &&
            (m_PreviousButtons & buttons) == buttons;
    }

    virtual bool IsRepeated(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        if (buttons.IsAllOff())
        {
            return false;
        }

        const int buttonCount = sizeof(m_RepeatInfo) / sizeof(m_RepeatInfo[0]);
        for (int i = 0; i < buttonCount; i++)
        {
            if (!buttons.Test(i))
            {
                continue;
            }

            const auto& repeat = m_RepeatInfo[i];
            if (repeat.waitCount == 1)
            {
                // 押下直後
                return true;
            }

            if (repeat.waitCount < m_RepeatWait)
            {
                // 初回リピート待ち
                return false;
            }

            if (repeat.intervalCount == 0)
            {
                return true;
            }
        }

        return false;
    }

private:
    struct RepeatInfo
    {
        uint32_t    waitCount;
        uint32_t    intervalCount;
    };

private:
    void UpdateButtons() NN_NOEXCEPT
    {
        m_PreviousButtons = m_CurrentButtons;

        typename Npad::State state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_CurrentButtons = state.buttons;

        UpdateRepeat();
    }

    void UpdateRepeat() NN_NOEXCEPT
    {
        const int buttonCount = sizeof(m_RepeatInfo) / sizeof(m_RepeatInfo[0]);
        for (int i = 0; i < buttonCount; i++)
        {
            auto& repeat = m_RepeatInfo[i];
            if (!m_CurrentButtons.Test(i))
            {
                repeat.waitCount     = 0;
                repeat.intervalCount = 0;
                continue;
            }

            if (repeat.waitCount < m_RepeatWait)
            {
                repeat.waitCount++;
                repeat.intervalCount = 0;
            }
            else
            {
                repeat.intervalCount = (repeat.intervalCount + 1) % m_RepeatInterval;
            }

        }
    }

private:
    const nn::hid::NpadIdType m_NpadId;             //!< Npad ID
    nn::hid::NpadButtonSet    m_CurrentButtons;     //!< 現在のボタン情報
    nn::hid::NpadButtonSet    m_PreviousButtons;    //!< 前回のボタン情報
    RepeatInfo                m_RepeatInfo[64];
    uint32_t                  m_RepeatWait;
    uint32_t                  m_RepeatInterval;
};

/**
 * @brief   Npad を初期化します。
 */
void Initialize() NN_NOEXCEPT;

/**
 * @brief   指定された NpadIdType に対応する NpadController を取得します。
 */
INpadController* GetNpadController(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT;

}}  // noftwriter::npad
