﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <new>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include "NoftWriter_NpadController.h"

namespace noftwriter { namespace npad {

namespace
{

struct ControllerType
{
    char storage[sizeof(NpadController<NpadRightJoy>)];
    nn::hid::NpadStyleSet style;

    bool IsAvailable() NN_NOEXCEPT
    {
        return style.IsAnyOn();
    }
};

ControllerType g_NpadControllers[NpadIdCountMax];

int GetControllerIndex(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    for (int i = 0; i < NpadIdCountMax; i++)
    {
        if (npadId == NpadIds[i])
        {
            return i;
        }
    }

    return -1;
}

/**
 * @brief   指定された NpadIdType のスタイルに対応する NpadController を作成します。
 */
INpadController* CreateNpadController(void* pStorage, const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pStorage);

    int index = GetControllerIndex(npadId);
    if (index < 0)
    {
        return nullptr;
    }

    // 接続されたデバイスに応じたコントローラクラスを作成
    auto style = nn::hid::GetNpadStyleSet(npadId);
    if (style.Test<nn::hid::NpadStyleFullKey>())
    {
        return new(pStorage) NpadController<NpadFullKey>(npadId);
    }
    else if (style.Test<nn::hid::NpadStyleHandheld>())
    {
        return new(pStorage) NpadController<NpadHandheld>(npadId);
    }

    return nullptr;
}

}  // anonymous

void Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();

    // 使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleFullKey::Mask |
        nn::hid::NpadStyleHandheld::Mask);

    // 使用するNpadを設定
    nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);

    std::memset(g_NpadControllers, 0, sizeof(g_NpadControllers));
}

INpadController* GetNpadController(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    int index = GetControllerIndex(npadId);
    if (index < 0)
    {
        return nullptr;
    }

    auto& controller = g_NpadControllers[index];
    if (controller.IsAvailable())
    {
        // 有効状態の再検証
        controller.style = nn::hid::GetNpadStyleSet(npadId);
    }

    if (controller.IsAvailable())
    {
        return reinterpret_cast<INpadController*>(controller.storage);
    }

    auto* pController = CreateNpadController(controller.storage, npadId);
    if (pController == nullptr)
    {
        return nullptr;
    }

    controller.style = nn::hid::GetNpadStyleSet(npadId);
    return pController;
}

}}  // noftwriter::npad
