﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nfp.h>
#include <nn/nfp/nfp_DebugApi.h>
#include <nn/nfp/nfp_PrivateResult.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "graphics/NoftWriter_Renderer.h"
#include "npad/NoftWriter_NpadController.h"
#include "ui/NoftWriter_Footer.h"
#include "ui/NoftWriter_MessageBox.h"
#include "ui/NoftWriter_DeviceListWindow.h"
#include "ui/NoftWriter_WaitDialog.h"

#include "ui/NoftWriter_NtfFileListWindow.h"
#include "ui/NoftWriter_WriteNtfWindow.h"

#include "NoftWriter_NfpProcessor.h"
#include "NoftWriter_Util.h"
#include "NoftWriter_Types.h"

//#define SHOW_UNSUPPORTED_COMMAND

 /**
  * @brief   メニューコマンドの定義
  */
#define NOFTWRITER_DEFINE_MENU_COMMAND(command) \
    void (command)(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT

/**
* @brief   メニュー項目をメニューオブジェクトに設定
*/
#define NOFTWRITER_SET_MENU_ITEMS(menu, items) \
    (menu).SetItems((items), NN_ARRAY_SIZE((items)));

namespace {

// プログラム名
const char ProgramName[] =
    "Noft Writer";

// ヒープサイズ
const size_t MemoryHeapSize = 256 * 1024 * 1024;
const size_t MallocHeapSize = 256 * 1024 * 1024;

NOFTWRITER_DEFINE_MENU_COMMAND(CommandWriteNtf);

// 優先実行アクション
enum class ForceAction
{
    None,           // なし
};

const char HelpNone[] = "";
const char HelpForNtfFileList[] =
    "[←→] Change page  [↑↓] Select  [A] Decide  [X] Reload";
const char HelpForSubMenu[] =
    "[↑↓] Select  [A] Decide  [B] Cancel";
const char HelpForMessageBox[] =
    "[A][B] Close";
const char HelpForSearchTag[] =
    "[B] Cancel";

noftwriter::ui::NtfFileListWindow     g_NtfFileList;
noftwriter::ui::WriteNtfWindow        g_WriteNtfWindow;
noftwriter::ui::DeviceListWindow      g_DeviceListWindow;

noftwriter::ui::MessageBox            g_MessageBox;
noftwriter::ui::WaitDialog            g_WaitDialog;
noftwriter::ui::Footer                g_Footer;

noftwriter::ui::Window* g_pActiveWindow = nullptr;

noftwriter::ui::Window* g_AllWindows[] =
{
    &g_NtfFileList,
    &g_WriteNtfWindow,
    &g_DeviceListWindow,
    &g_MessageBox,
    &g_WaitDialog,
};

noftwriter::NfpProcessor g_Nfp;

NN_OS_ALIGNAS_THREAD_STACK char g_NfpThreadStack[nn::os::ThreadStackAlignment * 4];
NN_OS_ALIGNAS_THREAD_STACK char g_HidThreadStack[nn::os::ThreadStackAlignment * 4];

nn::os::EventType   g_RenderEvent;
nn::os::ThreadType  g_HidThread;

ForceAction  g_ForceAction = ForceAction::None;

/**
 * @brief   アプリがフォーカス状態にあるか判定する
 */
bool IsFocused() NN_NOEXCEPT
{
    return nn::oe::GetCurrentFocusState() == nn::oe::FocusState_InFocus;
}

/**
 * @brief   キーヘルプを更新
 */
void UpdateHelp() NN_NOEXCEPT
{
    std::string help;

    if (g_pActiveWindow == &g_NtfFileList)
    {
        g_Footer.SetHelpText(HelpForNtfFileList);
    }
    else
    {
        g_Footer.SetHelpText(HelpForSubMenu);
    }
}

/**
 * @brief   指定したウィンドウをアクティブ化
 */
void ActivateWindow(noftwriter::ui::Window* pWindow, bool isVisibleLastWindow) NN_NOEXCEPT
{
    if (g_pActiveWindow != nullptr)
    {
        g_pActiveWindow->Deactivate();
        if (!isVisibleLastWindow)
        {
            g_pActiveWindow->Hide();
        }
    }

    g_pActiveWindow = pWindow;
    g_pActiveWindow->Show();
    g_pActiveWindow->Activate();

    UpdateHelp();
}

/**
 * @brief   指定したウィンドウをサブメニューとしてアクティブ化
 */
void ActivateSubMenu(noftwriter::ui::Window* pWindow, bool isVisibleLastWindow) NN_NOEXCEPT
{
    if (g_pActiveWindow != nullptr && isVisibleLastWindow)
    {
        auto* pActiveWindow = reinterpret_cast<noftwriter::ui::SelectableWindow*>(g_pActiveWindow);
        if(pActiveWindow->GetSelectedItemY() < 760 / 2)
        {
            pWindow->SetPosition(
                pActiveWindow->GetX() + 30,
                pActiveWindow->GetSelectedItemY() + 30);
        }
        else
        {
            pWindow->SetPosition(
                pActiveWindow->GetX() + 30,
                pActiveWindow->GetSelectedItemY() - pWindow->GetHeight() + 4);
        }
    }

    ActivateWindow(pWindow, isVisibleLastWindow);
}

/**
 * @brief   メッセージダイアログを表示
 */
void DisplayMessage(
    const char* message,
    const char* caption,
    noftwriter::ui::MessageBox::Icon icon,
    noftwriter::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(message);
    NN_ASSERT_NOT_NULL(caption);
    NN_ASSERT_NOT_NULL(cancelHandler);

    ActivateWindow(&g_MessageBox, false);
    g_WaitDialog.Hide();
    g_MessageBox.SetIcon(icon);
    g_MessageBox.SetCaption(caption);
    g_MessageBox.SetText(message);
    g_MessageBox.SetCancelHandler(cancelHandler);
    g_DeviceListWindow.ClearActiveNpadId();
    g_Footer.SetHelpText(HelpForMessageBox);
}

/**
 * @brief   情報メッセージを表示
 */
void DisplayInformationMessage(
    const char* message,
    noftwriter::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(message);
    NN_ASSERT_NOT_NULL(cancelHandler);

    DisplayMessage(
        message,
        "Information",
        noftwriter::ui::MessageBox::Icon::Information,
        cancelHandler);
}

/**
 * @brief   エラーメッセージを表示
 */
void DisplayErrorMessage(
    const char* message,
    noftwriter::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(message);
    NN_ASSERT_NOT_NULL(cancelHandler);

    DisplayMessage(
        message,
        "Error",
        noftwriter::ui::MessageBox::Icon::Error,
        cancelHandler);
}

/**
 * @brief   NTAG 以外を検出した場合のメッセージを表示
 */
void DisplayNotNtagMessage(noftwriter::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(cancelHandler);

    DisplayErrorMessage("Detected tag is not an NTAG.", cancelHandler);
}

/**
 * @brief   SD カードに正しくアクセスできなかった場合のメッセージを表示
 */
void DisplaySdAccessMessage(const char* message, bool isError) NN_NOEXCEPT
{
    auto cancelHandler = [](const nn::hid::NpadIdType& npadId, uintptr_t argument)
    {
        NN_UNUSED(npadId);
        NN_UNUSED(argument);
        g_NtfFileList.ReBuildListStart();
        ActivateWindow(&g_NtfFileList, false);
    };

    auto cancelHandlerNotError = [](const nn::hid::NpadIdType& npadId, uintptr_t argument)
    {
        NN_UNUSED(npadId);
        NN_UNUSED(argument);
        ActivateWindow(&g_NtfFileList, false);
    };

    if(isError)
    {
        DisplayMessage(
            message,
            "Error",
            noftwriter::ui::MessageBox::Icon::Error,
            cancelHandler);
    }
    else
    {
        DisplayMessage(
            message,
            "Information",
            noftwriter::ui::MessageBox::Icon::Information,
            cancelHandlerNotError);
    }
}

/**
 * @brief   汎用エラーメッセージを表示
 */
void DisplayCommonError(
    nn::Result result,
    noftwriter::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(cancelHandler);

    const char* message;
    if (nn::nfp::ResultNfcDeviceNotFound::Includes(result))
    {
        message = "Device not found.";
    }
    else if (nn::nfp::ResultTagNotFound::Includes(result) ||
        nn::nfp::ResultUidMisMatch::Includes(result))
    {
        message = "Target tag is not found.";
    }
    else
    {
        message = "Unexpected error.";
    }

    DisplayErrorMessage(message, cancelHandler);
}

/**
 * @brief   Flush 系の操作に失敗した場合のメッセージを表示
 */
void DisplayFlushFailMessage(
    const char* message,
    noftwriter::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(message);
    NN_ASSERT_NOT_NULL(cancelHandler);

    DisplayErrorMessage(message, cancelHandler);
}

/**
 * @brief   NTF ファイルリストに戻る
 */
void ReturnToNtfFileList(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);
    ActivateWindow(&g_NtfFileList, false);
}

/**
 * @brief   Write NTF Window に戻る
 */
void ReturnToWriteNtfWindow(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);
    ActivateWindow(&g_WriteNtfWindow, false);
}

/**
 * @brief   タグ検出処理の実行
 */
void ExecuteSearchAction(
    const nn::hid::NpadIdType& npadId,
    noftwriter::NfpTagDetectCallbackType detectCallback,
    noftwriter::NfpResultCallbackType finishCallback) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(detectCallback);

    auto* pPreviousWindow = g_pActiveWindow;

    auto Cleanup = [&]
    {
        g_Nfp.Deactivate();
        pPreviousWindow->Activate();
    };

    pPreviousWindow->Deactivate();
    g_Nfp.Deactivate();

    if (!g_Nfp.Activate(npadId))
    {
        NN_LOG("FAILED: Activate NFC\n");
        Cleanup();
        return;
    }

    // タグサーチ中断
    auto CancelCallback = [](void* pArg)
    {
        g_Nfp.Deactivate();
        ActivateWindow(reinterpret_cast<noftwriter::ui::Window*>(pArg), false);
        g_WaitDialog.Hide();
        g_DeviceListWindow.ClearActiveNpadId();
    };

    g_Nfp.ClearCallback();
    g_Nfp.SetDetectCallback(detectCallback, pPreviousWindow);
    g_Nfp.SetCancelCallback(CancelCallback, pPreviousWindow);
    g_Nfp.SetFinishCallback(finishCallback, pPreviousWindow);

    if (g_Nfp.StartDetection().IsFailure())
    {
        NN_LOG("FAILED: StartDetection\n");
        Cleanup();
        return;
    }

    g_DeviceListWindow.SetActiveNpadId(npadId);

    auto dialogCancelHandler = [](const nn::hid::NpadIdType& npadId, uintptr_t argument)
    {
        g_Nfp.StopDetection();
    };

    g_WaitDialog.SetCancelHandler(dialogCancelHandler);
    g_WaitDialog.SetText(
        "  Please touch an unlocked NTAG215!\n"
        "(Press B button to cancel)");
    ActivateWindow(&g_WaitDialog, true);

    g_Footer.SetHelpText(HelpForSearchTag);
}

void ApplyWriteNtf(const nn::hid::NpadIdType& npadId, uintptr_t arg, int index) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(arg);
    NN_UNUSED(index);

    // 処理完了
    auto FinishCallback = [](noftwriter::NfpResult result, void* pArg)
    {
        g_Nfp.Deactivate();

        if (result == noftwriter::NfpResult::Success)
        {
            DisplayInformationMessage("Finished!!", ReturnToWriteNtfWindow);
        }
        else
        {
            DisplayFlushFailMessage("Write failed...", ReturnToWriteNtfWindow);
        }
    };

    // タグ検出
    auto DetectCallback = [](void* pArg)
    {
        noftwriter::NfpInfo info;
        g_Nfp.GetNfpInfo(&info);
        if (info.isValidTag)
        {
            ActivateWindow(&g_WaitDialog, true);
            g_WaitDialog.SetText(
                "  Write in progress...\n"
                "Keep touching the unlocked NTAG215!");

            auto* pWindow = reinterpret_cast<noftwriter::ui::WriteNtfWindow*>(pArg);
            auto result = g_Nfp.WriteNtf(pWindow->GetNtf(), pWindow->GetNtfWriteType());
            if (result.IsFailure())
            {
                DisplayCommonError(result, ReturnToWriteNtfWindow);
            }
            else
            {
                g_Footer.SetHelpText(HelpNone);
            }
        }
        else
        {
            g_Nfp.Deactivate();
            DisplayNotNtagMessage(ReturnToWriteNtfWindow);
        }
    };

    ExecuteSearchAction(npadId, DetectCallback, FinishCallback);
}

void CommandWriteNtf(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);

    g_WriteNtfWindow.SetNtf(argument);
    g_WriteNtfWindow.SetDecideHandler(ApplyWriteNtf);

    ActivateSubMenu(&g_WriteNtfWindow, true);
}

/**
 * @brief   ウィンドウの初期化
 */
void InitializeWindow(noftwriter::graphics::Renderer* pRenderer) NN_NOEXCEPT
{
    for (auto* pWindow : g_AllWindows)
    {
        pWindow->Initialize();
        pWindow->SetRenderer(pRenderer);
    }

    g_DeviceListWindow.SetPosition(960, 330);
    g_DeviceListWindow.SetSize(300, 320);
    g_DeviceListWindow.SetCaption("Connected devices");
    g_DeviceListWindow.SetCaptionColor(
        noftwriter::ui::DefaultActiveCaptionBackColor,
        noftwriter::ui::DefaultActiveCaptionBackColor,
        noftwriter::ui::DefaultCaptionColor);
    g_DeviceListWindow.Show();

    g_MessageBox.SetPosition(380, 304);
    g_MessageBox.SetSize(520, 128);

    g_WaitDialog.SetPosition(380, 256);
    g_WaitDialog.SetSize(520, 224);
    g_WaitDialog.SetCaption("Processing...");




    g_WriteNtfWindow.SetCancelHandler(ReturnToNtfFileList);
    g_WriteNtfWindow.SetCaption("＜ NTF file list");

    g_NtfFileList.SetPosition(32, 108);
    g_NtfFileList.SetSize(900, 548);
    g_NtfFileList.SetCaption("NTF file list");
    g_NtfFileList.SetSdAccessHandler(DisplaySdAccessMessage);
    g_NtfFileList.SetDecideHandler(CommandWriteNtf);



    g_Footer.SetRenderer(pRenderer);
    g_Footer.Show();

    ActivateWindow(&g_NtfFileList, false);
}

/**
 * @brief   優先実行アクションの更新
 */
void UpdateForceAction() NN_NOEXCEPT
{
    if (g_ForceAction == ForceAction::None)
    {
        return;
    }

    switch (g_ForceAction)
    {
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    g_ForceAction = ForceAction::None;
}

/**
 * @brief   パッド入力の更新
 */
void UpdateInput() NN_NOEXCEPT
{
    for (int i = 0; i < noftwriter::npad::NpadIdCountMax; i++)
    {
        auto id = noftwriter::npad::NpadIds[i];
        auto* pController = noftwriter::npad::GetNpadController(id);
        if (pController == nullptr)
        {
            continue;
        }

        auto time1 = nn::os::GetSystemTick().ToTimeSpan();
        pController->Update();
        auto time2 = nn::os::GetSystemTick().ToTimeSpan();
        g_pActiveWindow->UpdateInput(*pController);
        auto time3 = nn::os::GetSystemTick().ToTimeSpan();
        g_Nfp.UpdateInput(*pController);
        auto time4 = nn::os::GetSystemTick().ToTimeSpan();

#if 0
        if (time4.GetMilliSeconds() - time1.GetMilliSeconds() >= 100)
        {
            auto d1 = time2 - time1;
            NN_LOG("  T1: %2d.%03d\n", d1.GetSeconds(), d1.GetMilliSeconds() % 1000);
            auto d2 = time3 - time2;
            NN_LOG("  T2: %2d.%03d\n", d2.GetSeconds(), d2.GetMilliSeconds() % 1000);
            auto d3 = time4 - time3;
            NN_LOG("  T3: %2d.%03d\n", d3.GetSeconds(), d3.GetMilliSeconds() % 1000);
        }
#else
        NN_UNUSED(time1);
        NN_UNUSED(time2);
        NN_UNUSED(time3);
        NN_UNUSED(time4);
#endif
    }

    UpdateForceAction();
}

/**
 * @brief   HID 関連の処理を行うスレッド関数
 */
void HidThreadFunction(void* pArg) NN_NOEXCEPT
{
    bool* pRuns = reinterpret_cast<bool*>(pArg);
    while (*pRuns)
    {
        nn::os::WaitEvent(&g_RenderEvent);
        UpdateInput();
    }
}

/**
 * @brief   ウィンドウを更新
 */
void UpdateWindows() NN_NOEXCEPT
{
    for (auto* pWindow : g_AllWindows)
    {
        pWindow->Update();
    }
}

/**
 * @brief   システムの状態変化監視
 */
void CheckSystemMessage() NN_NOEXCEPT
{
    nn::oe::Message message;
    if (!nn::oe::TryPopNotificationMessage(&message))
    {
        return;
    }

    if (message != nn::oe::MessageFocusStateChanged)
    {
        return;
    }

    switch (nn::oe::GetCurrentFocusState())
    {
    case nn::oe::FocusState_Background:
    case nn::oe::FocusState_OutOfFocus:
        g_Nfp.NotifyLostFocus();
        break;
    default:
        // 何もしない
        break;
    }
}

/**
 * @brief   描画処理
 */
void Render() NN_NOEXCEPT
{
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::SignalEvent(&g_RenderEvent);
    };

    if (!IsFocused())
    {
        // フォーカス喪失中は描画せず 1 フレーム程度の時間待つだけ
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(16666));
        return;
    }

    auto& renderer = *noftwriter::graphics::Renderer::GetInstance();

    // 初期描画設定
    renderer.SetTextScale(1.2f, 1.2f);
    renderer.SetTextColor(noftwriter::graphics::Colors::White);

    noftwriter::PrintHeader(&renderer, ProgramName);

    for (auto* pWindow : g_AllWindows)
    {
        pWindow->Render();
    }

    g_Footer.Render();
    renderer.Render();
}

}  // anonymous

// アプリケーションのメモリ管理機構を初期化
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

extern "C" void nnMain()
{
    InitializeWindow(noftwriter::graphics::Renderer::GetInstance());

    noftwriter::npad::Initialize();

    nn::nfp::InitializeDebug();
    g_Nfp.Setup(g_NfpThreadStack, sizeof(g_NfpThreadStack));

    bool runs = true;

    // フォーカス状態変化通知を受け取る
    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);

    nn::os::InitializeEvent(&g_RenderEvent, false, nn::os::EventClearMode_AutoClear);

    nn::os::CreateThread(
        &g_HidThread,
        HidThreadFunction,
        &runs,
        g_HidThreadStack,
        sizeof(g_HidThreadStack),
        nn::os::DefaultThreadPriority);
    nn::os::StartThread(&g_HidThread);

    NN_LOG("<< %s Started >>\n", ProgramName);

    while (NN_STATIC_CONDITION(runs))
    {
        UpdateWindows();
        CheckSystemMessage();
        Render();
    }

    nn::os::WaitThread(&g_HidThread);
    nn::os::DestroyThread(&g_HidThread);

    NN_LOG("Finished\n");

    g_Nfp.Shutdown();
    nn::nfp::FinalizeDebug();
    noftwriter::graphics::Renderer::Shutdown();
}
