﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_Npad.h>

#include "../graphics/NfpDebugTool_Renderer.h"
#include "../npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_Window.h"

namespace
{

const int FadeSpeed            =  64;
const int CursorFadeFrameCount = 120;
const int CursorFadeStrength   =  24;

}  // anonymous

namespace nfpdebug { namespace ui {

const graphics::Color DefaultWindowFrameColor =
    graphics::Colors::White;

const graphics::Color DefaultWindowBackColor =
    graphics::Colors::GetAlphaColor(graphics::Colors::DimGray, 244);

const graphics::Color DefaultCaptionColor =
    graphics::Colors::Black;

const graphics::Color DefaultActiveCaptionBackColor =
    graphics::Colors::GetAlphaColor(graphics::Colors::Silver, 224);

const graphics::Color DefaultInactiveCaptionBackColor =
    graphics::Colors::GetAlphaColor(graphics::Colors::Gray, 224);

const float Window::FixedModeTextWidth = 14.0f;

void Window::Initialize() NN_NOEXCEPT
{
    m_FrameColor               = DefaultWindowFrameColor;
    m_BackColor                = DefaultWindowBackColor;
    m_ActiveCaptionBackColor   = DefaultActiveCaptionBackColor;
    m_InactiveCaptionBackColor = DefaultInactiveCaptionBackColor;
    m_CaptionColor             = DefaultCaptionColor;
}

void Window::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
    if (!IsActive() || !IsVisible())
    {
        return;
    }

    if (controller.IsTriggered(nn::hid::NpadButton::B::Mask))
    {
        auto handler = GetCancelHandler();
        if (handler != nullptr)
        {
            handler(controller.GetNpadId(), 0);
        }
    }
}

void Window::Update() NN_NOEXCEPT
{
    DisplayObject::Update();

    if (m_IsShowing)
    {
        SetOpacity(std::min<int>(GetOpacity() + FadeSpeed, 255));
        if (GetOpacity() >= 255)
        {
            m_IsShowing = false;
        }
    }
    else if (m_IsHiding)
    {
        SetOpacity(std::max<int>(GetOpacity() - FadeSpeed, 0));
        if (GetOpacity() <= 0)
        {
            m_IsHiding = false;
            DisplayObject::Hide();
        }
    }

    if (IsActive())
    {
        m_CursorFadeDuration = (m_CursorFadeDuration + 1) % CursorFadeFrameCount;
    }
}

void Window::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    auto& renderer = *GetRenderer();

    float x = GetX() - 4;
    float y = GetY() - 2;

    float renderWidth  = GetRenderWidth();
    float renderHeight = GetRenderHeight();
    float headerHeight = GetHeaderHeight();
    renderer.FillRect(
        x,
        y + headerHeight,
        renderWidth - 1,
        renderHeight - headerHeight,
        GetDisplayColor(m_BackColor));
    renderer.FillRect(
        x,
        y,
        renderWidth - 1,
        headerHeight + 1,
        GetDisplayColor(IsActive() ?
            m_ActiveCaptionBackColor :
            m_InactiveCaptionBackColor));
    renderer.DrawRect(
        x,
        y,
        renderWidth,
        renderHeight,
        GetDisplayColor(m_FrameColor),
        2.0f);
    renderer.DrawHorizontalLine(
        x,
        y + headerHeight,
        renderWidth,
        GetDisplayColor(m_FrameColor));

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    renderer.SetTextScale(1.3f, 1.3f);
    renderer.SetTextColor(GetDisplayColor(m_CaptionColor));
    renderer.DrawText(x + 8, y + 5, "%s", m_Caption != nullptr ? m_Caption : "");

    renderer.SetTextColor(graphics::Colors::White);
    renderer.SetTextScale(prevScale.x, prevScale.y);
}

graphics::Color Window::GetCursorFadeColor() const NN_NOEXCEPT
{
    auto color = m_CursorColor;
    auto rate = sin(m_CursorFadeDuration * M_PI * 2 / CursorFadeFrameCount);
    color.v[3] = static_cast<uint8_t>(
        32 + CursorFadeStrength * (IsActive() ? rate : -1));

    return GetDisplayColor(color);
}

void Window::DrawCursor(float x, float y, float width, float height, const graphics::Color& color) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    nn::util::Float2 scale;
    renderer.GetTextScale(&scale);

    float newScaleH = width  / 15.0f;
    float newScaleV = height / 14.5f;
    renderer.SetTextScale(newScaleH, newScaleV);
    renderer.SetTextColor(GetDisplayColor(color));
    renderer.DrawText(x - newScaleH * 0.7f, y - newScaleV, "■");

    renderer.SetTextScale(scale.x, scale.y);
}

void Window::GetClientRectangle(graphics::Rectangle* pOutRect) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutRect);

    pOutRect->x = GetX();
    pOutRect->y = GetY() + GetHeaderHeight();
    pOutRect->width = GetRenderWidth();
    pOutRect->height = GetRenderHeight() - GetHeaderHeight();
}

}}  // nfpdebug::ui
