﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define _USE_MATH_DEFINES

#include <cmath>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/hid/hid_Npad.h>

#include "NfpDebugTool_TagDumpWindow.h"

namespace
{

const int AnimationFrameCount = 90;

}

namespace nfpdebug { namespace ui {

void TagDumpWindow::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
    if (controller.IsTriggered(nn::hid::NpadButton::Right::Mask))
    {
        m_DrawMode = static_cast<DrawMode>(
            (m_DrawMode + 1) % DrawMode_Max);
    }
    else if (controller.IsTriggered(nn::hid::NpadButton::Left::Mask))
    {
        m_DrawMode = static_cast<DrawMode>(
            (m_DrawMode + DrawMode_Max - 1) % DrawMode_Max);
    }

    Window::UpdateInput(controller);
}

void TagDumpWindow::Update() NN_NOEXCEPT
{
    Window::Update();

    m_AnimationDuration = (m_AnimationDuration + 1) % AnimationFrameCount;
}

void TagDumpWindow::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    Window::Render();

    if (!m_NfpInfo.IsValidTag())
    {
        return;
    }

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);
    renderer.SetTextScale(1.2f, 1.2f);

    DrawPageCursor();

    graphics::Rectangle rect;
    GetClientRectangle(&rect);


    float x = rect.x + 32;
    float y = rect.y + 6;

    switch (m_DrawMode)
    {
    case DrawMode_Basic:
        DrawPageBasic(x, y);
        break;
    case DrawMode_System:
        DrawPageSystem(x, y);
        break;
    case DrawMode_SystemForGeneral:
        DrawPageSystemForGeneral(x, y);
        break;
    case DrawMode_Register:
        DrawPageRegister(x, y);
        break;
    case DrawMode_Application:
        DrawPageApplication(x, y);
        break;
    default:
        break;
    }

    DrawPageIndicator();

    renderer.SetTextScale(prevScale.x, prevScale.y);
}

void TagDumpWindow::DrawPageCursor() NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    float offsetX = 4 * ::sinf(2 * M_PI * m_AnimationDuration / AnimationFrameCount);
    float arrowY = rect.y + rect.height / 2;
    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));
    renderer.DrawText(rect.x - offsetX + 8, arrowY, "<");
    renderer.DrawText(rect.x + offsetX + rect.width - 28, arrowY, ">");

}

void TagDumpWindow::DrawPageIndicator() NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    float offsetX = 20;
    float x = rect.x + rect.width - offsetX * DrawMode_Max - 16;
    float y = rect.y + 8;

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));
    for (int i = 0; i < DrawMode_Max; i++)
    {
        renderer.DrawText(x + offsetX * i, y, "%s",
            m_DrawMode == i ? "●" : "・");
    }
}

void TagDumpWindow::DrawPageBasic(float x, float y) NN_NOEXCEPT
{
    float dy = y;
    float offset = 180;

    dy += DrawTagInfo(x, dy, offset, m_NfpInfo.tagInfo) + 8;

    if (m_NfpInfo.IsAmiibo())
    {
        dy += DrawModelInfo(x, dy, offset, m_NfpInfo.modelInfo) + 8;
        if (m_NfpInfo.HasValidRamArea())
        {
            DrawCommonInfo(x, dy, offset, m_NfpInfo.allData.commonInfo);
        }
        else
        {
            DrawErrorText(x + 8, dy + 16, "This amiibo is broken");
        }
    }
    else
    {
        DrawErrorText(x + 8, dy + 16, "This tag is not an amiibo");
    }
}

void TagDumpWindow::DrawPageSystem(float x, float y) NN_NOEXCEPT
{
    if (!m_NfpInfo.IsAmiibo())
    {
        DrawErrorText(x + 8, y + 16, "This tag is not an amiibo");
        return;
    }
    else if (!m_NfpInfo.HasValidRamArea())
    {
        DrawErrorText(x + 8, y + 16, "This amiibo is broken");
        return;
    }

    float dy = y;
    float offset = 220;

    dy += DrawSystemInfo(x, dy, offset, m_NfpInfo.allData.systemInfo) + 8;
    DrawAdminInfo(x, dy, offset, m_NfpInfo.allData.adminInfo);
}

void TagDumpWindow::DrawPageSystemForGeneral(float x, float y) NN_NOEXCEPT
{
    if (!m_NfpInfo.IsAmiibo())
    {
        DrawErrorText(x + 8, y + 16, "This tag is not an amiibo");
        return;
    }
    else if (!m_NfpInfo.HasValidRamArea())
    {
        DrawErrorText(x + 8, y + 16, "This amiibo is broken");
        return;
    }

    float dy = y;
    float offset = 220;

    dy += DrawAdminInfo(x, dy, offset, m_NfpInfo.allData.adminInfo) + 8;

    if (m_NfpInfo.HasRegisterInfo())
    {
        DrawRegisterInfo(x, dy, offset, m_NfpInfo.allData.registerInfo);
    }
    else
    {
        DrawRegisterInfo(x, dy);
    }
}

void TagDumpWindow::DrawPageRegister(float x, float y) NN_NOEXCEPT
{
    if (!m_NfpInfo.IsAmiibo())
    {
        DrawErrorText(x + 8, y + 16, "This tag is not an amiibo");
        return;
    }
    else if (!m_NfpInfo.HasValidRamArea())
    {
        DrawErrorText(x + 8, y + 16, "This amiibo is broken");
        return;
    }

    float offset = 220;

    if (m_NfpInfo.HasRegisterInfo())
    {
        DrawRegisterInfo(x, y, offset, m_NfpInfo.allData.registerInfo);
    }
    else
    {
        DrawRegisterInfo(x, y);
    }
}

void TagDumpWindow::DrawPageApplication(float x, float y) NN_NOEXCEPT
{
    if (!m_NfpInfo.IsAmiibo())
    {
        DrawErrorText(x + 8, y + 16, "This tag is not an amiibo");
        return;
    }
    else if (!m_NfpInfo.HasValidRamArea())
    {
        DrawErrorText(x + 8, y + 16, "This amiibo is broken");
        return;
    }

    if (m_NfpInfo.HasApplicationArea())
    {
        DrawApplicationArea(
            x,
            y,
            m_NfpInfo.allData.applicationArea,
            m_NfpInfo.allData.commonInfo.applicationAreaSize);
    }
    else
    {
        DrawApplicationArea(x, y);
    }
}

}}  // nfpdebug::ui
