﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_Npad.h>

#include "../graphics/NfpDebugTool_Renderer.h"
#include "../npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_SelectableWindow.h"

namespace nfpdebug { namespace ui {

/**
 * @brief   メニュー項目の内容を表す型です。
 */
struct MenuItem
{
    const char*         caption;    //!< 項目名
    WindowHandlerType   handler;    //!< 決定時に実行する処理
    uintptr_t           argument;   //!< 決定時に渡される引数
};

/**
 * @brief   選択可能なメニューを表示するウィンドウです。
 */
class MenuWindow : public SelectableWindow
{
public:
    MenuWindow() NN_NOEXCEPT :
        m_pItems(nullptr),
        m_ItemCount(0),
        m_CursorIndex(-1)
    {
        SetCursorVisible(true);
    }

    /**
     * @brief   メニュー項目一覧を設定します。
     */
    void SetItems(const MenuItem* pItems, int itemCount) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER_EQUAL(itemCount, 0);

        m_pItems    = pItems;
        m_ItemCount = itemCount;
        SetCursorIndex(m_CursorIndex);
    }

    /**
     * @brief   カーソル位置を設定します。
     */
    virtual void SetCursorIndex(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        m_CursorIndex = std::min(std::max(index, 0), m_ItemCount - 1);
    }

    /**
     * @brief   カーソル位置を取得します。
     */
    virtual int GetCursorIndex() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_CursorIndex;
    }

    /**
     * @brief   カーソル位置の項目の Y 座標を取得します。
     */
    virtual float GetSelectedItemY() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GetY() + GetDefaultLineHeight() * GetCursorIndex() + GetHeaderHeight();
    }

    virtual void UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

protected:
    virtual float GetRenderHeight() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GetDefaultLineHeight() * m_ItemCount + GetHeaderHeight() + 8;
    }

private:
    /**
     * @brief   指定したメニュー項目を実行します。
     */
    void Invoke(int cursorIndex, const nn::hid::NpadIdType& npadId) NN_NOEXCEPT;

private:
    const MenuItem*     m_pItems;           //!< 項目一覧
    int                 m_ItemCount;        //!< 項目数
    int                 m_CursorIndex;      //!< カーソルの位置
};

}}  // nfpdebug::ui
